  program create_simulated_data

! ***************************************************************0**
! create simulated data of an imaginary search-coil,
!                       or an imaginary magnetometer
!                       or an imaginary trajectory
!
! P. Robert, ScientiDev, Feb. 2020
! ***************************************************************0**


  use rff_param_def
  use rff_data_def

  real(kind=8)      :: Ga,Ex,r
  character(len=27) :: credate
  character(len=70) :: rff_name
  character(len=3)  :: sdata

!-------------------------------------------------------------------0--
  print*, 'kinf od data ? ULF,MAG or POS'
  read *, sdata
  print*, sdata

  if(sdata /= 'ULF' .and. sdata /= 'MAG' .and. sdata /= 'POS') then
        print*, 'Only ULF, MAG or POS'
        stop 'create_simulated_data.exe        : *** ABORTED ! Only ULG, MAG or POS'
  endif
!---------------------------------------------------------------------

  call set_common_param


!------------------------
  IF(sdata == 'ULF') THEN
!------------------------

! ===========================================================
  print*, 'CREATE RFF FILE OF SIMULATED ULF DATA'
! ===========================================================

  rff_name='VTL2_simulated_ULF_data.rff'

  manda_param%FILE_NAME                 = rff_name
  manda_param%EXPERIMENT_NAME           = 'SIM_ULF'
  manda_param%INSTRUMENT_TYPE           = 'alternative_magnetometer'
  manda_param%MEASUREMENT_TYPE          = 'ULF B field'
  manda_param%DATA_LABEL                = 'Bx ; By ; Bz'
  manda_param%DATA_UNITS                = 'nT ; nT ; nT'
  manda_param%DATA_COORDINATE_SYSTEM    = 'GSE'

!---------------------------------------------------------------------
  print*, 'define ULF simulated data'

! time duration: 3 hours from 2020-02-05T15:00:00.000000Z

  deltaT=0.04
  nbp=int(3.*3600./deltaT +0.5)

  print*, 'nbp=',nbp

  allocate(R_data_vector(3,nbp))
  allocate(data_index(nbp))

  data_index(1)='2020-02-05T15:00:00.000000Z'

! *   computation of a single plane elliptic-polarized waves
!     in a coordinate system where K vector is defined.
!     The frequency is variing between f1 and f2.

  akx=0.5
  aky=0.5
  akz=1.

  a=10.
  b= 9.
  f1=3.
  f2=8.
  dtvf=3600.
  akdr=0.

  call co_plane_wave_fv(akx,aky,akz,a,b,f1,f2,dtvf,nbp,deltaT,akdr, &
                        R_data_vector(1,:),R_data_vector(2,:),   R_data_vector(3,:) )

!     select wawe span time and modify intensity

  nvdt=int(dtvf/deltaT) ! nb pts to go from f1 to f2
  nseq=nbp/nvdt       ! nb of sequences of nvdt pts

  i1=nvdt
  i2=(nseq-1)*nvdt
  pi=acos(-1.)

  do i=1,nbp
     if(i.lt.i1 .or. i.gt.i2) then
        do j=1,3
           if(i.le.i1) then
              R_data_vector(j,i)= 0.01*R_data_vector(j,i)*(sin(2.*pi*float(i-1)/float(i1)))**2
           else
              R_data_vector(j,i)= 0.01*R_data_vector(j,i)*(sin(2.*pi*float(i-i2)/float(nbp-i2)))**2
           endif
        end do
     else
        do j=1,3
           R_data_vector(j,i)= R_data_vector(j,i)*sin(float(nseq-1)*pi*float(i-i1)/float(i2-i1))
        end do
     endif
  end do

!     add random noise of 1/100 nt on the full period
!     and define time index

  do i=1,nbp
     t=float(i-1)*deltaT
     call addsec_datiso('2020-02-05T15:00:00.000000Z',t,data_index(i))

        do j=1,3
           call crandom01(ranval,1.e-2)
           R_data_vector(j,i)= R_data_vector(j,i) +ranval
        end do
  end do

!---------------------------------------------------------------------
  print*, 'complete meta-data'

  manda_param%BLOCK_NUMBER              = nbp
  manda_param%BLOCK_FIRST_INDEX         = data_index(1)
  manda_param%BLOCK_LAST_INDEX          = data_index(nbp)
  manda_param%DATA_LABEL                = 'Bx ; By ; Bz'
  manda_param%DATA_UNITS                = 'nT ; nT ; nT'
  manda_param%DATA_COORDINATE_SYSTEM    = 'GSE'
  optio_param%TIME_RESOLUTION           = deltaT
  optio_param%SUB_TITLE                 = 'nT in GSE system'
  const_data%SAMPLE_RATE                =  1./deltaT

  call rff_set_default_DATA_DESCRIPTION
  call rff_set_default_BLOCK_DESCRIPTION
  call rff_set_default_INDEX_DESCRIPTION
  call rff_set_default_INDEX_EXTENSION_DESCRIP

  optio_param%EXPERIMENT_DESCRIPTION(1)  = 'Imaginary AC magnetometer'
  EX_DE = 1
!---------------------------------------------------------------------
  print*, 'create ULF RFF file'

  call rff_W_metadata(1,rff_name)
  call rff_W_const_data(1)
  call rff_W_indexed_data(1)
  call rff_W_tail(1)

!------------------------
  ENDIF
!------------------------


!------------------------
  IF(sdata == 'MAG') THEN
!------------------------

! ===========================================================
  print*, 'CREATE RFF FILE OF SIMULATED MAG DATA'
! ===========================================================

  rff_name='VTL2_simulated_MAG_data.rff'

  manda_param%FILE_NAME                 = rff_name
  manda_param%EXPERIMENT_NAME           = 'SIM_MAG'
  manda_param%INSTRUMENT_TYPE           = 'magnetometer'
  manda_param%MEASUREMENT_TYPE          = 'DC B field'
  optio_param%TITLE                     = 'Simulated MAG field'

!---------------------------------------------------------------------
  print*, 'define MAG simulated data'

! time duration: 3 hours from 2020-02-05T15:00:00.000000Z
! but not same sample frequency

  deltaT=1.
  nbp=int(3.*3600./deltaT +0.5)

  print*, 'nbp=',nbp

  allocate(R_data_vector(3,nbp))
  allocate(data_index(nbp))

  data_index(1)='2020-02-05T15:00:00.000000Z'

!     compute DC magnetic field

  do i=1,nbp

     t=float(i-1)*deltaT
     call addsec_datiso('2020-02-05T15:00:00.000000Z',t,data_index(i))

     x=  50.*float(i-1)/float(nbp)
     y=  50.*float(i-1)/float(nbp)
     z= 100.*float(i-1)/float(nbp)

     call crandom01(ranval,1.)

     R_data_vector(1,i)=x +ranval
     R_data_vector(2,i)=y -ranval
     R_data_vector(3,i)=z +ranval/2.

  end do

!---------------------------------------------------------------------
  print*, 'complete meta-data'

  manda_param%BLOCK_NUMBER              = nbp
  manda_param%BLOCK_FIRST_INDEX         = data_index(1)
  manda_param%BLOCK_LAST_INDEX          = data_index(nbp)
  manda_param%DATA_LABEL                = 'Bx ; By ; Bz'
  manda_param%DATA_UNITS                = 'nT ; nT ; nT'
  manda_param%DATA_COORDINATE_SYSTEM    = 'GSE'
  optio_param%TIME_RESOLUTION           = deltaT
  optio_param%SUB_TITLE                 = 'nT in GSE system'
  const_data%SAMPLE_RATE                =  1./deltaT

  call rff_set_default_DATA_DESCRIPTION
  call rff_set_default_BLOCK_DESCRIPTION
  call rff_set_default_INDEX_DESCRIPTION
  call rff_set_default_INDEX_EXTENSION_DESCRIP

  optio_param%EXPERIMENT_DESCRIPTION(1)  = 'Imaginary DC magnetometer'
  EX_DE = 1
!---------------------------------------------------------------------
  print*, 'create MAG RFF file'

  call rff_W_metadata(2,rff_name)
  call rff_W_const_data(2)
  call rff_W_indexed_data(2)
  call rff_W_tail(2)
!------------------------
  ENDIF
!------------------------


!------------------------
  IF(sdata == 'POS') THEN
!------------------------

! ===========================================================
  print*, 'CREATE RFF FILE OF SIMULATED POS DATA'
! ===========================================================

!---------------------------------------------------------------------
  print*, 'define mandatory parameters, optional parameter and constant data'

  rff_name='VTL2_simulated_POS_data.rff'

  manda_param%FILE_NAME                 = rff_name
  manda_param%EXPERIMENT_NAME           = 'SIM_POS'
  manda_param%INSTRUMENT_TYPE           = 'Aux'
  manda_param%MEASUREMENT_TYPE          = 'Positions'
  optio_param%TITLE                     = 'Simulated S/C position'
  optio_param%SUB_TITLE                 = 'Re in GSE system'

!---------------------------------------------------------------------
  print*, 'define POS simulated data'

!     Earth mass (kg)
  GM=5.9736e24
!     Exccentricity
  Ex=0.7D0
!     semi major axis of the elliptical trajectory (km)
  Ga=80000.D0

  Ga1=1.
  Ga2=0.
  Ga3=0.

  Gb1=0.
  Gb2=1.
  Gb3=0.

! computation of revolution time period (s)

  call co_period(Ga, GM, GT)

  GTh= GT/3600.

  print*,'revolution time in sec. : ',GT
  print*,'revolution time in hours: ',GTh

! time duration: one revolution from 2020-02-05T15:00:00.000000Z
! sample time: 60s.

  deltaT=60.
  nbp=int(GT/deltaT +0.5)

  print*, 'nbp=',nbp

  allocate(R_data_vector(3,nbp))
  allocate(data_index(nbp))

  data_index(1)='2020-02-05T15:00:00.000000Z'

! compute trajectory; suppose starting time at perigee (Kepler's law)

  do i=1,nbp

     time=float(i-1)*deltaT

     call addsec_datiso('2020-02-05T15:00:00.000000Z',time,data_index(i))
     call co_anom_excen(Ex,time,GT,AE)
     call co_anom_vraie(AE,Ex,Ga,AV,r)
     call posgei(r,AV,Ga1,Ga2,Ga3,Gb1,Gb2,Gb3,1., x,y,z)

     call decode_datiso(data_index(i),iyear,imon,iday,ih,im,is,ims,imc)
     call cp_time_param(iyear,imon,iday,ih,im,is)
     call t_gei_to_gse(x,y,z, xgse,ygse,zgse)

     R_data_vector(1,i)=xgse/6378.
     R_data_vector(2,i)=ygse/6378.
     R_data_vector(3,i)=zgse/6378.

  end do

!---------------------------------------------------------------------
  print*, 'complete meta-data'

  manda_param%BLOCK_NUMBER              = nbp
  manda_param%BLOCK_FIRST_INDEX         = data_index(1)
  manda_param%BLOCK_LAST_INDEX          = data_index(nbp)
  manda_param%DATA_LABEL                = 'Px ; Py ; Pz'
  manda_param%DATA_UNITS                = 'Re ; Re ; Re'
  manda_param%DATA_COORDINATE_SYSTEM    = 'GSE'
  optio_param%TIME_RESOLUTION           = deltaT
  optio_param%SUB_TITLE                 = 'Rein GSE system'
  const_data%SAMPLE_RATE                =  1./deltaT

  call rff_set_default_DATA_DESCRIPTION
  call rff_set_default_BLOCK_DESCRIPTION
  call rff_set_default_INDEX_DESCRIPTION
  call rff_set_default_INDEX_EXTENSION_DESCRIP

  optio_param%EXPERIMENT_DESCRIPTION(1)  = 'Imaginary Position recorder'
  EX_DE = 1
!---------------------------------------------------------------------
  print*, 'create POS RFF file'

  call rff_W_metadata(3,rff_name)
  call rff_W_const_data(3)
  call rff_W_indexed_data(3)
  call rff_W_tail(3)

!------------------------
  ENDIF
!------------------------

! ===========================================================
  print*, END
! ===========================================================

  print*
  print*, 'create_simulated_data.exe        : NORMAL TERMINATION'
     stop 'create_simulated_data.exe        : NORMAL TERMINATION'
!XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
contains
!XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine set_common_param

  call rff_set_default_init
  call gdatiso(credate)

  manda_param%FILE_CLASS                = 'VecTime'
  manda_param%FILE_FORMAT_VERSION       = 'Roproc_Format_File V 2.3'
  manda_param%FILE_CREATION_DATE        = credate
  manda_param%MISSION_NAME              = 'IMAGINE'
  manda_param%OBSERVATORY_NAME          = 'Ima_obs'
  manda_param%OBSERVATORY_NUMBER        = 1
  manda_param%EXPERIMENT_MODE           = 'Normal'
  manda_param%INDEX_LABEL               = 'Time'
  manda_param%INDEX_TYPE                = 'STR'
  manda_param%INDEX_UNITS               = 'ISO_TIME'
  manda_param%INDEX_FORMAT              = '(a27)'
  manda_param%INDEX_FORM                = 'Scalar'
  manda_param%INDEX_DIMENSION           = 1
  manda_param%INDEX_PROPERTIES          = 'Regularly Spaced'
  manda_param%INDEX_EXTENSION_LABEL     = 'No'
  manda_param%INDEX_EXTENSION_TYPE      = 'STR'
  manda_param%INDEX_EXTENSION_UNITS     = 'None'
  manda_param%INDEX_EXTENSION_FORMAT    = 'None'
  manda_param%INDEX_EXTENSION_LENGTH    = 0
  manda_param%DATA_TYPE                 = 'FLT'
  manda_param%DATA_FORMAT               = '(3e15.6)'
  manda_param%DATA_FORM                 = 'Vector'
  manda_param%DATA_DIMENSION            = 3
  manda_param%DATA_REPRESENTATION       = 'xyz Cartesian'
  manda_param%DATA_FILL_VALUE           = '1.e30'

  optio_param%DISCIPLINE_NAME           = 'Space and  Magnetospheric Physics'
  optio_param%EXPERIMENT_PI_NAME        = 'No'
  optio_param%EXPERIMENT_PI_MAIL        = 'patrick.robert@lpp.polytechnique.fr'

  call rff_set_default_DATA_DESCRIPTION
  call rff_set_default_BLOCK_DESCRIPTION
  call rff_set_default_INDEX_DESCRIPTION
  call rff_set_default_INDEX_EXTENSION_DESCRIP

  optio_param%MISSION_DESCRIPTION(1)     = 'Imaginary mission'

  MI_DE = 1

  return

  end subroutine set_common_param

!XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  end program create_simulated_data

!XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX


