    !XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
    !
    ! Laboratoire de Physique des Plasmas
    ! Modules and subroutines to write CLUSTER/STAFF-SC cef files
    ! Derived from rw_ceflib of L. Mirioni, RPC project, 2008 February 22
    !
    ! Simplified and optimized by Rodrigue PIBERNE, April   2011
    ! Cleaning and comments       Patrick  ROBERT,  April   2011
    ! Cleaning and comments       Patrick  ROBERT,  March   2020
    !
    !XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

    module type_def_cef
    
    !----------------------------------------------------------------------!
    ! Object : modules for CEF type definitions
    ! Author : L. Mirioni, LPP, 2008
    !----------------------------------------------------------------------!

    integer, parameter :: field_length_cef=255! length of the field to be read
    integer, parameter :: line_length_cef=255 ! maximum length of a CEF file line
    !integer, parameter :: txt_line_length=80  ! maximum length of a TXT type line
    integer, parameter :: num_lines_cef=200   ! maximum number of lines CEF ENTRY
    integer, parameter :: num_lines_cef_data=2000 ! maximum number of lines CEF DATA

    type CEF_META
        character(len=field_length_cef) :: name
        character(len=field_length_cef) :: value_type
        character(len=line_length_cef)  :: entry(num_lines_cef)
    end type CEF_META

    type CEF_PARAMETER
        character(len=field_length_cef) :: dataset_id
        character(len=field_length_cef) :: parameter_id
        character(len=field_length_cef) :: parent_dataset_id
        character(len=field_length_cef) :: parameter_type
        character(len=field_length_cef) :: entity
        character(len=field_length_cef) :: property
        character(len=field_length_cef) :: fluctuations
        character(len=field_length_cef) :: error_plus
        character(len=field_length_cef) :: error_minus
        character(len=field_length_cef) :: compound
        character(len=field_length_cef) :: compound_def
        character(len=field_length_cef) :: units
        character(len=field_length_cef) :: si_conversion
        character(len=field_length_cef) :: tensor_order
        character(len=field_length_cef) :: coordinate_system
        character(len=field_length_cef) :: frame_origin
        character(len=field_length_cef) :: frame_velocity
        character(len=field_length_cef) :: frame
        character(len=field_length_cef) :: tensor_frame
        character(len=line_length_cef)  :: catdesc
        character(len=field_length_cef) :: representation_1
        character(len=field_length_cef) :: representation_2
        character(len=field_length_cef) :: sizes
        character(len=field_length_cef) :: depend_0
        character(len=line_length_cef)  :: data
        character(len=field_length_cef) :: label_1
        character(len=field_length_cef) :: delta_plus
        character(len=field_length_cef) :: delta_minus
        character(len=field_length_cef) :: value_type
        character(len=field_length_cef) :: significant_digits
        character(len=field_length_cef) :: fillval
        character(len=field_length_cef) :: quality
        character(len=field_length_cef) :: parameter_caveats
        character(len=field_length_cef) :: fieldnam
        character(len=field_length_cef) :: lablaxis
        character(len=field_length_cef) :: scalemin
        character(len=field_length_cef) :: scalemax
        character(len=field_length_cef) :: scaletyp
        character(len=field_length_cef) :: displaytype
        character(len=field_length_cef) :: target_system
    end type CEF_PARAMETER
    end module type_def_cef

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

    function get_cef_paramC(param,ifc,data_reach,end_of_search)
    
    !----------------------------------------------------------------------!
    ! Object : Read cef file until searched Char. param. & return it
    ! Author : P. Robert , LPP, 2011 Jan.
    !----------------------------------------------------------------------!

    implicit none

    character(len=255)  :: get_cef_paramC
    character(len=*)   :: param, end_of_search
    character(len=255) :: com

    integer            ::eof, ifc, data_reach, i, k, ll, l_end

    ! -----------------------------------

    com='xxx'
    ll=LEN_TRIM(param)
    l_end=LEN_TRIM(end_of_search)

    DO WHILE (trim(adjustl(com(1:ll))) /= param(1:ll))
        READ(ifc,'(a)',iostat=eof) com
        IF(com(1:l_end) == end_of_search) THEN
            data_reach = 1
            REWIND ifc
            EXIT
        ENDIF
        IF(eof /= 0) EXIT
    ENDDO

    IF(com(1:l_end) == end_of_search) THEN
        if (end_of_search == 'DATA_UNTIL =') then
            write(*,*) '****DATA_UNTIL = "END_OF_FILE reached"'
        endif
        get_cef_paramC=' '
        RETURN
    ENDIF

    IF(eof /= 0) THEN
        write(*,*) '*** eof reached'
        write(*,*) '    Program aborted !!!'
        stop '*** EOF REACHED, program aborted'
    ENDIF

    i=INDEX(com,'=')
    IF (i /= 0) THEN
        k=LEN_TRIM(com)
        get_cef_paramC=com(i+1:k)
    ELSE
        write(*,*) '*** ',TRIM(param),' cannot be found'
    ENDIF

    RETURN
    END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

    function get_cef_paramC_new(param,ifc,data_reach,end_of_search)
    
    !----------------------------------------------------------------------!
    ! Object : Read cef file until searched Char. param. & return it
    ! Author : R. Piberne, LPP, 2012 Feb.
    ! Modified version of get_cef_paramC
    !----------------------------------------------------------------------!

    implicit none

    character(len=255)  :: get_cef_paramC_new
    character(len=*)   :: param, end_of_search
    character(len=300) :: com, com_temp

    integer            ::eof, ifc, data_reach, i, k, ll, l_end

    ! -----------------------------------

    com='xxx'
    com_temp = com
    ll=LEN_TRIM(param)
    l_end=LEN_TRIM(end_of_search)

    DO WHILE (trim(adjustl(com_temp(1:ll))) /= param(1:ll))
        READ(ifc,'(a)',iostat=eof) com
        com_temp = com
        call suppress_blank(com_temp)
        IF(com(1:l_end) == end_of_search) THEN
            data_reach = 1
            EXIT
        ENDIF
        IF(eof /= 0) EXIT
    ENDDO

    IF(com(1:l_end) == end_of_search) THEN
        get_cef_paramC_new=' '
        RETURN
    ENDIF

    IF(eof /= 0) THEN
        write(*,*) '*** eof reached'
        write(*,*) '    Program aborted !!!'
        stop '*** EOF REACHED, program aborted'
    ENDIF

    i=INDEX(com,'=')
    IF (i /= 0) THEN
        k=LEN_TRIM(com)
        get_cef_paramC_new=com(i+1:k)
    ELSE
        write(*,*) '*** ',TRIM(param),' cannot be found'
    ENDIF

    RETURN
    END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

    subroutine go_back_to_the_beginning(ifc, param_name)
    
    !----------------------------------------------------------------------!
    ! Object : Rewind a file until START_VARIABLE=param_name is found
    ! Author : R. Piberne, LPP, 2012 Feb.
    ! Inputs:
    !        ifc = integer, file unit.
    !        param_name= character, keyword to look for.
    !----------------------------------------------------------------------!

    implicit none

    character(len=*) :: param_name
    integer          ::ifc,status

    rewind (ifc,iostat=status)
    call go_to_keyword('START_VARIABLE='//param_name,ifc)

    end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

    subroutine go_to_keyword(param, ifc)
    
    !----------------------------------------------------------------------!
    ! Object : Read cef file until searched parameter is found
    ! Author : R. Piberne, LPP, 2011 Nov.
    ! Inputs:
    !        param= character, keyword to look for.
    !        ifc = integer, file unit.
    !----------------------------------------------------------------------!

    implicit none

    character(len=*)   :: param
    character(len=255) :: com, com_temp
    integer            ::eof, ifc, ll

    ! -----------------------------------

    com='xxx'
    com_temp=com
    ll=LEN_TRIM(param)

    DO WHILE (com_temp(1:ll) /= param(1:ll))
        READ(ifc,'(a)',iostat=eof) com
        com_temp = com
        call suppress_blank(com_temp)
        call suppress_tab(com_temp)
        if(com_temp(1:1) == '!') cycle
        IF(com(1:10) == 'DATA_UNTIL') EXIT
        IF(eof /= 0) EXIT
    ENDDO

    IF(com(1:10) == 'DATA_UNTIL') THEN
        if(param(1:ll) == 'DATA_UNTIL') then
        write(*,*) 'cef file read until "DATA_UNTIL" keyword'
                                       else
        write(*,*) '*** Keyword ',param(1:ll),' not found before "DATA_UNTIL"'
        endif
        RETURN
    ENDIF

    IF(eof /= 0) THEN
        write(*,*) '*** eof reached'
        write(*,*) '    Program aborted !!!'
        stop '*** EOF REACHED, program aborted'
    ENDIF

    RETURN
    END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

    subroutine cef_find_n_include(ifc,n_include)
    
    !----------------------------------------------------------------------!
    ! Object : Find number of include files in the cef header
    ! Author : R. Piberne, LPP, 2011 Nov.
    ! Inputs:
    !        ifc = integer, file unit.
    !        n_include= integer, number of include files.
    !----------------------------------------------------------------------!

    use type_def_cef

    implicit none

    integer        :: ifc,data_reach,n_include
    character(len=255)  :: get_cef_paramC, end_of_search
    character(len=255) :: dummy_name

    data_reach  = 0

    end_of_search = 'DATA_UNTIL'

    do while (data_reach == 0)
        dummy_name = TRIM(get_cef_paramC('INCLUDE',&
        ifc,data_reach, end_of_search))
        n_include = n_include + 1
    enddo

    n_include = n_include-1

! test PR
    if (abs(n_include) .GT. 100) then
                     write(*,*) 'n_include=',n_include
                     write(*,*) '*** PROGRAM ABORTED *** in subroutine cef_find_n_include'
                     stop '*** PROGRAM ABORTED *** in subroutine cef_find_n_include'
    endif

    ! -----------------------------------

    RETURN
    END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

    subroutine cef_find_n_metadata(ifc,n_metadata)
    
    !----------------------------------------------------------------------!
    ! Object : Find number of metadata in the cef header
    ! Author : R. Piberne, LPP, 2011 Nov.
    ! Input :
    !        ifc = integer, file unit.
    !        cef_metadata= integer, number of metadata.
    !----------------------------------------------------------------------!

    use type_def_cef

    implicit none

    integer        :: ifc,data_reach,n_metadata
    character(len=255)  :: get_cef_paramC, end_of_search
    character(len=255) :: dummy_name

    data_reach  = 0

    end_of_search = 'DATA_UNTIL'

    do while (data_reach == 0)
        dummy_name = TRIM(get_cef_paramC('START_META',&
        ifc,data_reach, end_of_search))
        n_metadata = n_metadata + 1
    enddo

    n_metadata = n_metadata-1

    ! -----------------------------------

    RETURN
    END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

    subroutine cef_find_n_variables(ifc,n_variables)
    
    !----------------------------------------------------------------------!
    ! Object : Find number of variables in the cef header
    ! Author : R. Piberne, LPP, 2011 Nov.
    ! Input :
    !        ifc = integer, file unit.
    !        n_variables= integer, number of variables.
    !----------------------------------------------------------------------!

    use type_def_cef

    implicit none

    integer        :: ifc,data_reach,n_variables
    character(len=255)  :: get_cef_paramC, end_of_search
    character(len=255) :: dummy_name

    data_reach  = 0

    end_of_search = 'DATA_UNTIL'

    rewind(ifc)

    do while (data_reach == 0)
        dummy_name = TRIM(get_cef_paramC('START_VARIABLE',&
        ifc,data_reach, end_of_search))
        n_variables = n_variables + 1
    enddo

    n_variables = n_variables-1

    ! -----------------------------------

    RETURN
    END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

    subroutine cef_R_include(ifc,n_include,include_names,include_dirname)

    !----------------------------------------------------------------------!
    ! Object : Read include filenames in the cef file
    ! Author : R. Piberne, LPP, 2011 Nov.
    ! Input :
    !        ifc = integer, file unit.
    !        n_include = integer,number of include files.
    !        include_names = character array, basename of include files
    !        include_dirname = character, directory name of the include files
    !----------------------------------------------------------------------!

    use type_def_cef

    implicit none

    integer            :: ifc,i,data_reach,n_include,index_ceh
    character(len=255)  :: get_cef_paramC, end_of_search
    character(len=*)   :: include_dirname
    character(len=*), dimension(n_include) :: include_names

    data_reach= 0

    end_of_search = 'DATA_UNTIL = "END_OF_FILE"'

    do i=1, n_include
        include_names(i) = TRIM(get_cef_paramC('INCLUDE', &
        ifc,data_reach,end_of_search))
        index_ceh=index(include_names(i),'ceh')
        include_names(i)=include_names(i)(1:index_ceh+3)
        include_names(i)=trim(include_dirname)//trim(adjustl(include_names(i)))
    enddo

    ! -----------------------------------

    RETURN
    END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

    subroutine cef_R_metadata(ifc,n_metadata, metadata_names)
    
    !----------------------------------------------------------------------!
    ! Object : Read  mandatory parameter in the cef file
    ! Author : R. Piberne, LPP, 2011 Nov.
    ! Input :
    !        ifc = integer, file unit.
    !        n_metadata = integer, number of metadata.
    !        metadata_names = character array, metadata names
    !----------------------------------------------------------------------!

    use type_def_cef

    implicit none

    integer        :: ifc,i,data_reach,n_metadata
    character(len=255)  :: get_cef_paramC, end_of_search
    character(len=*), dimension(n_metadata) :: metadata_names

    data_reach= 0

    end_of_search = 'DATA_UNTIL ='

    do i=1, n_metadata
        metadata_names(i) = TRIM(adjustl(get_cef_paramC('START_META ', &
        ifc,data_reach,end_of_search)))
    enddo

    ! -----------------------------------

    RETURN
    END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

    subroutine cef_R_entry(ifc,cef_entry,n_metadata, metadata_names)
    
    !----------------------------------------------------------------------!
    ! Object : Read metadata entry in the cef file
    ! Author : R. Piberne, LPP, 2012 Jan.
    ! Input :
    !        ifc = integer, file unit.
    !        cef_entry= entry structure
    !        n_metadata= integer, number of metadata
    !        metadata_names = character array, names of metadata
    !----------------------------------------------------------------------!

    use type_def_cef

    implicit none

    character(len=255)  :: end_of_search
    character(len=255) :: get_cef_paramC_new
    character(len=*), dimension(*)   :: metadata_names
    integer            :: data_reach, blank, n_metadata,i,j,ifc,stop_loop
    type(CEF_META), dimension(*):: cef_entry
    type(CEF_META):: cef_meta_temp

    ! -----------------------------------

    data_reach = 0
    blank = 4
    end_of_search = 'END_META'

    do i=1,n_metadata
        stop_loop = 0
        j = 1
        rewind (ifc)
        call go_to_keyword('START_META='//metadata_names(i),ifc)
        cef_meta_temp%value_type = TRIM(get_cef_paramC_new('VALUE_TYPE',&
        ifc,data_reach, end_of_search))
        rewind(ifc)
        call go_to_keyword('START_META='//metadata_names(i),ifc)
        do while(stop_loop == 0)
            cef_meta_temp%entry(j)   = TRIM(adjustl(get_cef_paramC_new('ENTRY',&
            ifc,data_reach, end_of_search)))
            if(cef_meta_temp%entry(j) == ' ') then
                stop_loop = 1
            endif
            j=j+1
        end do

        cef_entry(i) = cef_meta_temp

    enddo

    RETURN
    end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

    subroutine cef_R_variables(ifc,n_variables,variable_names)
    
    !----------------------------------------------------------------------!
    ! Object : Read variable parameter in the cef file
    ! Author : R. Piberne, LPP, 2011 Nov.
    ! Input :
    !        ifc = integer, file unit.
    !        n_variables = integer, number of variables.
    !        variable_names = character array, names of variables
    !----------------------------------------------------------------------!

    use type_def_cef

    implicit none

    integer        :: ifc,i,data_reach,n_variables
    character(len=255)  :: get_cef_paramC, end_of_search
    character(len=*), dimension(n_variables) :: variable_names

    data_reach= 0

    end_of_search = 'DATA_UNTIL = "END_OF_FILE"'

    do i=1, n_variables
        variable_names(i) = TRIM(adjustl(get_cef_paramC('START_VARIABLE', &
        ifc,data_reach,end_of_search)))
    enddo

    ! -----------------------------------

    RETURN
    END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

    subroutine cef_R_parameter(ifc,cef_param,n_variables,variable_names)
    
    !----------------------------------------------------------------------!
    ! Object : Read parameters inside a variable of a cef file
    ! Author : R. Piberne, LPP, 2011 Nov.
    ! Input :
    !        ifc = integer, file unit.
    !        cef_param= CEF_PARAMETER array, structure of parameters defined
    !                   in type_def_cef
    !        n_variables = integer, number of variables.
    !        variable_names = character array, names of variables
    !----------------------------------------------------------------------!

    use type_def_cef

    implicit none

    character(len=255)  :: get_cef_paramC_new, end_of_search
    character(len=*), dimension(*)   :: variable_names
    integer            :: data_reach, blank, n_variables,i, ifc
    type(CEF_PARAMETER), dimension(*):: cef_param
    type(CEF_PARAMETER):: cef_param_temp

    ! -----------------------------------

    data_reach = 0
    blank = 4
    end_of_search = 'END_VARIABLE'
    rewind (ifc)

    do i=1,n_variables

        !WARNING: Due to an unknown bug, if a write or a print is inserted in this loop,
        !the program will stop without any error message

        !Go to the wanted variable
        call go_to_keyword('START_VARIABLE='//variable_names(i),ifc)

        !Find the wanted parameter
        cef_param_temp%catdesc            = TRIM(adjustl(get_cef_paramC_new('CATDESC',&
        ifc,data_reach, end_of_search)))

        !When the parameter is found, go back to "START_VARIABLE=wanted variable"
        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%dataset_id         = TRIM(adjustl(get_cef_paramC_new('DATASET_ID',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%parameter_id       = TRIM(adjustl(get_cef_paramC_new('PARAMETER_ID',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%parent_dataset_id  = TRIM(adjustl(get_cef_paramC_new('PARENT_DATASET_ID',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%parameter_type     = TRIM(adjustl(get_cef_paramC_new('PARAMETER_TYPE',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%entity             = TRIM(adjustl(get_cef_paramC_new('ENTITY',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%property           = TRIM(adjustl(get_cef_paramC_new('PROPERTY',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%fluctuations       = TRIM(adjustl(get_cef_paramC_new('FLUCTUATIONS',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%error_plus         = TRIM(adjustl(get_cef_paramC_new('ERROR_PLUS',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%error_minus        = TRIM(adjustl(get_cef_paramC_new('ERROR_MINUS',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%compound           = TRIM(adjustl(get_cef_paramC_new('COMPOUND',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%compound_def       = TRIM(adjustl(get_cef_paramC_new('COMPOUND_DEF',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%units              = TRIM(adjustl(get_cef_paramC_new('UNITS',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%si_conversion      = TRIM(adjustl(get_cef_paramC_new('SI_CONVERSION',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%tensor_order       = TRIM(adjustl(get_cef_paramC_new('TENSOR_ORDER',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%coordinate_system  = TRIM(adjustl(get_cef_paramC_new('COORDINATE_SYSTEM',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%frame_origin       = TRIM(adjustl(get_cef_paramC_new('FRAME_ORIGIN',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%frame_velocity     = TRIM(adjustl(get_cef_paramC_new('FRAME_VELOCITY',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%frame              = TRIM(adjustl(get_cef_paramC_new('FRAME',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%tensor_frame       = TRIM(adjustl(get_cef_paramC_new('TENSOR_FRAME',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%representation_1   = TRIM(adjustl(get_cef_paramC_new('REPRESENTATION_1',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%representation_2   = TRIM(adjustl(get_cef_paramC_new('REPRESENTATION_2',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%sizes              = TRIM(adjustl(get_cef_paramC_new('SIZES',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%depend_0           = TRIM(adjustl(get_cef_paramC_new('DEPEND_0',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%data               = TRIM(adjustl(get_cef_paramC_new('DATA',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%label_1            = TRIM(adjustl(get_cef_paramC_new('LABEL_1',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%delta_plus         = TRIM(adjustl(get_cef_paramC_new('DELTA_PLUS',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%delta_minus        = TRIM(adjustl(get_cef_paramC_new('DELTA_MINUS',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%value_type         = TRIM(adjustl(get_cef_paramC_new('VALUE_TYPE',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%significant_digits = TRIM(adjustl(get_cef_paramC_new('SIGNIFICANT_DIGITS',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%fillval            = TRIM(adjustl(get_cef_paramC_new('FILLVAL',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%quality            = TRIM(adjustl(get_cef_paramC_new('QUALITY',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%parameter_caveats  = TRIM(adjustl(get_cef_paramC_new('PARAMETER_CAVEATS',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%fieldnam           = TRIM(adjustl(get_cef_paramC_new('FIELD_NAM',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%lablaxis           = TRIM(adjustl(get_cef_paramC_new('LABLAXIS',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%scalemin           = TRIM(adjustl(get_cef_paramC_new('SCALEMIN',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%scalemax           = TRIM(adjustl(get_cef_paramC_new('SCALEMAX',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%scaletyp           = TRIM(adjustl(get_cef_paramC_new('SCALETYP',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%displaytype        = TRIM(adjustl(get_cef_paramC_new('DISPLAYTYPE',&
        ifc,data_reach, end_of_search)))

        call go_back_to_the_beginning(ifc, variable_names(i))
        cef_param_temp%target_system      = TRIM(adjustl(get_cef_paramC_new('TARGET_SYSTEM',&
        ifc,data_reach, end_of_search)))

        cef_param(i) = cef_param_temp

    enddo

    RETURN
    end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

    subroutine cef_R_indexed_data(ifc,n_of_vec_in_file,n_block,indexed_data)
    
    !----------------------------------------------------------------------!
    ! Object : Read indexed data for cef files
    ! Author : R. Piberne, LPP, 2012 March
    !----------------------------------------------------------------------!

    implicit none

    integer :: ifc, n_of_vec_in_file, n_block, eof, comma_index,dollar_index,&
    i, j
    character(len=*), dimension(n_of_vec_in_file,n_block) ::indexed_data
    character(len=255) :: com, dummy_char, dummy_char2

    rewind(ifc)

    call go_to_keyword('DUMMY',ifc)

    do i=1, n_block

        READ(ifc,'(a)',iostat=eof) com

        dummy_char = com

        do j=1, n_of_vec_in_file-1

            comma_index = index(dummy_char, ',')
            dummy_char2 = dummy_char(1:comma_index-1)
            dummy_char  = dummy_char(comma_index+1:255)

            indexed_data(j,i) = trim(dummy_char2)

        enddo

        dollar_index = index(dummy_char, '$')
        if (dollar_index /= 0) then
            dummy_char = dummy_char(1:dollar_index-1)
        endif
        indexed_data(n_of_vec_in_file,i) = trim(dummy_char)

    enddo

    end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

    subroutine concatenate_include_files(ifc,include_dirname,cef_file_name,n_block,block_first_index,block_last_index)
    
    !----------------------------------------------------------------------!
    ! Object : Concatenate ceh files to CEF files if necessary
    ! Author : R. Piberne, LPP, 2012 Jan.
    !
    ! Input :
    !         ifc = integer, file unit.
    !         include_dirname = character, name of the directory where the
    !                           include files are stored
    !         cef_file_name = character, name of the cef file
    !         n_block = integer, number of blocks
    !         block_first_index = character, first data line
    !         block_last_index = character, last data line
    !
    !----------------------------------------------------------------------!

    implicit none

    integer            :: n_include, ifc, read_unit, write_unit, err, eof,&
    i, start_count,n_block

    character(len=*)   :: include_dirname, cef_file_name
    character(len=255) :: com, Fbasename, Fdirname, cef_file_name_temp
    character(len=255), dimension(:), allocatable :: include_names
    character(len=27)  :: block_first_index, block_last_index

    !inititalisation
    read_unit=2
    write_unit=3
    com='xxx'
    start_count=0
! PR
    n_include=0

    !find number of include files
    OPEN(ifc,file=trim(cef_file_name),status='old',action='read', iostat=err)
    call cef_find_n_include(ifc,n_include)

    !if there is no include files, no concatenation is done
    if (n_include == 0) then

        rewind(ifc)
        eof=0
        DO WHILE (eof == 0)
            if(com(1:12) == 'DATA_UNTIL =') then
                n_block=0
                start_count=1
            endif
            READ(ifc,'(a)',iostat=eof) com
            ! A inclure et tester pour les VT !!!!
            if ((com(1:7) == "INCLUDE").or.(com(1:1) =='!').or.(eof == 1)) then
                CYCLE
            endif
            WRITE(write_unit,'(a)') trim(com)
            if(start_count == 1) then
                n_block=n_block+1
            endif
            if(n_block == 1) then
                block_first_index=com(1:27)
            endif
            block_last_index=com(1:27)
        ENDDO
        close(ifc)

        RETURN

    end if

    !read include files
    print*, 'n_include=',n_include
    allocate (include_names(n_include))
    call cef_R_include(ifc,n_include,include_names,include_dirname)
    CLOSE(ifc)

    !write first FILE_NAME and FILE_FORMAT
    cef_file_name_temp=trim(Fdirname(cef_file_name))//'/concatenated_'//&
    trim((Fbasename(cef_file_name)))
    OPEN(write_unit,file=trim(cef_file_name_temp),action='write', iostat=err)
    OPEN(ifc,file=trim(cef_file_name),action='read', iostat=err)
    DO i=1, 7
        READ(ifc,'(a)',iostat=eof) com
        WRITE(write_unit,'(a)') trim(com)
    ENDDO

    !concatenate include files
    do i=1,n_include
        OPEN(read_unit,file=trim(include_names(i)),status='old',action='read', iostat=err)
        eof=0
        DO WHILE (eof == 0)
            READ(read_unit,'(a)',iostat=eof) com
            if(eof == 0) then
                WRITE(write_unit,'(a)') trim(com)
            endif
        ENDDO
        CLOSE(read_unit)
        write(write_unit,'(a)')
    enddo

    !add cef file
    eof=0
    DO WHILE (com /= 'END_OF_FILE')
        if(com == 'DATA_UNTIL = "END_OF_FILE"') then
            n_block=0
            start_count=1
        endif
        READ(ifc,'(a)',iostat=eof) com
        if(com(1:7) == "INCLUDE") then
            CYCLE
        endif
        if(com(1:1) == '!') then
            CYCLE
        endif
        WRITE(write_unit,'(a)') trim(com)
        if(start_count == 1) then
            n_block=n_block+1
        endif
        if(n_block == 1) then
             block_first_index=com(1:27)
        endif
    ENDDO
    n_block=n_block-1
    backspace(ifc)
    backspace(ifc)
    READ(ifc,'(a)',iostat=eof) com
    block_last_index=com(1:27)
    CLOSE(ifc)
    CLOSE(write_unit)

    cef_file_name = cef_file_name_temp

    RETURN
    end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

    subroutine w_cef_meta(output_file_unit,dummy_cef_meta)
    
    !----------------------------------------------------------------------!
    ! Object : Writes Meta Data dummy_cef_meta in output_file_unit
    ! Author : L. Mirioni, LPP, 2005, Rev RP 2011
    ! Revis.: R. Piberne, LPP, 2011 Apr 01
    !
    ! Input :
    !        dummy_cef_meta
    ! Output:
    !        output_file_unit
    !----------------------------------------------------------------------!
    
    !------------------------!
    ! Declaration Statements !
    !------------------------!

    use type_def_cef

    implicit none

    integer :: output_file_unit
    type(CEF_META) :: dummy_cef_meta
    integer :: i

    !-----------------------!
    ! Executable Statements !
    !-----------------------!

    write(output_file_unit,101,err=199) dummy_cef_meta%name(1:len_trim(dummy_cef_meta%name))
    if (dummy_cef_meta%value_type(1:1) /= ' ') then 
        write(output_file_unit,102,err=199) dummy_cef_meta%value_type(1:len_trim(dummy_cef_meta%value_type))
    end if
    do i=1,num_lines_cef
        if (dummy_cef_meta%entry(i)(1:1) /= ' ') then 
            if (dummy_cef_meta%entry(i)(1:1) == '@') then ! '@' character is set if no quotes are required
                write(output_file_unit,103,err=199) dummy_cef_meta%entry(i)(2:len_trim(dummy_cef_meta%entry(i)))
            else
                if (dummy_cef_meta%name == 'FILE_TIME_SPAN') then 
                        write(output_file_unit,103,err=199) dummy_cef_meta%entry(i)(1:len_trim(dummy_cef_meta%entry(i)))
                else
                        write(output_file_unit,104,err=199) dummy_cef_meta%entry(i)(1:len_trim(dummy_cef_meta%entry(i)))
                endif
            endif
        end if
    end do
    write(output_file_unit,105,err=199) dummy_cef_meta%name(1:len_trim(dummy_cef_meta%name))



101 format('START_META = ',a)
102 format('   VALUE_TYPE = ',a)
103 format('   ENTRY = ',a)
104 format('   ENTRY = "',a,'"')
105 format('END_META = ',a,/,'!')


    !-----!
    ! End !
    !-----!
    return

    !----------------!
    ! On error exits !
    !----------------!

199 write(*,*)
    write(*,*)
    close(UNIT=output_file_unit)
    write(*,*) ' *** ERROR  W_CEF_META / CAN''T WRITE INTO CEF OUTPUT FILE. /***'
     stop 'lib_rw_cef.o/w_cef_meta         : *** ERROR !! Program aborted !'

    end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

    subroutine w_cef_parameter(output_file_unit,dummy_var)
    
    !----------------------------------------------------------------------!
    ! Object : Writes var Data dummy_var in file output_file_unit.
    ! Author : L. Mirioni, LPP, 2005, Rev RP 2011
    ! Revis.: R. PIBERNE, LPP, 2011 Apr 01
    !
    ! Input:
    !   dummy_var
    ! Output:
    !   output_file_unit
    !----------------------------------------------------------------------!

    !------------------------!
    ! Declaration Statements !
    !------------------------!

    use type_def_cef

    implicit none

    integer :: output_file_unit
    type(CEF_PARAMETER) :: dummy_var

    !-----------------------!
    ! Executable Statements !
    !-----------------------!

    ! PARAMETER START
    write(output_file_unit,107,err=199) dummy_var%parameter_id(1:len_trim(dummy_var%parameter_id)), &
    dummy_var%dataset_id(1:len_trim(dummy_var%dataset_id))
107 format('START_VARIABLE = ',a,'__',a)

    ! CATDESC
    if (dummy_var%catdesc(1:1) /= ' ') then
        write(output_file_unit,137,err=199) dummy_var%catdesc(1:len_trim(dummy_var%catdesc))
    end if
137 format('   CATDESC            = "',a,'"')

    ! COMPOUND
    if (dummy_var%compound(1:1) /= ' ') then
        write(output_file_unit,197,err=199) dummy_var%compound(1:len_trim(dummy_var%compound))
    end if
197 format('   COMPOUND           = "',a,'"')

    ! COMPOUND_DEF
    if (dummy_var%compound_def(1:1) /= ' ') then
        write(output_file_unit,207,err=199) dummy_var%compound_def(1:len_trim(dummy_var%compound_def))
    end if
207 format('   COMPOUND_DEF       = "',a,'"')

    ! COORDINATE_SYSTEM
    if (dummy_var%coordinate_system(1:1) /= ' ') then
        write(output_file_unit,247,err=199) dummy_var%coordinate_system(1:len_trim(dummy_var%coordinate_system))
    end if
247 format('   COORDINATE_SYSTEM  = "',a,'"')

    ! DATA
    if (dummy_var%data(1:1) /= ' ') then
        write(output_file_unit,327,err=199) dummy_var%data(1:len_trim(dummy_var%data))
    end if
327 format('   DATA               = ',a)

    ! DELTA_MINUS
    if (dummy_var%delta_minus(1:1) /= ' ') then
        write(output_file_unit,357,err=199) dummy_var%delta_minus(1:len_trim(dummy_var%delta_minus))
    end if
357 format('   DELTA_MINUS        = ',a)

    ! DELTA_PLUS
    if (dummy_var%delta_plus(1:1) /= ' ') then
        write(output_file_unit,347,err=199) dummy_var%delta_plus(1:len_trim(dummy_var%delta_plus))
    end if
347 format('   DELTA_PLUS         = ',a)

    ! DEPEND_0
    if (dummy_var%depend_0(1:1) /= ' ') then
        write(output_file_unit,307,err=199) 0,dummy_var%depend_0(1:len_trim(dummy_var%depend_0)), &
        dummy_var%dataset_id(1:len_trim(dummy_var%dataset_id))
    end if
307 format('   DEPEND_',i1.1,'           = ',a,'__',a)

    ! DISPLAYTYPE
    if (dummy_var%displaytype(1:1) /= ' ') then
        write(output_file_unit,467,err=199) dummy_var%displaytype(1:len_trim(dummy_var%displaytype))
    end if
467 format('   DISPLAYTYPE        = "',a,'"')

    ! TARGET_SYSTEM
    if (dummy_var%target_system(1:1) /= ' ') then
        write(output_file_unit,468,err=199) dummy_var%target_system(1:len_trim(dummy_var%target_system))
    end if
468 format('   TARGET_SYSTEM      = "',a,'"')

    ! ENTITY
    if (dummy_var%entity(1:1) /= ' ') then
        write(output_file_unit,147,err=199) dummy_var%entity(1:len_trim(dummy_var%entity))
    end if
147 format('   ENTITY             = "',a,'"')

    ! ERROR_MINUS
    if (dummy_var%error_plus(1:1) /= ' ') then
        write(output_file_unit,187,err=199) dummy_var%error_minus(1:len_trim(dummy_var%error_minus))
    end if
187 format('   ERROR_MINUS        = "',a,'"')

    ! ERROR_PLUS
    if (dummy_var%error_plus(1:1) /= ' ') then
        write(output_file_unit,177,err=199) dummy_var%error_plus(1:len_trim(dummy_var%error_plus))
    end if
177 format('   ERROR_PLUS         = "',a,'"')

    ! FIELDNAM
    if (dummy_var%fieldnam(1:1) /= ' ') then
        write(output_file_unit,417,err=199) dummy_var%fieldnam(1:len_trim(dummy_var%fieldnam))
    end if
417 format('   FIELDNAM           = "',a,'"')

    ! FILLVAL
    if (dummy_var%fillval(1:1) /= ' ') then
        write(output_file_unit,387,err=199) dummy_var%fillval(1:len_trim(dummy_var%fillval))
    end if
387 format('   FILLVAL            = ',a)

    ! FLUCTUATIONS
    if (dummy_var%fluctuations(1:1) /= ' ') then
        write(output_file_unit,167,err=199) dummy_var%fluctuations(1:len_trim(dummy_var%fluctuations))
    end if
167 format('   FLUCTUATIONS       = "',a,'"')

    ! FRAME
    if (dummy_var%frame(1:1) /= ' ') then
        write(output_file_unit,277,err=199) dummy_var%frame(1:len_trim(dummy_var%frame))
    end if
277 format('   FRAME              = "',a,'"')

    ! FRAME_ORIGIN
    if (dummy_var%frame_origin(1:1) /= ' ') then
        write(output_file_unit,257,err=199) dummy_var%frame_origin(1:len_trim(dummy_var%frame_origin))
    end if
257 format('   FRAME_ORIGIN       = "',a,'"')

    ! FRAME_VELOCITY
    if (dummy_var%frame_velocity(1:1) /= ' ') then
        write(output_file_unit,267,err=199) dummy_var%frame_velocity(1:len_trim(dummy_var%frame_velocity))
    end if
267 format('   FRAME_VELOCITY     = "',a,'"')

    ! LABEL_1
    if (dummy_var%label_1(1:1) /= ' ') then
        write(output_file_unit,338,err=199) dummy_var%label_1(1:len_trim(dummy_var%label_1))
    endif
338 format('   LABEL_1            = ',a)

    ! LABLAXIS
    if (dummy_var%lablaxis(1:1) /= ' ') then
        write(output_file_unit,427,err=199) dummy_var%lablaxis(1:len_trim(dummy_var%lablaxis))
    end if
427 format('   LABLAXIS           = "',a,'"')

    ! PARAMETER_CAVEATS
    if (dummy_var%parameter_caveats(1:1) /= ' ') then
        write(output_file_unit,407,err=199) dummy_var%parameter_caveats(1:len_trim(dummy_var%parameter_caveats))
    end if
407 format('   PARAMETER_CAVEATS  = ',a)

    ! PARAMETER_TYPE
    if (dummy_var%parameter_type(1:1) /= ' ') then
        write(output_file_unit,127,err=199) dummy_var%parameter_type(1:len_trim(dummy_var%parameter_type))
    end if
127 format('   PARAMETER_TYPE     = "',a,'"')

    ! PARENT_DATASET_ID
    if (dummy_var%parent_dataset_id(1:1) /= ' ') then
        write(output_file_unit,117,err=199) dummy_var%parent_dataset_id(1:len_trim(dummy_var%parent_dataset_id))
    end if
117 format('   PARENT_DATASET_ID  = "',a,'"')

    ! PROPERTY
    if (dummy_var%property(1:1) /= ' ') then
        write(output_file_unit,157,err=199) dummy_var%property(1:len_trim(dummy_var%property))
    end if
157 format('   PROPERTY           = "',a,'"')

    ! QUALITY
    if (dummy_var%quality(1:1) /= ' ') then
        write(output_file_unit,397,err=199) dummy_var%quality(1:len_trim(dummy_var%quality))
    end if
397 format('   QUALITY            = ',a)

    ! REPRESENTATION_1
    if (dummy_var%representation_1(1:1) /= ' ') then
        write(output_file_unit,288,err=199) dummy_var%representation_1(1:len_trim(dummy_var%representation_1))
    endif
288 format('   REPRESENTATION_1   = ',a)

    ! REPRESENTATION_2
    if (dummy_var%representation_2(1:1) /= ' ') then
        write(output_file_unit,289,err=199) dummy_var%representation_2(1:len_trim(dummy_var%representation_2))
    endif
289 format('   REPRESENTATION_2   = ',a)

    ! SCALEMAX
    if (dummy_var%scalemax(1:1) /= ' ') then
        write(output_file_unit,447,err=199) dummy_var%scalemax(1:len_trim(dummy_var%scalemax))
    end if
447 format('   SCALEMAX           = "',a,'"')

    ! SCALEMIN
    if (dummy_var%scalemin(1:1) /= ' ') then
        write(output_file_unit,437,err=199) dummy_var%scalemin(1:len_trim(dummy_var%scalemin))
    end if
437 format('   SCALEMIN           = "',a,'"')

    ! SCALETYP
    if (dummy_var%scaletyp(1:1) /= ' ') then
        write(output_file_unit,457,err=199) dummy_var%scaletyp(1:len_trim(dummy_var%scaletyp))
    end if
457 format('   SCALETYP           = "',a,'"')

    ! SI_CONVERSION
    if (dummy_var%si_conversion(1:1) /= ' ') then
        write(output_file_unit,227,err=199) dummy_var%si_conversion(1:len_trim(dummy_var%si_conversion))
    end if
227 format('   SI_CONVERSION      = "',a,'"')

    ! SIGNIFICANT_DIGITS
    if (dummy_var%significant_digits(1:1) /= ' ') then
        write(output_file_unit,377,err=199) dummy_var%significant_digits(1:len_trim(dummy_var%significant_digits))
    end if
377 format('   SIGNIFICANT_DIGITS = ',a)

    ! SIZES
    if (dummy_var%sizes(1:1) /= ' ') then
        write(output_file_unit,297,err=199) dummy_var%sizes(1:len_trim(dummy_var%sizes))
    end if
297 format('   SIZES              = ',a)

    ! TENSOR_FRAME
    if (dummy_var%tensor_frame(1:1) /= ' ') then
        write(output_file_unit,279,err=199) dummy_var%tensor_frame(1:len_trim(dummy_var%tensor_frame))
    end if
279 format('   TENSOR_FRAME       = "',a,'"')

    ! TENSOR_ORDER
    if (dummy_var%tensor_order(1:1) /= ' ') then
        write(output_file_unit,237,err=199) dummy_var%tensor_order(1:len_trim(dummy_var%tensor_order))
    end if
237 format('   TENSOR_ORDER       = ',a)

    ! UNITS
    if (dummy_var%units(1:1) /= ' ') then
        write(output_file_unit,217,err=199) dummy_var%units(1:len_trim(dummy_var%units))
    end if
217 format('   UNITS              = "',a,'"')

    ! VALUE_TYPE
    if (dummy_var%value_type(1:1) /= ' ') then
        write(output_file_unit,367,err=199) dummy_var%value_type(1:len_trim(dummy_var%value_type))
    end if
367 format('   VALUE_TYPE         = ',a)

    ! PARAMETER END
    write(output_file_unit,477,err=199) dummy_var%parameter_id(1:len_trim(dummy_var%parameter_id)), &
    dummy_var%dataset_id(1:len_trim(dummy_var%dataset_id))
477 format('END_VARIABLE = ',a,'__',a,/,'!')

    !-----!
    ! End !
    !-----!
    return

    !----------------!
    ! On error exits !
    !----------------!

199 write(*,*)
    write(*,*)
    close(UNIT=output_file_unit)
    write(*,*) '*** ERROR / W_CEF_PARAMETER / CAN''T WRITE INTO CEF OUTPUT FILE. /***'
     stop 'lib_rw_cef.o/w_cef_parameter    : *** ERROR !! Program aborted !'

    end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

    subroutine u_cef_clear_meta(dummy_cef_meta)
    
    !----------------------------------------------------------------------!
    ! Object : Fills each field of dummy_cef_meta with blanks.
    ! Author : L. Mirioni, LPP, 2005, Rev RP 2011
    ! Revis.: R. PIBERNE, LPP, 2011 Apr 06
    !
    ! Input:
    !   dummy_cef_meta
    ! Output:
    !   dummy_cef_meta
    !----------------------------------------------------------------------!

    !------------------------!
    ! Declaration Statements !
    !------------------------!

    use type_def_cef

    implicit none

    type(CEF_META) :: dummy_cef_meta

    !-----------------------!
    ! Executable Statements !
    !-----------------------!

    ! Clearing dummy_cef_meta%name
    dummy_cef_meta%name = " "

    ! Clearing dummy_cef_meta%value_type
    dummy_cef_meta%value_type = " "

    ! Clearing dummy_cef_meta%entry
    dummy_cef_meta%entry = " "


    !-----!
    ! End !
    !-----!
    return

    end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

    subroutine u_cef_clear_var(dummy_var)
    
    !----------------------------------------------------------------------!
    ! Object : Fills each field of dummy_var with blanks
    ! Author : L. Mirioni, LPP, 2005, Rev RP 2011
    ! Modification history:
    ! Revis.: R. Piberne, LPP, 2011 Apr 01
    !
    ! Input:
    !   dummy_var
    ! Output:
    !   dummy_var
    !----------------------------------------------------------------------!

    !------------------------!
    ! Declaration Statements !
    !------------------------!

    use type_def_cef

    implicit none

    type(CEF_PARAMETER) :: dummy_var
    !integer :: i,j

    !-----------------------!
    ! Executable Statements !
    !-----------------------!

    dummy_var%dataset_id = " "
    dummy_var%parameter_id = " "
    dummy_var%parent_dataset_id = " "
    dummy_var%parameter_type = " "
    dummy_var%catdesc = " "
    dummy_var%entity = " "
    dummy_var%property = " "
    dummy_var%fluctuations = " "
    dummy_var%error_plus = " "
    dummy_var%error_minus = " "
    dummy_var%compound = " "
    dummy_var%compound_def = " "
    dummy_var%units = " "
    dummy_var%si_conversion = " "
    dummy_var%tensor_order = " "
    dummy_var%coordinate_system = " "
    dummy_var%frame_origin = " "
    dummy_var%frame_velocity = " "
    dummy_var%frame = " "
    dummy_var%tensor_frame = " "
    dummy_var%representation_1 = " "
    dummy_var%representation_2 = " "
    dummy_var%sizes = " "
    dummy_var%depend_0 = " "
    dummy_var%data = " "
    dummy_var%label_1 = " "
    dummy_var%delta_plus = " "
    dummy_var%delta_minus = " "
    dummy_var%value_type = " "
    dummy_var%significant_digits = " "
    dummy_var%fillval = " "
    dummy_var%quality = " "
    dummy_var%parameter_caveats = " "
    dummy_var%fieldnam = " "
    dummy_var%lablaxis = " "
    dummy_var%scalemin = " "
    dummy_var%scalemax = " "
    dummy_var%scaletyp = " "
    dummy_var%displaytype = " "
    dummy_var%target_system = " "

    !-----!
    ! End !
    !-----!
    return

    end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

    function u_present_iso_time()
    
    !----------------------------------------------------------------------!
    ! Object : Present time in ISO format
    ! Author : L. Mirioni, LPP, 2008
    !------------------------------------------------------------------------------!

    !------------------------!
    ! Declaration Statements !
    !------------------------!

    implicit none

    character(len=27) :: u_present_iso_time ! this function
    character(len=80) :: date_format="(i4.4,'-',i2.2,'-',i2.2,'T',i2.2,':',i2.2,':',i2.2,'.000Z')"! how to write the date
    integer :: iyear,imonth,iday,ihour,imin,isec,ims

    !-----------------------!
    ! Executable statements !
    !-----------------------!

    call gdatime(iyear,imonth,iday,ihour,imin,isec,ims)
    write(u_present_iso_time,date_format) iyear,imonth,iday,ihour,imin,isec

    !--------!
    ! Return !
    !--------!
    return

    end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
!
    subroutine suppress_blank(arg_c)
    
    !----------------------------------------------------------------------!
    ! Object : supprime les blancs d'une chaine de caractere
    ! Author : R. Piberne, 2011
    !----------------------------------------------------------------------!


    implicit none

    character(len=*) :: arg_c
    integer          :: blank_pos,length_c
    !
    !     objet    : supprime les blancs d'une chaine de caractère
    !
    !

    arg_c = trim(adjustl(arg_c))

    if (arg_c == '') then

        return

    endif

    blank_pos = 1

    do while (blank_pos /= 0)

        length_c = len_trim(arg_c)
        blank_pos = index(trim(arg_c),' ')

        if (blank_pos /= 0) then

            arg_c=arg_c(1:blank_pos-1)//arg_c(blank_pos+1:length_c)

        endif

    enddo

    return
    end

!
!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
!
    subroutine suppress_tab(arg_c)
    
    !----------------------------------------------------------------------!
    ! Object : supprime les tabulations d'une chaine de caractere
    ! Author : R. Piberne, LPP, 2011
    !----------------------------------------------------------------------!


    implicit none

    character(len=*) :: arg_c
    integer          :: blank_pos,length_c
    !
    !     objet    : supprime les tabulations d'une chaine de caractère
    !
    !

    arg_c = trim(adjustl(arg_c))

    if (arg_c == '') then

        return

    endif

    blank_pos = 1

    do while (blank_pos /= 0)

        length_c = len_trim(arg_c)
        blank_pos = index(trim(arg_c),achar(9)) !9 code ascii de la tabulation

        if (blank_pos /= 0) then

            arg_c=arg_c(1:blank_pos-1)//arg_c(blank_pos+1:length_c)

        endif

    enddo

    return
    end
!
!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
!
