!XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  program vectime_to_spectro

!--------------------------------------------------------------------0--
! Laboratoire de Physique des Plasmas
!
! vectime_to_spectro : Read input RFF file VecTime type,
!                      compute spectra and create spectrogram RFF file.
!
! Input : input & output RFF file names, data are in input RFF file
! Output: Run report on stdout, and new output RFF file for the data
!
! Written by: Patrick Robert, 2012-Dec-05
!
! Modification history:
!   revision PR Feb 2018
!   Patrick Robert, 2019-Dec-21 :comments and cleaning code
!   revision PR June 2020: test value of input N_shift
!--------------------------------------------------------------------0--



! Variables declaration

  use rff_param_def
  use rff_data_def

  implicit none

  character(len=255) :: in_file_name, ou_file_name
  character(len=120) :: in_data_format, ou_data_format, ou_data_format_L
  character(len=100) :: R_ind_ext_for

  character(len= 27) :: nor_index, ext_index

! character(len=1),   dimension(3) :: sensor=(/'X','Y','Z'/)
  character(len= 27), dimension(:),allocatable ::  data_exind
  character(len=32)  :: apod
  character(len=24)  :: credate
  character(len=255) :: work
  character(len=27)  :: first_index, last_index


  integer :: in_file_unit=1, ou_file_unit=2, tmp_file_unit=3
  integer :: i, i1, i2, j,jj
  integer :: i_win, ncomp1, ncomp2, le1, block_size,ncompvec
  integer :: ierr , fill_val_int, get_pos
  integer :: k_block, i_block, o_block ! Number of blocks read and written
  integer :: block_number1   !!,if1,if2
  integer :: tm_range_min, tm_range_max
  integer :: N_Kern, N_Shift, N_win, M_Kern,N_read
  integer :: imsres,ierrtime


  logical :: pr_out=.true., test

  real (kind=4) :: R1_tmp,pui_W,Dx,Dy
  real (kind=4) :: vo_range_min, vo_range_max
  real (kind=4) :: fill_val,sigma,edge

  real (kind=8) :: spin_period, sample_rate,time_res
  real (kind=8) :: Pi,arg,D1_tmp
  real (kind=4), dimension(5) :: flt_block ! for reading block


  real (kind=4), dimension(:,:),allocatable :: data_vec, proc_vec
  real (kind=4), dimension(:),  allocatable :: RA1D_tmp  ! for working
  real (kind=4), dimension(:),  allocatable :: Weight    ! for apodisation
  real (kind=4), dimension(:),  allocatable :: DSS       ! for   sine table
  real (kind=4), dimension(:),  allocatable :: DCS       ! for cosine table

  complex (kind=4), dimension(:,:),allocatable :: TFcor  ! for TF correc.


!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!
!
!  SUMMARY:
!  ~~~~~~~
!
!   I) INPUT  PARAMETER  &  DATA  FILE
!      1) Reading input parameters
!      2) Check validity of input parameters
!
!  II) METADATA  &  CONSTANT  DATA  PROCESSING
!      1) Reading Metadata
!      2) Test on Metadata
!      3) Reading Constant Data
!      4) Modify metadata parameters
!
! III) PRE-PROC. INITIALISATIONS
!      1) Setting various parameters
!      2) Memory block allocation of all arrays
!
!
!  IV) DATA  PROCESSING
!      1) Test if there are data, if not make empty file
!      2) Load successives windows of N_Kern size
!         a) Big loop on N_win calibration window
!         c) FFT and so on...
!         d) Store in temporary file kept blocs from the calibration window
!      3) Memory block desallocation
!
!   V) CHECK  NORMAL  TERMINATION
!      1) check if bloc number read is the same as expected
!      2) check if bloc number written is the same as expected
!
!  VI) CREATE  OUTPUT  RFF FILE
!      1) Update first & last index
!      2) Opening  output_file and writing header, meta & constant data
!      3) Complete output_file by indexed data taken from temporary file
!      4) Terminate the output_file and close
!
! VII) END
!
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!

!-----------------------!
! Executable Statements !
!-----------------------!

  write(*,*) ('-',i=1,72)
  write(*,*) 'Begin vectime_to_spectro'
  write(*,*) ('-',i=1,72)
  write(*,*)

  call print_date_time('Start of program')


! --------------------------------------------------------!
! I)  I N P U T   P A R A M E T E R  &  D A T A   F I L E !
!---------------------------------------------------------!

! 1) Reading input parameters
!    ========================

! in_file_name : name of an input  vectime RFF file
! ou_file_name : name of an output vectime RFF file
!       N_Kern : Kernel size; 1024 recommanded for NBR cont. cal.
!      N_shift : for sliding window;
!        apod  : trapezium (t), Gaussian (g) or nothing (n)
!                t recommandes for classical method, with N_shift=N_Kern

  call read_input(in_file_name,ou_file_name,N_Kern, N_Shift,apod)

  write(*,*)
  write(*,*) 'Input  file : ', trim(in_file_name)
  write(*,*) 'Output file : ', trim(ou_file_name)

  write(*,*) 'Weighting function : ',trim(apod)


! 2) Check validity of input parameters
!    ==================================

! call print_date_time('Check input parameters...')

  if(apod(1:1) /= 't' .and. apod(1:1) /= 'n'  .and. apod(1:1) /= 'g') then
     write(*,*)
     write(*,'(a)') '*** error in vectime_to_spectro: apod not valid, only "t" or "n"'
     write(*,'(a)') '    Program aborted ! '
     stop 'vectime_to_spectro.exe          : *** ERROR !! Program aborted !'
  endif
!
  if(N_shift < 2 .or. N_shift > N_Kern) then
     write(*,*)
     write(*,'(a,i8,a)') '*** error in vectime_to_spectro: N_shift=',N_shift,'  must be > 2 and < N_Kern'
     write(*,'(a)')    '    common value is N_shift = N_Kern (no overlapping)'
     write(*,'(a)')    '    Program aborted ! '
     stop 'vectime_to_spectro.exe          : *** ERROR !! Program aborted !'
  endif

!  call print_date_time('End of part I')

! --------------------------------------------------------------------
! II)  M E T A    &   C O N S T A N T   D A T A   P R O C E S S I N G
! --------------------------------------------------------------------


! 1) Reading Metadata
!    ================

  write(*,*)
  write(*,*) 'READING METADATA...'
  write(*,*) '----------------'

! reading AND writing on std output manda_param & optio_param only

  call rff_R_metadata(in_file_unit,in_file_name)

  write(*,*) 'Done...'


! 2) Test on Metadata
!    ================

!    b) VecTime class only with TM data as integer

  if (manda_param%FILE_CLASS /= 'VecTime' ) then
     write(*,*)
     write(*,'(a)') ' -> ERROR vectime_to_spectro: CLASS OF INPUT FILE IS NOT VECTIME ***'
     write(*,'(a)') ' -> manda_param%FILE_CLASS='//trim(manda_param%FILE_CLASS)
     write(*,'(a)') ' -> PROGRAM ABORTED !!!'
     stop 'vectime_to_spectro.exe          : *** ERROR !! Program aborted !'
     endif

  ncompvec=manda_param%DATA_DIMENSION(1)

  if (ncompvec .LT. 3 ) then
     write(*,*)
     write(*,'(a)') ' -> ERROR vectime_to_spectro: DATA_DIMENSION(1) of INPUT FILE < 3 ***'
     write(*,'(a)') ' -> manda_param%FILE_CLASS='//trim(manda_param%FILE_CLASS)
     write(*,'(a)') ' -> PROGRAM ABORTED !!!'
     stop 'vectime_to_spectro.exe          : *** ERROR !! Program aborted !'
     endif

  ! *** autres tests eventuels a faire ***

     block_size = 1


! 3) Reading Constant Data
!    =====================

  write(*,*)
  write(*,*) 'READING CONSTANT DATA...'
  write(*,*) '----------------------'

  call rff_R_const_data(in_file_unit)


  write(*,*) 'Done...'

! Retrieve usefull constant data

  sample_rate  =const_data%SAMPLE_RATE
  vo_range_min =const_data%VOLT_RANGE_MIN
  vo_range_max =const_data%VOLT_RANGE_MAX
  tm_range_min =const_data%TM_RANGE_MIN
  tm_range_max =const_data%TM_RANGE_MAX
  spin_period  =const_data%SPIN_PERIOD
  time_res     =optio_param%TIME_RESOLUTION

! test sample rate

  if(abs(time_res -1.D0/sample_rate) .gt. 1.D-10) then
                     write(*,*)
                     write(*,*) '*** sample_rate is not coherent with time_resolution'
                     write(*,*) '    sample_rate set to 1.D0/time_resolution'
                     sample_rate= 1.D0/time_res
  endif

  write(*,*)
  write(*,*) 'sample_rate     =', sample_rate
  write(*,*) 'time_resolution =', optio_param%TIME_RESOLUTION
  write(*,*) 'vo_range_min    =', vo_range_min
  write(*,*) 'vo_range_min    =', vo_range_max
  write(*,*) 'tm_range_min    =', tm_range_min
  write(*,*) 'tm_range_max    =', tm_range_max
  write(*,*) 'spin_period     =', spin_period


! 4) Modify metadata parameters
!    ==========================

  write(*,*)
  write(*,*) 'MODIFY METADATA....'
  write(*,*) '-------------------'

! Update file name
! ----------------

  manda_param%FILE_NAME=trim(ou_file_name)

! Update history field
! --------------------

  call gdatiso(credate)
  call rff_update_history(credate,'RPC_vectime_to_spectro')


! Change metadata according to data processing, done in "data processing" section
! ---------------

! Common ouput values changes for cal_step in [1-5]
! -------------------------------------------------

! FILE_CLASS

  manda_param%FILE_CLASS='Spectrogram'

! MEASUREMENT_TYPE

  manda_param%MEASUREMENT_TYPE='B-AC Magnetic field spectra'

! DATA_TYPE

  manda_param%DATA_TYPE= 'FLT'

! DATA_DIMENSION

  ncomp1=manda_param%DATA_DIMENSION(1)

! one force 3 components
  ncomp1=3

  ncomp2=2*ncomp1 ! complex spectra
  manda_param%DATA_DIMENSION(1)= ncomp2
  manda_param%DATA_DIMENSION(2)= N_Kern/2

! DATA_FORM

  manda_param%DATA_FORM='Matrix'

! DATA_FORMAT

! manda_param%data_format must be something as '(3(E14.6,1x))'
! set default value

  in_data_format=manda_param%DATA_FORMAT
  ou_data_format  = '(5(E11.4,", "),E11.4,",")'
  ou_data_format_L= '(5(E11.4,", "),E11.4)'
  manda_param%DATA_FORMAT= '(5(E11.4,2x),E11.4)'

! DATA_UNITS

  manda_param%DATA_UNITS       = 'nT ; nT ; nT ; nT ; nT; nT' !default

! DATA_LABEL

  manda_param%DATA_LABEL       = 'BxR ; BxI ; ByR ; ByI ; BzR ; BzI '

! DATA_FILL_VALUE

  fill_val=-0.1000E+31
  manda_param%DATA_FILL_VALUE= '-0.1000E+31'

! DATA_BLOCK_NUMBER

  block_number1=manda_param%BLOCK_NUMBER

! computation of number of windows
  N_win= ((block_number1 -N_Kern)/N_Shift) +1

! computation of number of blocks to be read:
  N_read= N_Kern + (N_win-1)*N_Shift

! computation of number of block of the spectrogram
  manda_param%BLOCK_NUMBER= N_win*N_shift

! FEQUENCY_RESOLUTION

  optio_param%FREQUENCY_RESOLUTION=1.D0/(optio_param%TIME_RESOLUTION*dble(N_Kern))

! TIME_RESOLUTION

  optio_param%TIME_RESOLUTION=optio_param%TIME_RESOLUTION*dble(N_Shift)


! DATA_DESCRIPTION

  call rff_set_default_DATA_DESCRIPTION

! SPECTRA caracteristics

  const_data%SPECTRA_KERNEL_SIZE    = N_Kern
  const_data%SPECTRA_SHIFT_SIZE     = N_Shift
  const_data%SPECTRA_WEIGHTING      = apod


! Print all changed mandatory parameters
! --------------------------------------

  write(*,*)
  write(*,*)'new file_class          =', trim(manda_param%FILE_CLASS)
  write(*,*)'new measurement_type    =', trim(manda_param%MEASUREMENT_TYPE)
  write(*,*)'new data_fill_val.      =', trim(manda_param%DATA_FILL_VALUE)
  write(*,*)'new data_form           =', trim(manda_param%DATA_FORM)
  write(*,*)'new data_format         =', trim(manda_param%DATA_FORMAT)
  write(*,*)'new data_dimension      =',      manda_param%DATA_DIMENSION(1), &
                                              manda_param%DATA_DIMENSION(2)
  write(*,*)'new data_type           =', trim(manda_param%DATA_TYPE)
  write(*,*)'new data_label          =', trim(manda_param%DATA_LABEL)
  write(*,*)'new data_units          =', trim(manda_param%DATA_UNITS)
  write(*,*)'new coordinate          =', trim(manda_param%DATA_COORDINATE_SYSTEM)
  write(*,*)'new block number        =',      manda_param%BLOCK_NUMBER
  write(*,*)

! Print all changed optional parameters
! -------------------------------------

  write(*,*)
  write(*,*)'new sub_title           = ', trim(optio_param%SUB_TITLE)
  write(*,*)'new time_resolution     = ',      optio_param%TIME_RESOLUTION
  write(*,*)'new frequency_resolution= ',      optio_param%FREQUENCY_RESOLUTION


! Print all changed constant data
! -------------------------------

  write(*,*)
  write(*,*)'spectra_kernel_size   =',const_data%SPECTRA_KERNEL_SIZE
  write(*,*)'spectra_shift_size    =',const_data%SPECTRA_SHIFT_SIZE
  write(*,*)'spectra_wheighting    =',trim(const_data%SPECTRA_WEIGHTING)

  write(*,*)

! call print_date_time('End of part II')



! --------------------------------------------------------------------
! III)  P R E - P R O C .  I N I T I A L I S A T I O N S
! --------------------------------------------------------------------


! 1) Setting various parameters
!    ==========================

! Setting "read" index_extension_format used after Step 3

  call rff_format_W_to_R(manda_param%INDEX_EXTENSION_FORMAT,R_ind_ext_for)


! 2) Memory block allocation of all requested arrays
!    ===============================================

  write(*,*) 'Allocate memory begin'

  M_Kern=int(alog(float(N_Kern))/alog(2.)+0.5)

  write(*,*) '  Data Vector for window acquisition         : ', ncomp1, N_Kern
                allocate(data_vec(ncomp1,N_Kern),stat=ierr)

  write(*,*) '  Data Vector for calibration processing     : ', ncomp2, N_Kern
                allocate(proc_vec(ncomp2,N_Kern),stat=ierr)

  write(*,*) '  Float   working 1D array dimension         : ', N_Kern
                allocate(RA1D_tmp(N_Kern),stat=ierr)

  write(*,*) '  Float   Weight  1D array dimension         : ', N_Kern
                allocate(Weight(N_Kern),stat=ierr)

  write(*,*) '  Complex working 1D array dimension         : ', N_Kern
                allocate(TFcor(N_Kern,ncomp1),stat=ierr)

  write(*,*) '  Character 1D array dimension for index     : ', N_Kern
                allocate(data_index(N_Kern),stat=ierr)

  write(*,*) '  Character 1D array dimension for ext_index : ', N_Kern
                allocate(data_exind(N_Kern),stat=ierr)

  write(*,*) '  Float   sine table 1D array dimension      : ', M_Kern
                allocate(DSS(M_Kern),stat=ierr)

  write(*,*) '  Float cosine table 1D array dimension      : ', M_Kern
                allocate(DCS(M_Kern),stat=ierr)

  write(*,*) 'Allocate memory done...'


! 4) Ititialisation of tables for FFT
!    ================================

! *  Weighting fonction for waveform before FFT

   Weight(:)=1.

   IF(apod(1:1) == 't') then
!      Trapezoidal weighting function

      if( N_Kern < 64) then
         write(*,*)
         write(*,'(a)') '*** error in vectime_to_spectro: for apod, N_Kern must be > 32'
         write(*,'(a)') '    Program aborted ! '
         stop 'vectime_to_spectro.exe          : *** ERROR !! Program aborted !'
      endif

       i=N_Kern/16

       do j=1,i
          Weight(j) = float(j-1)/float(i-1)
          jj=N_Kern -j +1
          Weight(jj)= float(j-1)/float(i-1)
       enddo
   endif

   if(apod == 'g') then
!      Gaussian weighting function

       edge= 1.e-3
       sigma= float(N_Kern/2)/sqrt(-alog(edge))

!      le -0.5 est pour symétriser la gaussienne avec le sommet entre N/2 et N/2 +1

       do j=1,N_Kern
          D1_tmp=(dble(j-N_Kern/2)-0.5D0)/dble(sigma)
          Weight(j)= sngl(exp(-(D1_tmp**2D0)))
       enddo
   endif


!  total power of the weihting function

   pui_W=0.

   do j=1,N_Kern
      pui_W=pui_W +Weight(j)*Weight(j)
   enddo

   pui_W=pui_W/float(N_Kern)

   write(*,*) 'total power   of the weihting function : ',pui_W
   pui_W=sqrt(pui_W)
   write(*,*) 'rms amplitude of the weihting function : ',pui_W

! * Loadind sine and cosine table for FFT

   Pi= Dacos(-1.D0)

   do i=1,M_Kern
      Le1= (2**(M_Kern+1-i))/2
      arg=Pi/dble(Le1)
      DSS(i)=sngl(Dsin(arg))
      DCS(i)=sngl(Dcos(arg))
   enddo


!  call print_date_time('End of part III')


! --------------------------------------!
!   IV )  D A T A   P R O C E S S I N G !
! --------------------------------------!


  write(*,*)
  write(*,*) 'READING / MODIFY / WRITING DATA...'
  write(*,*) '-------------------------------'


! Achtung ! All metadata are kept in memory and last_index
! will be changed after total processing.
! Indexed_data are stored in temporary file
! Full RFF output file will be written at the end of processing.

  open(tmp_file_unit,status='scratch')

! 1) Test if there are data, if not make empty file
!    ==============================================

! Check existence of data and close files if no data
! Usefull to produce empty files

  if(manda_param%BLOCK_NUMBER==0) then
     write(*,*)
     write(*,'(2a)') '*** WARNING vectime_to_spectro: ', &
                          'BLOCK NUMBER =0, NO DATA ***'
     write(*,'(a)')  '     NEW FILE CLOSED WITH NO DATA'

     o_block=0
     first_index='None'
     last_index ='None'

     go to 99  ! for abnormal termination (this is the only one go to ! )
!    --------

  endif


! 2) load successives windows of N_Kern size for calibration
!    =======================================================

  write(*,*)
  write(*,*) 'Reading and processing all blocks, supposed equal to ', &
              manda_param%BLOCK_NUMBER
  write(*,*) '---------------------------------------'
  write(*,*)

! Positionning on START INDEXED_DATA line of input RFF file

  ierr=get_pos('START INDEXED_DATA',in_file_unit)

  IF(ierr /= 0) THEN
        write(*,*) '* WARNING ! keyword START INDEXED_DATA not found'
                ELSE
        write(*,*) 'Positionning at begin of data done'
  ENDIF

! possible blank line after the START INDEXED_DATA
  READ(in_file_unit,*) work
  IF(len_trim(work) /= 0) THEN
      BACKSPACE in_file_unit
  ENDIF

  write(*,*) 'Input file positionned on START INDEXED_DATA line'
  write(*,*)
  write(*,*) 'processing of ', N_win, ' windows '
  write(*,*) '--------------------------------------'
  write(*,*)


!    +++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! a) load  first windows
!    +++++++++++++++++++++++++++++++++++++++++++++++++++++++++


! Read N_Kern Blocks to load the first calibration window
! -------------------------------------------------------

! indice of first block to keep in the windows of N_Kern points
  i1= (N_Kern -N_Shift)/2 +1

! indice of last block to keep in the windows of N_Kern points
  i2= i1 +N_Shift -1

! time resolution in ms
  imsres= int(1000.D0/sample_rate +0.5D0)
  if(imsres < 1) then
                write(*,*) '*** time resolution is < 1 msec'
                write(*,*) '    maybe something will be wrong...'
  endif

  write(*,*) 'sample_rate=',sample_rate
  write(*,*) 'time res ms=',imsres

! set time error for data gap
  ierrtime=1

! set the count number of current point and points read/written
  k_block=0
  i_block=0
  o_block= 0
  ierrtime=0
  i_win=1
  i=0

! This first windows of N_Kern points must be without data gaps
! so, we get N_Kern continuous vectors without data gaps

  call read_N_cont_vectors(N_Kern)

! stop if EOF reached

  if (nor_index(1:3) == 'EOF' .or. nor_index(1:3) == 'END' ) then
     write(*,*)
     write(*,'(a)') ' -> WARNING : vectime_to_spectro: EOF REACHED ***'
     write(*,'(a)') '    during first window, so output empty file !!'
     o_block=0
     first_index='None'
     last_index ='None'
     go to 99
  endif


! define first_index as the one of the first block to keep
! whatever cal_step, time is the beginning of the current window

  first_index=data_index(i1)

  write(*,*) 'first index      : ',trim(first_index)


!    ++++++++++++++++++++++++++++++++++++
! b) Big loop on N_win calibration window
!    ++++++++++++++++++++++++++++++++++++


  write(*,*) '=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-'
  write(*,*) 'Big loop on ', N_win,' calibration window'
  write(*,*) '=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-'

Big_Loop:  DO i_win =1, N_win

! No shift for the first window but look for the firts continuous window

  IF (i_win > 1) THEN

! shift window data of N_Shift points

     DO i=1,N_Kern -N_Shift
        data_vec(1,i)= data_vec(1,i +N_Shift)
        data_vec(2,i)= data_vec(2,i +N_Shift)
        data_vec(3,i)= data_vec(3,i +N_Shift)

        data_index(i)= data_index(i +N_Shift)
        data_exind(i)= data_exind(i +N_Shift)
     END DO

! read N_Shift new blocks to complete the previous window

     DO i= N_Kern -N_Shift +1, N_Kern

!       reading block from input file

        flt_block(:)=fill_val_int
        call r_rff_data_block_flt(in_file_unit,nor_index,ext_index,flt_block)

!      stop if EOF reached

       if (nor_index(1:3) == 'EOF' .or. &
           nor_index(1:3) == 'END') then
          exit Big_Loop
       endif

       i_block=i_block +1

!   *  test on block continuity

       if(nor_index == ' ') stop 'vectime_to_spectro.exe          : *** ERROR nor_index vide'

       if(imsres > 0) then
                  call test_blk_cont(nor_index,imsres,ierrtime)
                      else
                  write(*,*) ' *** time res < 1 msec. => test_block_continuity impossible'
                  ierrtime=0
       endif

       if (ierrtime == 0) then

!         terminate load of calibration window

          data_vec(1,i)=flt_block(1)
          data_vec(2,i)=flt_block(2)
          data_vec(3,i)=flt_block(3)
          Dx           =flt_block(4)
          Dy           =flt_block(5)

          data_index(i)= nor_index
          data_exind(i)= ext_index

                          else
!         we have a data gap: we keep this new vector
!         and get a new series of N_Kern continuous vectors
!         without data gaps

          k_block=1

          data_vec(1,1)=flt_block(1)
          data_vec(2,1)=flt_block(2)
          data_vec(3,1)=flt_block(3)
          Dx           =flt_block(4)
          Dy           =flt_block(5)

          data_index(1)= nor_index
          data_exind(1)= ext_index

          call read_N_cont_vectors(N_Kern)

!         stop if EOF reached

          if (nor_index(1:3) == 'EOF' .or. &
              nor_index(1:3) == 'END') then
             exit Big_Loop
          endif

          exit

       endif

     END DO
   END IF

  if (i_win > 10) then
     pr_out= .false.
     else
     pr_out= .true.
  endif

  if (i_win == 11) then
     write(*,*)
     write(*,*) '/// no more details on windows calibration after i_win=10 ///'
     write(*,*)
  endif

! loading array for data processing
! ---------------------------------

  proc_vec(1:ncomp1,:)= data_vec(1:ncomp1,:)
  proc_vec(ncomp1+1:ncomp2,:)= 0.

  if(pr_out) then
             write(*,'(a, 3e15.6)') 'first data block=',proc_vec(1:ncomp1,1)
  endif


!    ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! d) Store in temporary file spectra computed from the calibration window
!    ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

     nor_index= data_index(1)
     ext_index= data_exind(1)
     o_block= o_block +1

     write(tmp_file_unit,'(a,",")') trim(nor_index)

!    compute complex spectra for each component

     do i=1,ncomp1

        test=.true.
        do j=1, N_Kern
           if(abs(proc_vec(i,j)-fill_val) < 1.e-20) then
              test= .false.
              exit
           endif
        enddo

        if(test) then
           R1_tmp=sum(proc_vec(i,:))/float(N_Kern)
           proc_vec(i,:)= proc_vec(i,:) -R1_tmp   ! substract continue
           proc_vec(i,:)= proc_vec(i,:)*Weight(:) ! weighting function
           RA1D_tmp=0.

           call fftpat_XY(proc_vec(i,:),RA1D_tmp(:),N_Kern,DSS,DCS,M_Kern, 1)

           proc_vec(i,1)       = R1_tmp     ! re_add continue on real part for f=0
           proc_vec(i+ncomp1,:)=RA1D_tmp(:) ! store imaginary part

!          normalisation pour conservation de la puissance (on a pris que 1/2 spectre)
!          rappel: pui_w est l'amplitude efficace de la fenetre
           proc_vec(i,2:N_Kern/2)=proc_vec(i,2:N_Kern/2)*1.4142137/pui_W
                 else
           write(*,*) '*** fill val at ',trim(nor_index)
           proc_vec(i,:)       = fill_val
           proc_vec(i+ncomp1,:)= fill_val
        endif

     enddo

!    add despin values if present
     proc_vec(1,1)=proc_vec(1,1)+Dx
     proc_vec(2,1)=proc_vec(2,1)+Dy


     do j=1,N_Kern/2
        if(j < N_Kern/2) then
           write(tmp_file_unit,ou_data_format) &
                (proc_vec(i,j),proc_vec(i+ncomp1,j), i=1,ncomp1 )
                            else
           write(tmp_file_unit,ou_data_format_L) &
                (proc_vec(i,j),proc_vec(i+ncomp1,j), i=1,ncomp1 )
        endif
     enddo

     last_index=nor_index


! =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
! End loop on all shifted windows
! =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

  END DO Big_Loop

  write(*,*)
  write(*,*) 'All blocks are read and written'
  write(*,*) '-------------------------------'


! 3) Memory block desallocation
!    ==========================================

  write(*,*) 'Deallocate all arays'

  write(*,*) '  Data Vector for window acquisition         : ', ncomp1, N_Kern
                deallocate(data_vec,stat=ierr)

  write(*,*) '  Data Vector for calibration processing     : ', ncomp2, N_Kern
                deallocate(proc_vec,stat=ierr)

  write(*,*) '  Float   working 1D array dimension         : ', N_Kern
                deallocate(RA1D_tmp,stat=ierr)

  write(*,*) '  Float   Weight  1D array dimension         : ', N_Kern
                deallocate(Weight,stat=ierr)

  write(*,*) '  Complex working 1D array dimension         : ', N_Kern
                deallocate(TFcor,stat=ierr)

  write(*,*) '  Character 1D array dimension for index     : ', N_Kern
                deallocate(data_index,stat=ierr)

  write(*,*) '  Character 1D array dimension for ext_index : ', N_Kern
                deallocate(data_exind,stat=ierr)

  write(*,*) '  Float   sine table 1D array dimension      : ', M_Kern
                deallocate(DSS,stat=ierr)

  write(*,*) '  Float cosine table 1D array dimension      : ', M_Kern
                deallocate(DCS,stat=ierr)

  write(*,*) 'Deallocation done'

  call print_date_time('End of part IV')



! ------------------------------------------------------!
!  V )  C H E C K   N O R M A L   T E R M I N A T I O N !
! ------------------------------------------------------!

! connection for empy or incomplete files
! ---------------------------------------

  99 continue

! Closing input file

  close(in_file_unit)

  write(*,*)
  write(*,*) 'CHECKING NORMAL TERMINATION :'
  write(*,*) '-----------------------------'


! 1) check if bloc number read is the same as expected
!    =================================================

  write(*,*)
  write(*,*) 'Number of blocks of input file: ', block_number1
  write(*,*) 'Number of blocks to be read   : ', N_read
  write(*,*) 'Number of blocks in fact read : ', i_block
  write(*,*) 'Number of blocks missing      : ', N_read-i_block

  if (i_block/=N_read) then
     write(*,*)
     if(i_block < N_read) write(*,'(a)') ' -> There was unread blocks !!!'
     if(i_block > N_read) write(*,'(a)') ' -> There was useless blocks !!!'
                       else
     write(*,*) 'OK, all requested blocks has been read'
  endif


! 2) check if bloc number written is the same as expected
!    ====================================================

  write(*,*)
  write(*,*) 'Number of blocks expected to be written : ', N_win
  write(*,*) 'Number of blocks effectively written    : ', o_block
  write(*,*) 'Number of blocks missing                : ', N_win -o_block


  if (o_block/=N_win) then
     write(*,*)
     write(*,'(a)')   ' -> There has been data gaps '
                       else
     write(*,*) 'OK, no data gap'
  endif

  call print_date_time('End of part V')


! ------------------------------------------------------!
! VI )  C R E A T E   O U T P U T   R F F  F I L E      !
! ------------------------------------------------------!

! 1) update first & last index, and block written
!    ============================================

  manda_param%BLOCK_FIRST_INDEX=trim(first_index)
  manda_param%BLOCK_LAST_INDEX=trim(last_index)
  manda_param%BLOCK_NUMBER = o_block

!    update extended index

  manda_param%INDEX_EXTENSION_LABEL ='none'
  manda_param%INDEX_EXTENSION_TYPE  ='none'
  manda_param%INDEX_EXTENSION_UNITS ='none'
  manda_param%INDEX_EXTENSION_FORMAT='none'
  manda_param%INDEX_EXTENSION_LENGTH= 0


! 2) Opening output_file and writing header, metadata and constant_data
!    ==================================================================

  write(*,*)
  write(*,*) 'OPENING RFF OUTPUT FILE & WRITING NEW METADATA....'
  write(*,*) '--------------------------------------------------'

    write(*,*)'new time resolution     =',      optio_param%TIME_RESOLUTION
  call rff_W_metadata(ou_file_unit,ou_file_name)

  write(*,*)
  write(*,*) 'First block index from Output Mand. Param. : ', &
              trim(manda_param%BLOCK_FIRST_INDEX)
  write(*,*) 'Last  block index from Output Mand. Param. : ', &
              trim(manda_param%BLOCK_LAST_INDEX)

  write(*,*) 'Done...'

! writting constant data with new one

  write(*,*)
  write(*,*) 'WRITING OLD CONSTANT DATA and NEW ONES...'
  write(*,*) '-----------------------------------------'

  call rff_W_const_data(ou_file_unit)

  write(*,*) 'Done...'


! 2) complete output_file by indexed data
!    ====================================

  write(*,*)
  write(*,*) 'WRITING INDEXED DATA...'
  write(*,*) '-----------------------'

  write(ou_file_unit,'(a)')
  write(ou_file_unit,'(a)') 'START INDEXED_DATA'

  rewind(tmp_file_unit)

  do j=1,o_block*(N_Kern/2 +1)
     read(tmp_file_unit,'(a)') work
     write(ou_file_unit,'(a)') trim(work)
  end do

  close(tmp_file_unit)

  write(*,*) 'Done...'


! 3) terminate the output_file and close
!    ===================================

  write(*,*)
  write(*,*) 'TERMINATE AND CLOSE RFF OUTPUT FILE'
  write(*,*) '-----------------------------------'

  call rff_W_tail(ou_file_unit)

  write(*,*) 'Done...'

  call print_date_time('End of part VI')


!---------------!
! VII )   E N D !
!---------------!

  write(*,*)
  write(*,'(a)')  ' -> INFO : vectime_to_spectro / NORMAL TERMINATION'

  if(manda_param%BLOCK_NUMBER==0) then
     write(*,'(3a)') '           file ', trim(ou_file_name), &
                     ' created with no data'
     write(*,*)
     write(*,'(a)')  'XXXXXXXXXX  E N D XXXXXXXXXX'
     stop 'vectime_to_spectro.exe           : ABNORMAL TERMINATION, NO DATA'
     else
     write(*,'(3a)') '           file ', trim(ou_file_name), ' created'
     write(*,*)
     write(*,'(a)')  'XXXXXXXXXX  E N D XXXXXXXXXX'
     print*, 'vectime_to_spectro.exe           : NORMAL TERMINATION'
        stop 'vectime_to_spectro.exe           : NORMAL TERMINATION'
  endif

  call print_date_time('End of part VII')

!//////////////////////////////////////////////////////////////////////!
!
! INTERNAL SUBROUTINES
!
!//////////////////////////////////////////////////////////////////////!

contains

  subroutine read_N_cont_vectors(N_Vectors)

! ------------------------------------------------------------------
! read a suit of N_Vectors without data gaps
! ------------------------------------------------------------------


  integer :: N_Vectors

  DO WHILE (k_block /= N_Vectors)

!    reading block from input file
!    ----------------------------

     flt_block(:)=fill_val_int

     call r_rff_data_block_flt(in_file_unit,nor_index,ext_index,flt_block)

!    stop if EOF reached

     if (nor_index(1:3) == 'EOF' ) then
        write(*,'(a)') ' -> WARNING : vectime_to_spectro: EOF REACHED ***'
        write(*,'(a,i8)')      '    at k_block=', k_block
        write(*,'(a,i8,a,i8)') '    for  i_win=', i_win,' i=',i
        exit
     endif

     if ( nor_index(1:3) == 'END') then
        write(*,'(3a)') ' -> WARNING : vectime_to_spectro: ',trim(nor_index)
        write(*,'(a,i8)')      '    at k_block=', k_block
        write(*,'(a,i8,a,i8)') '    for  i_win=', i_win,' i=',i
        exit
     endif

     i_block=i_block +1

! *  test on block continuity

     if(imsres > 0) then
                call test_blk_cont(nor_index,imsres,ierrtime)
                    else
               if(i_block == 1)  write(*,*) ' *** time res < 1 msec. => test_block_continuity impossible'
                  ierrtime=0
     endif     

     if(ierrtime == 0) then
!                      OK, block counter +1
                       k_block=k_block+1
                       else
                       k_block=1
     endif

!    load calibration window and index info

     data_vec(1,k_block)=flt_block(1)
     data_vec(2,k_block)=flt_block(2)
     data_vec(3,k_block)=flt_block(3)
     Dx           =flt_block(4)
     Dy           =flt_block(5)

     data_index(k_block)= nor_index
     data_exind(k_block)= ext_index

  END DO

  return

  end subroutine read_N_cont_vectors

!//////////////////////////////////////////////////////////////////////!

  subroutine r_rff_data_block_flt(in_file_unit,nor_index,ext_index,flt_block)

! ------------------------------------------------------------------
! read a data block integer of the vectime RFF file
! ------------------------------------------------------------------

  integer :: in_file_unit, ieof,ipos1,ipos2,kb
  real(kind=4), dimension(5) :: flt_block
  character(len= 27) :: nor_index, ext_index
  character(len=255) :: work

  READ(in_file_unit,'(a)', iostat=ieof) work
  IF(ieof /= 0) THEN
              print*, '-> WARNING ! eof reached on unit ',in_file_unit
              nor_index='EOF'
              RETURN
  ENDIF

  IF(work(1:4) == 'END ') THEN
              print*, '-> WARNING ! end of indexed data on unit ',in_file_unit
              print*, '    ',trim(work)
              nor_index='END '
              RETURN
  ENDIF


! find where the ISO time ends
  ipos1=INDEX(work,'Z')

  IF(ipos1 == 0) THEN
            print*, '*** Error, no Z terminator in ISO time index'
            print*, '    at i_win=',i_win,' i=',i
            print*, '    Program aborted'
            stop '*** Error, no Z terminator in ISO time index'
  ENDIF

! save ISO time
! -------------

  nor_index=work(1:ipos1)

! save extented index
! -------------------

  IF (manda_param%INDEX_EXTENSION_LENGTH == 0) THEN
     ext_index=' '
     ipos2=ipos1+2
  ELSE
     ipos2=ipos1 +1 +manda_param%INDEX_EXTENSION_LENGTH +2
     ext_index=work(ipos1+2:ipos2-2)
  ENDIF

! save vector
! -----------

  flt_block(:)=0.
! read(work(ipos2:255),in_data_format,iostat=ierr) flt_block
  read(work(ipos2:255),*,iostat=ierr) (flt_block(kb),kb=1,ncompvec)

  return

  end subroutine r_rff_data_block_flt

!//////////////////////////////////////////////////////////////////////!

  end program vectime_to_spectro

!XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX!
!
! EXTERNAL SUBROUTINES
!
!XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX!

  subroutine read_input(in_file_name,ou_file_name,N_Kern, N_Shift,apod)

!----------------------------------------------------------------------!
! Reading input parameters for vectime_to_spectro program
!----------------------------------------------------------------------!

  character*(*) in_file_name, ou_file_name, apod
  integer :: N_Kern, N_Shift

!                    *********************

! *** exemple de fichier vectime_to_spectro.in

!   WF_CLU4_NBR_TM_256R.rff
!   WF_CLU4_NBR_nT_256R.rff
!   512           # N_Kern = Kernel size
!   512           # N_shift for sliding window
!   trapezium     # trapezium (t), Gaussian (g) or nothing (n)

! *   lecture du path du fichiers N1

  write(*,*) 'nom du fichier de donnees N1 ?'
  write(*,*) '(path complet)'
  read (*,'(a)')  in_file_name
  write(*,*) trim(in_file_name)

! *   lecture du path du fichiers N2

  write(*,*) 'Name of RFF output file ?'
  write(*,*) '(path complet)'
  read (*,'(a)')  ou_file_name
  write(*,*) trim(ou_file_name)

! *   Kernel size

  write(*,*) 'Kernel size ? (must ne M**N, ex : 512)'
  read (*,*) N_Kern
  write(*,*) N_Kern

! *   Shift size and NBP kept in the kernel

  write(*,*) 'N_shift ? (ex: 4)'
  read (*,*) N_Shift
  write(*,*) N_Shift

! *   Weighting function for calibration (trapezium or nothing)

  write(*,*) 'Name of Weighting function for calibration ? '// &
             '(trapezium or nothing)'
  read (*,'(a)')  apod
  write(*,*) trim(apod)


  write(*,*) ' '
  write(*,*) 'resume des parametres d''entree:'
  write(*,*) 'fichier N1 :'
  write(*,*) trim(in_file_name)
  write(*,*) 'parametres d''entree:'
  write(*,*) 'N_Kern, N_Shift =',N_Kern, N_Shift
  write(*,*) trim(apod)

  return
  end

!XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX!
