      program give_dip_dir
      
      VGSEX=-400.0
      VGSEY=0.
      VGSEZ=0.
      ih=12
      im=0
      is=0
	  idoty=186

      do ian=1965,2025
     
      write(*,100,advance='no') ian

      call RECALC_08(ian,idoty,ih,im,is,VGSEX,VGSEY,VGSEZ)
      
      enddo


  100 format(i4)    
      stop
      end
C
c=====================================================================================
C
      SUBROUTINE RECALC_08 (IYEAR,IDAY,IHOUR,MIN,ISEC,VGSEX,VGSEY,VGSEZ)
C
C  1. PREPARES ELEMENTS OF ROTATION MATRICES FOR TRANSFORMATIONS OF VECTORS BETWEEN
C     SEVERAL COORDINATE SYSTEMS, MOST FREQUENTLY USED IN SPACE PHYSICS.
C
C  2. PREPARES COEFFICIENTS USED IN THE CALCULATION OF THE MAIN GEOMAGNETIC FIELD
C      (IGRF MODEL)
C
C  THIS SUBROUTINE SHOULD BE INVOKED BEFORE USING THE FOLLOWING SUBROUTINES:
C  IGRF_GEO_08, IGRF_GSW_08, DIP_08, GEOMAG_08, GEOGSW_08, MAGSW_08, SMGSW_08, GSWGSE_08,
c  GEIGEO_08, TRACE_08, STEP_08, RHAND_08.
C
C  THERE IS NO NEED TO REPEATEDLY INVOKE RECALC_08, IF MULTIPLE CALCULATIONS ARE MADE
C    FOR THE SAME DATE/TIME AND SOLAR WIND FLOW DIRECTION.
C
C-----INPUT PARAMETERS:
C
C     IYEAR   -  YEAR NUMBER (FOUR DIGITS)
C     IDAY  -  DAY OF YEAR (DAY 1 = JAN 1)
C     IHOUR -  HOUR OF DAY (00 TO 23)
C     MIN   -  MINUTE OF HOUR (00 TO 59)
C     ISEC  -  SECONDS OF MINUTE (00 TO 59)
C     VGSEX,VGSEY,VGSEZ - GSE (GEOCENTRIC SOLAR-ECLIPTIC) COMPONENTS OF THE OBSERVED
C                              SOLAR WIND FLOW VELOCITY (IN KM/S)
C
C  IMPORTANT: IF ONLY QUESTIONABLE INFORMATION (OR NO INFORMATION AT ALL) IS AVAILABLE
C             ON THE SOLAR WIND SPEED, OR, IF THE STANDARD GSM AND/OR SM COORDINATES ARE
C             INTENDED TO BE USED, THEN SET VGSEX=-400.0 AND VGSEY=VGSEZ=0. IN THIS CASE,
C             THE GSW COORDINATE SYSTEM BECOMES IDENTICAL TO THE STANDARD GSM.
C
C             IF ONLY SCALAR SPEED V OF THE SOLAR WIND IS KNOWN, THEN SETTING
C             VGSEX=-V, VGSEY=29.78, VGSEZ=0.0 WILL TAKE INTO ACCOUNT THE ~4 degs
C             ABERRATION OF THE MAGNETOSPHERE DUE TO EARTH'S ORBITAL MOTION
C
C             IF ALL THREE GSE COMPONENTS OF THE SOLAR WIND VELOCITY ARE AVAILABLE,
C             PLEASE NOTE THAT IN SOME SOLAR WIND DATABASES THE ABERRATION EFFECT
C             HAS ALREADY BEEN TAKEN INTO ACCOUNT BY SUBTRACTING 29.78 KM/S FROM VYGSE;
C             IN THAT CASE, THE UNABERRATED (OBSERVED) VYGSE VALUES SHOULD BE RESTORED
C             BY ADDING BACK THE 29.78 KM/S CORRECTION. WHETHER OR NOT TO DO THAT, MUST
C             BE EITHER VERIFIED WITH THE DATA ORIGINATOR OR DETERMINED BY AVERAGING
C             VGSEY OVER A SUFFICIENTLY LONG TIME INTERVAL.
C
C-----OUTPUT PARAMETERS:  NONE (ALL OUTPUT QUANTITIES ARE PLACED
C                         INTO THE COMMON BLOCKS /GEOPACK1/ AND /GEOPACK2/)
C
C    OTHER SUBROUTINES CALLED BY THIS ONE: SUN_08
C
C    AUTHOR:  N.A. TSYGANENKO
C
C    ORIGINALLY WRITTEN:    DEC.1, 1991
C
C    MOST RECENT REVISION: JANUARY 01, 2020:
C
C     The table of IGRF coefficients was extended to include those for the epoch 2020 (igrf-13)
c         (for details, see https://www.ngdc.noaa.gov/IAGA/vmod/coeffs/igrf13coeffs.txt)
C-----------------------------------------------------------------------------------   
c    EARLIER REVISIONS:
c                                
C    REVISION OF NOVEMBER 30, 2010:
C
C     The table of IGRF coefficients was extended to include those for the epoch 2010
c         (for details, see http://www.ngdc.noaa.gov/IAGA/vmod/igrf.html)
C------------------------------------------------------------------------------------
C    REVISION OF NOVEMBER 15, 2007: ADDED THE POSSIBILITY TO TAKE INTO ACCOUNT THE OBSERVED
C     DEFLECTION OF THE SOLAR WIND FLOW FROM STRICTLY RADIAL DIRECTION. TO THAT END, THREE
C     GSE COMPONENTS OF THE SOLAR WIND VELOCITY WERE ADDED TO THE INPUT PARAMETERS.
C ---------------------------------------------------------------------------------------
c    CORRECTION OF MAY 9, 2006:  INTERPOLATION OF THE COEFFICIENTS (BETWEEN
C     LABELS 50 AND 105) IS NOW MADE THROUGH THE LAST ELEMENT OF THE ARRAYS
C     G(105)  AND H(105) (PREVIOUSLY MADE ONLY THROUGH N=66, WHICH IN SOME
C     CASES CAUSED RUNTIME ERRORS)
c --------------------------------------------------------------------------------------------
C    REVISION OF MAY 3, 2005:
C     The table of IGRF coefficients was extended to include those for the epoch 2005
c       the maximal order of spherical harmonics was also increased up to 13
c         (for details, see http://www.ngdc.noaa.gov/IAGA/vmod/igrf.html)
c ---------------------------------------------------------------------------------------------
C    REVISION OF APRIL 3, 2003:
c    The code now includes preparation of the model coefficients for the subroutines
c    IGRF_08 and GEOMAG_08. This eliminates the need for the SAVE statements, used
c    in the old versions, making the codes easier and more compiler-independent.
C---------------------------------------------------------------------------------------------------
C
      COMMON /GEOPACK1/ ST0,CT0,SL0,CL0,CTCL,STCL,CTSL,STSL,SFI,CFI,
     * SPS,CPS,DS3,CGST,SGST,PSI,A11,A21,A31,A12,A22,A32,A13,A23,A33,
     * E11,E21,E31,E12,E22,E32,E13,E23,E33
C
C  THE COMMON BLOCK /GEOPACK1/ CONTAINS ELEMENTS OF THE ROTATION MATRICES AND OTHER
C   PARAMETERS RELATED TO THE COORDINATE TRANSFORMATIONS PERFORMED BY THIS PACKAGE
C
      COMMON /GEOPACK2/ G(105),H(105),REC(105)
C
C  THE COMMON BLOCK /GEOPACK2/ CONTAINS COEFFICIENTS OF THE IGRF FIELD MODEL, CALCULATED
C    FOR A GIVEN YEAR AND DAY FROM THEIR STANDARD EPOCH VALUES. THE ARRAY REC CONTAINS
C    COEFFICIENTS USED IN THE RECURSION RELATIONS FOR LEGENDRE ASSOCIATE POLYNOMIALS.
C
      DIMENSION G65(105),H65(105),G70(105),H70(105),G75(105),H75(105),
     + G80(105),H80(105),G85(105),H85(105),G90(105),H90(105),G95(105),
     + H95(105),G00(105),H00(105),G05(105),H05(105),G10(105),H10(105),
     + G15(105),H15(105),DG15(45),DH15(45),
     + G20(105),H20(105),DG20(45),DH20(45)
C
      DATA G65/0.,-30334.,-2119.,-1662.,2997.,1594.,1297.,-2038.,1292.,
     *856.,957.,804.,479.,-390.,252.,-219.,358.,254.,-31.,-157.,-62.,
     *45.,61.,8.,-228.,4.,1.,-111.,75.,-57.,4.,13.,-26.,-6.,13.,1.,13.,
     *5.,-4.,-14.,0.,8.,-1.,11.,4.,8.,10.,2.,-13.,10.,-1.,-1.,5.,1.,-2.,
     *-2.,-3.,2.,-5.,-2.,4.,4.,0.,2.,2.,0.,39*0./
      DATA H65/0.,0.,5776.,0.,-2016.,114.,0.,-404.,240.,-165.,0.,148.,
     *-269.,13.,-269.,0.,19.,128.,-126.,-97.,81.,0.,-11.,100.,68.,-32.,
     *-8.,-7.,0.,-61.,-27.,-2.,6.,26.,-23.,-12.,0.,7.,-12.,9.,-16.,4.,
     *24.,-3.,-17.,0.,-22.,15.,7.,-4.,-5.,10.,10.,-4.,1.,0.,2.,1.,2.,
     *6.,-4.,0.,-2.,3.,0.,-6.,39*0./
c
      DATA G70/0.,-30220.,-2068.,-1781.,3000.,1611.,1287.,-2091.,1278.,
     *838.,952.,800.,461.,-395.,234.,-216.,359.,262.,-42.,-160.,-56.,
     *43.,64.,15.,-212.,2.,3.,-112.,72.,-57.,1.,14.,-22.,-2.,13.,-2.,
     *14.,6.,-2.,-13.,-3.,5.,0.,11.,3.,8.,10.,2.,-12.,10.,-1.,0.,3.,
     *1.,-1.,-3.,-3.,2.,-5.,-1.,6.,4.,1.,0.,3.,-1.,39*0./
      DATA H70/0.,0.,5737.,0.,-2047.,25.,0.,-366.,251.,-196.,0.,167.,
     *-266.,26.,-279.,0.,26.,139.,-139.,-91.,83.,0.,-12.,100.,72.,-37.,
     *-6.,1.,0.,-70.,-27.,-4.,8.,23.,-23.,-11.,0.,7.,-15.,6.,-17.,6.,
     *21.,-6.,-16.,0.,-21.,16.,6.,-4.,-5.,10.,11.,-2.,1.,0.,1.,1.,3.,
     *4.,-4.,0.,-1.,3.,1.,-4.,39*0./
c
      DATA G75/0.,-30100.,-2013.,-1902.,3010.,1632.,1276.,-2144.,1260.,
     *830.,946.,791.,438.,-405.,216.,-218.,356.,264.,-59.,-159.,-49.,
     *45.,66.,28.,-198.,1.,6.,-111.,71.,-56.,1.,16.,-14.,0.,12.,-5.,
     *14.,6.,-1.,-12.,-8.,4.,0.,10.,1.,7.,10.,2.,-12.,10.,-1.,-1.,4.,
     *1.,-2.,-3.,-3.,2.,-5.,-2.,5.,4.,1.,0.,3.,-1.,39*0./
      DATA H75/0.,0.,5675.,0.,-2067.,-68.,0.,-333.,262.,-223.,0.,191.,
     *-265.,39.,-288.,0.,31.,148.,-152.,-83.,88.,0.,-13.,99.,75.,-41.,
     *-4.,11.,0.,-77.,-26.,-5.,10.,22.,-23.,-12.,0.,6.,-16.,4.,-19.,6.,
     *18.,-10.,-17.,0.,-21.,16.,7.,-4.,-5.,10.,11.,-3.,1.,0.,1.,1.,3.,
     *4.,-4.,-1.,-1.,3.,1.,-5.,39*0./
c
      DATA G80/0.,-29992.,-1956.,-1997.,3027.,1663.,1281.,-2180.,1251.,
     *833.,938.,782.,398.,-419.,199.,-218.,357.,261.,-74.,-162.,-48.,
     *48.,66.,42.,-192.,4.,14.,-108.,72.,-59.,2.,21.,-12.,1.,11.,-2.,
     *18.,6.,0.,-11.,-7.,4.,3.,6.,-1.,5.,10.,1.,-12.,9.,-3.,-1.,7.,2.,
     *-5.,-4.,-4.,2.,-5.,-2.,5.,3.,1.,2.,3.,0.,39*0./
      DATA H80/0.,0.,5604.,0.,-2129.,-200.,0.,-336.,271.,-252.,0.,212.,
     *-257.,53.,-297.,0.,46.,150.,-151.,-78.,92.,0.,-15.,93.,71.,-43.,
     *-2.,17.,0.,-82.,-27.,-5.,16.,18.,-23.,-10.,0.,7.,-18.,4.,-22.,9.,
     *16.,-13.,-15.,0.,-21.,16.,9.,-5.,-6.,9.,10.,-6.,2.,0.,1.,0.,3.,
     *6.,-4.,0.,-1.,4.,0.,-6.,39*0./
c
      DATA G85/0.,-29873.,-1905.,-2072.,3044.,1687.,1296.,-2208.,1247.,
     *829.,936.,780.,361.,-424.,170.,-214.,355.,253.,-93.,-164.,-46.,
     *53.,65.,51.,-185.,4.,16.,-102.,74.,-62.,3.,24.,-6.,4.,10.,0.,21.,
     *6.,0.,-11.,-9.,4.,4.,4.,-4.,5.,10.,1.,-12.,9.,-3.,-1.,7.,1.,-5.,
     *-4.,-4.,3.,-5.,-2.,5.,3.,1.,2.,3.,0.,39*0./
      DATA H85/0.,0.,5500.,0.,-2197.,-306.,0.,-310.,284.,-297.,0.,232.,
     *-249.,69.,-297.,0.,47.,150.,-154.,-75.,95.,0.,-16.,88.,69.,-48.,
     *-1.,21.,0.,-83.,-27.,-2.,20.,17.,-23.,-7.,0.,8.,-19.,5.,-23.,11.,
     *14.,-15.,-11.,0.,-21.,15.,9.,-6.,-6.,9.,9.,-7.,2.,0.,1.,0.,3.,
     *6.,-4.,0.,-1.,4.,0.,-6.,39*0./
c
      DATA G90/0., -29775.,  -1848.,  -2131.,   3059.,   1686.,   1314.,
     *     -2239.,   1248.,    802.,    939.,    780.,    325.,   -423.,
     *       141.,   -214.,    353.,    245.,   -109.,   -165.,    -36.,
     *        61.,     65.,     59.,   -178.,      3.,     18.,    -96.,
     *        77.,    -64.,      2.,     26.,     -1.,      5.,      9.,
     *         0.,     23.,      5.,     -1.,    -10.,    -12.,      3.,
     *         4.,      2.,     -6.,      4.,      9.,      1.,    -12.,
     *         9.,     -4.,     -2.,      7.,      1.,     -6.,     -3.,
     *        -4.,      2.,     -5.,     -2.,      4.,      3.,      1.,
     *         3.,      3.,      0.,  39*0./
C
      DATA H90/0.,      0.,   5406.,      0.,  -2279.,   -373.,      0.,
     *      -284.,    293.,   -352.,      0.,    247.,   -240.,     84.,
     *      -299.,      0.,     46.,    154.,   -153.,    -69.,     97.,
     *         0.,    -16.,     82.,     69.,    -52.,      1.,     24.,
     *         0.,    -80.,    -26.,      0.,     21.,     17.,    -23.,
     *        -4.,      0.,     10.,    -19.,      6.,    -22.,     12.,
     *        12.,    -16.,    -10.,      0.,    -20.,     15.,     11.,
     *        -7.,     -7.,      9.,      8.,     -7.,      2.,      0.,
     *         2.,      1.,      3.,      6.,     -4.,      0.,     -2.,
     *         3.,     -1.,     -6.,   39*0./
C
      DATA G95/0., -29692.,  -1784.,  -2200.,   3070.,   1681.,   1335.,
     *     -2267.,   1249.,    759.,    940.,    780.,    290.,   -418.,
     *       122.,   -214.,    352.,    235.,   -118.,   -166.,    -17.,
     *        68.,     67.,     68.,   -170.,     -1.,     19.,    -93.,
     *        77.,    -72.,      1.,     28.,      5.,      4.,      8.,
     *        -2.,     25.,      6.,     -6.,     -9.,    -14.,      9.,
     *         6.,     -5.,     -7.,      4.,      9.,      3.,    -10.,
     *         8.,     -8.,     -1.,     10.,     -2.,     -8.,     -3.,
     *        -6.,      2.,     -4.,     -1.,      4.,      2.,      2.,
     *         5.,      1.,      0.,    39*0./
C
      DATA H95/0.,      0.,   5306.,      0.,  -2366.,   -413.,      0.,
     *      -262.,    302.,   -427.,      0.,    262.,   -236.,     97.,
     *      -306.,      0.,     46.,    165.,   -143.,    -55.,    107.,
     *         0.,    -17.,     72.,     67.,    -58.,      1.,     36.,
     *         0.,    -69.,    -25.,      4.,     24.,     17.,    -24.,
     *        -6.,      0.,     11.,    -21.,      8.,    -23.,     15.,
     *        11.,    -16.,    -4.,      0.,    -20.,     15.,     12.,
     *        -6.,     -8.,      8.,      5.,     -8.,      3.,      0.,
     *         1.,      0.,      4.,      5.,     -5.,     -1.,     -2.,
     *         1.,     -2.,     -7.,    39*0./
C
      DATA G00/0.,-29619.4, -1728.2, -2267.7,  3068.4,  1670.9,  1339.6,
     *     -2288.,  1252.1,   714.5,   932.3,   786.8,    250.,   -403.,
     *      111.3,  -218.8,   351.4,   222.3,  -130.4,  -168.6,   -12.9,
     *       72.3,    68.2,    74.2,  -160.9,    -5.9,    16.9,   -90.4,
     *       79.0,   -74.0,      0.,    33.3,     9.1,     6.9,     7.3,
     *       -1.2,    24.4,     6.6,    -9.2,    -7.9,   -16.6,     9.1,
     *        7.0,    -7.9,     -7.,      5.,     9.4,      3.,   - 8.4,
     *        6.3,    -8.9,    -1.5,     9.3,    -4.3,    -8.2,    -2.6,
     *        -6.,     1.7,    -3.1,    -0.5,     3.7,      1.,      2.,
     *        4.2,     0.3,    -1.1,     2.7,    -1.7,    -1.9,     1.5,
     *       -0.1,     0.1,    -0.7,     0.7,     1.7,     0.1,     1.2,
     *        4.0,    -2.2,    -0.3,     0.2,     0.9,    -0.2,     0.9,
     *       -0.5,     0.3,    -0.3,    -0.4,    -0.1,    -0.2,    -0.4,
     *       -0.2,    -0.9,     0.3,     0.1,    -0.4,     1.3,    -0.4,
     *        0.7,    -0.4,     0.3,    -0.1,     0.4,      0.,     0.1/
C
      DATA H00/0.,      0.,  5186.1,      0., -2481.6,  -458.0,      0.,
     *     -227.6,   293.4,  -491.1,      0.,   272.6,  -231.9,   119.8,
     *     -303.8,      0.,    43.8,   171.9,  -133.1,   -39.3,   106.3,
     *         0.,   -17.4,    63.7,    65.1,   -61.2,     0.7,    43.8,
     *         0.,   -64.6,   -24.2,     6.2,     24.,    14.8,   -25.4,
     *       -5.8,     0.0,    11.9,   -21.5,     8.5,   -21.5,    15.5,
     *        8.9,   -14.9,    -2.1,     0.0,   -19.7,    13.4,    12.5,
     *       -6.2,    -8.4,     8.4,     3.8,    -8.2,     4.8,     0.0,
     *        1.7,     0.0,     4.0,     4.9,    -5.9,    -1.2,    -2.9,
     *        0.2,    -2.2,    -7.4,     0.0,     0.1,     1.3,    -0.9,
     *       -2.6,     0.9,    -0.7,    -2.8,    -0.9,    -1.2,    -1.9,
     *       -0.9,     0.0,    -0.4,     0.3,     2.5,    -2.6,     0.7,
     *        0.3,     0.0,     0.0,     0.3,    -0.9,    -0.4,     0.8,
     *        0.0,    -0.9,     0.2,     1.8,    -0.4,    -1.0,    -0.1,
     *        0.7,     0.3,     0.6,     0.3,    -0.2,    -0.5,    -0.9/
C
      DATA G05/0.,-29554.6, -1669.0, -2337.2,  3047.7,  1657.8,  1336.3,
     *    -2305.8,  1246.4,   672.5,   920.6,   798.0,   210.7,  -379.9,
     *      100.0,  -227.0,   354.4,   208.9,  -136.5,  -168.1,   -13.6,
     *       73.6,    69.6,    76.7,  -151.3,   -14.6,    14.6,   -86.4,
     *       79.9,   -74.5,    -1.7,    38.7,    12.3,     9.4,     5.4,
     *        1.9,    24.8,     7.6,   -11.7,    -6.9,   -18.1,    10.2,
     *        9.4,   -11.3,    -4.9,     5.6,     9.8,     3.6,    -6.9,
     *        5.0,   -10.8,    -1.3,     8.8,    -6.7,    -9.2,    -2.2,
     *       -6.1,     1.4,    -2.4,    -0.2,     3.1,     0.3,     2.1,
     *        3.8,    -0.2,    -2.1,     2.9,    -1.6,    -1.9,     1.4,
     *       -0.3,     0.3,    -0.8,     0.5,     1.8,     0.2,     1.0,
     *        4.0,    -2.2,    -0.3,     0.2,     0.9,    -0.4,     1.0,
     *       -0.3,     0.5,    -0.4,    -0.4,     0.1,    -0.5,    -0.1,
     *       -0.2,    -0.9,     0.3,     0.3,    -0.4,     1.2,    -0.4,
     *        0.8,    -0.3,     0.4,    -0.1,     0.4,    -0.1,    -0.2/
C
      DATA H05/0.,     0.0,  5078.0,     0.0, -2594.5,  -515.4,     0.0,
     *     -198.9,   269.7,  -524.7,     0.0,   282.1,  -225.2,   145.2,
     *     -305.4,     0.0,    42.7,   180.3,  -123.5,   -19.6,   103.9,
     *        0.0,   -20.3,    54.8,    63.6,   -63.5,     0.2,    50.9,
     *        0.0,   -61.1,   -22.6,     6.8,    25.4,    10.9,   -26.3,
     *       -4.6,     0.0,    11.2,   -20.9,     9.8,   -19.7,    16.2,
     *        7.6,   -12.8,    -0.1,     0.0,   -20.1,    12.7,    12.7,
     *       -6.7,    -8.2,     8.1,     2.9,    -7.7,     6.0,     0.0,
     *        2.2,     0.1,     4.5,     4.8,    -6.7,    -1.0,    -3.5,
     *       -0.9,    -2.3,    -7.9,     0.0,     0.3,     1.4,    -0.8,
     *       -2.3,     0.9,    -0.6,    -2.7,    -1.1,    -1.6,    -1.9,
     *       -1.4,     0.0,    -0.6,     0.2,     2.4,    -2.6,     0.6,
     *        0.4,     0.0,     0.0,     0.3,    -0.9,    -0.3,     0.9,
     *        0.0,    -0.8,     0.3,     1.7,    -0.5,    -1.1,     0.0,
     *        0.6,     0.2,     0.5,     0.4,    -0.2,    -0.6,    -0.9/
C
      DATA G10/0.00,-29496.57,-1586.42,-2396.06,3026.34,1668.17,1339.85,
     *     -2326.54,  1232.10,  633.73,  912.66, 808.97, 166.58,-356.83,
     *        89.40,  -230.87,  357.29,  200.26,-141.05,-163.17,  -8.03,
     *        72.78,    68.69,   75.92, -141.40, -22.83,  13.10, -78.09,
     *        80.44,   -75.00,   -4.55,   45.24,  14.00,  10.46,   1.64,
     *         4.92,    24.41,    8.21,  -14.50,  -5.59, -19.34,  11.61,
     *        10.85,   -14.05,   -3.54,    5.50,   9.45,   3.45,  -5.27,
     *         3.13,   -12.38,   -0.76,    8.43,  -8.42, -10.08,  -1.94,
     *        -6.24,     0.89,   -1.07,   -0.16,   2.45,  -0.33,   2.13,
     *         3.09,    -1.03,   -2.80,    3.05,  -1.48,  -2.03,   1.65,
     *        -0.51,     0.54,   -0.79,    0.37,   1.79,   0.12,   0.75,
     *         3.75,    -2.12,   -0.21,    0.30,   1.04,  -0.63,   0.95,
     *        -0.11,     0.52,   -0.39,   -0.37,   0.21,  -0.77,   0.04,
     *        -0.09,    -0.89,    0.31,    0.42,  -0.45,   1.08,  -0.31,
     *         0.78,    -0.18,    0.38,    0.02,   0.42,  -0.26,  -0.26/
C
      DATA H10/0.00,  0.00, 4944.26,    0.00,-2708.54, -575.73,    0.00,
     *      -160.40,251.75, -537.03,    0.00,  286.48, -211.03,  164.46,
     *      -309.72,  0.00,   44.58,  189.01, -118.06,   -0.01,  101.04,
     *         0.00,-20.90,   44.18,   61.54,  -66.26,    3.02,   55.40,
     *         0.00,-57.80,  -21.20,    6.54,   24.96,    7.03,  -27.61,
     *        -3.28,  0.00,   10.84,  -20.03,   11.83,  -17.41,   16.71,
     *         6.96,-10.74,    1.64,    0.00,  -20.54,   11.51,   12.75,
     *        -7.14, -7.42,    7.97,    2.14,   -6.08,    7.01,    0.00,
     *         2.73, -0.10,    4.71,    4.44,   -7.22,   -0.96,   -3.95,
     *        -1.99, -1.97,   -8.31,    0.00,    0.13,    1.67,   -0.66,
     *        -1.76,  0.85,   -0.39,   -2.51,   -1.27,   -2.11,   -1.94,
     *        -1.86,  0.00,   -0.87,    0.27,    2.13,   -2.49,    0.49,
     *         0.59,  0.00,    0.13,    0.27,   -0.86,   -0.23,    0.87,
     *         0.00, -0.87,    0.30,    1.66,   -0.59,   -1.14,   -0.07,
     *         0.54,  0.10,    0.49,    0.44,   -0.25,   -0.53,   -0.79/
C
      DATA G15/0.00,-29441.46,-1501.77,-2445.88,3012.20,1676.35,1350.33,
     *     -2352.26,  1225.85,  581.69,  907.42, 813.68, 120.49,-334.85,
     *        70.38,  -232.91,  360.14,  192.35,-140.94,-157.40,   4.30,
     *        69.55,    67.57,   72.79, -129.85, -28.93,  13.14, -70.85,
     *        81.29,   -75.99,   -6.79,   51.82,  15.07,   9.32,  -2.88,
     *         6.61,    23.98,    8.89,  -16.78,  -3.16, -20.56,  13.33,
     *        11.76,   -15.98,   -2.02,    5.33,   8.83,   3.02,  -3.22,
     *         0.67,   -13.20,   -0.10,    8.68,  -9.06, -10.54,  -2.01,
     *        -6.26,     0.17,    0.55,   -0.55,   1.70,  -0.67,   2.13,
     *         2.33,    -1.80,   -3.59,    3.00,  -1.40,  -2.30,   2.08,
     *        -0.79,     0.58,   -0.70,    0.14,   1.70,  -0.22,   0.44,
     *         3.49,    -2.09,   -0.16,    0.46,   1.23,  -0.89,   0.85,
     *         0.10,     0.54,   -0.37,   -0.43,   0.22,  -0.94,  -0.03,
     *        -0.02,    -0.92,    0.42,    0.63,  -0.42,   0.96,  -0.19,
     *         0.81,    -0.13,    0.38,    0.08,   0.46,  -0.35,  -0.36/
c
      DATA H15/0.00,   0.00,  4795.99,    0.00,-2845.41,-642.17,   0.00,
     *      -115.29, 245.04,  -538.70,    0.00,  283.54,-188.43, 180.95,
     *      -329.23,   0.00,    46.98,  196.98, -119.14,  15.98, 100.12,
     *         0.00, -20.61,    33.30,   58.74,  -66.64,   7.35,  62.41,
     *         0.00, -54.27,   -19.53,    5.59,   24.45,   3.27, -27.50,
     *        -2.32,   0.00,    10.04,  -18.26,   13.18, -14.60,  16.16,
     *         5.69,  -9.10,     2.26,    0.00,  -21.77,  10.76,  11.74,
     *        -6.74,  -6.88,     7.79,    1.04,   -3.89,   8.44,   0.00,
     *         3.28,  -0.40,     4.55,    4.40,   -7.92,  -0.61,  -4.16,
     *        -2.85,  -1.12,    -8.72,    0.00,    0.00,   2.11,  -0.60,
     *        -1.05,   0.76,    -0.20,   -2.12,   -1.44,  -2.57,  -2.01,
     *        -2.34,   0.00,    -1.08,    0.37,    1.75,  -2.19,   0.27,
     *         0.72,  -0.09,     0.29,    0.23,   -0.89,  -0.16,   0.72,
     *         0.00,  -0.88,     0.49,    1.56,   -0.50,  -1.24,  -0.10,
     *         0.42,  -0.04,     0.48,    0.48,   -0.30,  -0.43,  -0.71/
c
      DATA G20/0.0,  -29404.8, -1450.9, -2499.6, 2982.0, 1677.0, 1363.2,
     *     -2381.2,    1236.2,   525.7,   903.0,  809.5,   86.3, -309.4,
     *        48.0,    -234.3,   363.2,   187.8, -140.7, -151.2,   13.5,
     *        66.0,      65.5,    72.9,  -121.5,  -36.2,   13.5,  -64.7,
     *        80.6,     -76.7,    -8.2,    56.5,   15.8,    6.4,   -7.2,
     *         9.8,      23.7,     9.7,   -17.6,   -0.5,  -21.1,   15.3,
     *        13.7,     -16.5,    -0.3,     5.0,    8.4,    2.9,   -1.5,
     *        -1.1,     -13.2,     1.1,     8.8,   -9.3,  -11.9,   -1.9,
     *        -6.2,      -0.1,     1.7,    -0.9,    0.7,   -0.9,    1.9,
     *         1.4,      -2.4,    -3.8,     3.0,   -1.4,   -2.5,    2.3,
     *        -0.9,       0.3,    -0.7,    -0.1,    1.4,   -0.6,    0.2,
     *         3.1,      -2.0,    -0.1,     0.5,    1.3,   -1.2,    0.7,
     *         0.3,       0.5,    -0.3,    -0.5,    0.1,   -1.1,   -0.3,
     *         0.1,      -0.9,     0.5,     0.7,   -0.3,    0.8,    0.0,
     *         0.8,       0.0,     0.4,     0.1,    0.5,   -0.5,   -0.4/
c
      DATA H20/0.0,    0.0,    4652.5,     0.0, -2991.6, -734.6,    0.0,
     *       -82.1,  241.9,    -543.4,     0.0,   281.9, -158.4,  199.7,
     *      -349.7,    0.0,      47.7,   208.3,  -121.2,   32.3,   98.9,
     *         0.0,  -19.1,      25.1,    52.8,   -64.5,    8.9,   68.1,
     *         0.0,  -51.5,     -16.9,     2.2,    23.5,   -2.2,  -27.2,
     *        -1.8,    0.0,       8.4,   -15.3,    12.8,  -11.7,   14.9,
     *         3.6,   -6.9,       2.8,     0.0,   -23.4,   11.0,    9.8,
     *        -5.1,   -6.3,       7.8,     0.4,    -1.4,    9.6,    0.0,
     *         3.4,   -0.2,       3.6,     4.8,    -8.6,   -0.1,   -4.3,
     *        -3.4,   -0.1,      -8.8,     0.0,     0.0,    2.5,   -0.6,
     *        -0.4,    0.6,      -0.2,    -1.7,    -1.6,   -3.0,   -2.0,
     *        -2.6,    0.0,      -1.2,     0.5,     1.4,   -1.8,    0.1,
     *         0.8,   -0.2,       0.6,     0.2,    -0.9,    0.0,    0.5,
     *         0.0,   -0.9,       0.6,     1.4,    -0.4,   -1.3,   -0.1,
     *         0.3,   -0.1,       0.5,     0.5,    -0.4,   -0.4,   -0.6/
c
      DATA DG20/0.0,    5.7,     7.4,   -11.0,    -7.0,   -2.1,     2.2,  
     *         -5.9,    3.1,   -12.0,    -1.2,    -1.6,   -5.9,     5.2,
     *         -5.1,   -0.3,     0.5,    -0.6,     0.2,    1.3,     0.9,
     *         -0.5,   -0.3,     0.4,     1.3,    -1.4,    0.0,     0.9,
     *         -0.1,   -0.2,     0.0,     0.7,     0.1,   -0.5,    -0.8,
     *          0.8,    0.0,     0.1,    -0.1,     0.4,   -0.1,     0.4,
     *          0.3,   -0.1,     0.4/ 
c
      DATA DH20/0.0,    0.0,   -25.9,     0.0,   -30.2,  -22.4,     0.0,
     *          6.0,   -1.1,     0.5,     0.0,    -0.1,    6.5,     3.6,
     *         -5.0,    0.0,     0.0,     2.5,    -0.6,    3.0,     0.3,
     *          0.0,    0.0,    -1.6,    -1.3,     0.8,    0.0,     1.0,
     *          0.0,    0.6,     0.6,    -0.8,    -0.2,   -1.1,     0.1,
     *          0.3,    0.0,    -0.2,     0.6,    -0.2,    0.5,    -0.3,
     *         -0.4,    0.5,     0.0/
C
      IY=IYEAR
C
C  WE ARE RESTRICTED BY THE INTERVAL 1965-2025, FOR WHICH EITHER THE IGRF/DGRF COEFFICIENTS OR SECULAR VELOCITIES
c    ARE KNOWN; IF IYEAR IS OUTSIDE THIS INTERVAL, THEN THE SUBROUTINE USES THE
C      NEAREST LIMITING VALUE AND PRINTS A WARNING:
C
      IF(IY.LT.1965) THEN
       IY=1965
       WRITE (*,10) IYEAR,IY
      ENDIF

      IF(IY.GT.2025) THEN
       IY=2025
       WRITE (*,10) IYEAR,IY
      ENDIF
C
C  CALCULATE THE ARRAY REC, CONTAINING COEFFICIENTS FOR THE RECURSION RELATIONS,
C  USED IN THE IGRF SUBROUTINE FOR CALCULATING THE ASSOCIATE LEGENDRE POLYNOMIALS
C  AND THEIR DERIVATIVES:
c
      DO 20 N=1,14
         N2=2*N-1
         N2=N2*(N2-2)
         DO 20 M=1,N
            MN=N*(N-1)/2+M
20    REC(MN)=FLOAT((N-M)*(N+M-2))/FLOAT(N2)
C
      IF (IY.LT.1970) GOTO 50          !INTERPOLATE BETWEEN 1965 - 1970
      IF (IY.LT.1975) GOTO 60          !INTERPOLATE BETWEEN 1970 - 1975
      IF (IY.LT.1980) GOTO 70          !INTERPOLATE BETWEEN 1975 - 1980
      IF (IY.LT.1985) GOTO 80          !INTERPOLATE BETWEEN 1980 - 1985
      IF (IY.LT.1990) GOTO 90          !INTERPOLATE BETWEEN 1985 - 1990
      IF (IY.LT.1995) GOTO 100         !INTERPOLATE BETWEEN 1990 - 1995
      IF (IY.LT.2000) GOTO 110         !INTERPOLATE BETWEEN 1995 - 2000
      IF (IY.LT.2005) GOTO 120         !INTERPOLATE BETWEEN 2000 - 2005
      IF (IY.LT.2010) GOTO 130         !INTERPOLATE BETWEEN 2005 - 2010
      IF (IY.LT.2015) GOTO 140         !INTERPOLATE BETWEEN 2010 - 2015
      IF (IY.LT.2020) GOTO 150         !INTERPOLATE BETWEEN 2015 - 2020
C
C       EXTRAPOLATE BEYOND 2020:
C
      DT=FLOAT(IY)+FLOAT(IDAY-1)/365.25-2020.
      DO 40 N=1,105
         G(N)=G20(N)
         H(N)=H20(N)
         IF (N.GT.45) GOTO 40
         G(N)=G(N)+DG20(N)*DT
         H(N)=H(N)+DH20(N)*DT
40    CONTINUE
      GOTO 300
C
C       INTERPOLATE BETWEEEN 1965 - 1970:
C
50    F2=(FLOAT(IY)+FLOAT(IDAY-1)/365.25-1965)/5.
      F1=1.-F2
      DO 55 N=1,105
         G(N)=G65(N)*F1+G70(N)*F2
55       H(N)=H65(N)*F1+H70(N)*F2
      GOTO 300
C
C       INTERPOLATE BETWEEN 1970 - 1975:
C
60    F2=(FLOAT(IY)+FLOAT(IDAY-1)/365.25-1970)/5.
      F1=1.-F2
      DO 65 N=1,105
         G(N)=G70(N)*F1+G75(N)*F2
65       H(N)=H70(N)*F1+H75(N)*F2
      GOTO 300
C
C       INTERPOLATE BETWEEN 1975 - 1980:
C
70    F2=(FLOAT(IY)+FLOAT(IDAY-1)/365.25-1975)/5.
      F1=1.-F2
      DO 75 N=1,105
         G(N)=G75(N)*F1+G80(N)*F2
75       H(N)=H75(N)*F1+H80(N)*F2
      GOTO 300
C
C       INTERPOLATE BETWEEN 1980 - 1985:
C
80    F2=(FLOAT(IY)+FLOAT(IDAY-1)/365.25-1980)/5.
      F1=1.-F2
      DO 85 N=1,105
         G(N)=G80(N)*F1+G85(N)*F2
85       H(N)=H80(N)*F1+H85(N)*F2
      GOTO 300
C
C       INTERPOLATE BETWEEN 1985 - 1990:
C
90    F2=(FLOAT(IY)+FLOAT(IDAY-1)/365.25-1985)/5.
      F1=1.-F2
      DO 95 N=1,105
         G(N)=G85(N)*F1+G90(N)*F2
95       H(N)=H85(N)*F1+H90(N)*F2
      GOTO 300
C
C       INTERPOLATE BETWEEN 1990 - 1995:
C
100   F2=(FLOAT(IY)+FLOAT(IDAY-1)/365.25-1990)/5.
      F1=1.-F2
      DO 105 N=1,105
         G(N)=G90(N)*F1+G95(N)*F2
105      H(N)=H90(N)*F1+H95(N)*F2
      GOTO 300
C
C       INTERPOLATE BETWEEN 1995 - 2000:
C
110   F2=(FLOAT(IY)+FLOAT(IDAY-1)/365.25-1995)/5.
      F1=1.-F2
      DO 115 N=1,105   !  THE 2000 COEFFICIENTS (G00) GO THROUGH THE ORDER 13, NOT 10
         G(N)=G95(N)*F1+G00(N)*F2
115      H(N)=H95(N)*F1+H00(N)*F2
      GOTO 300
C
C       INTERPOLATE BETWEEN 2000 - 2005:
C
120   F2=(FLOAT(IY)+FLOAT(IDAY-1)/365.25-2000)/5.
      F1=1.-F2
      DO 125 N=1,105
         G(N)=G00(N)*F1+G05(N)*F2
125      H(N)=H00(N)*F1+H05(N)*F2
      GOTO 300
C
C       INTERPOLATE BETWEEN 2005 - 2010:
C
130   F2=(FLOAT(IY)+FLOAT(IDAY-1)/365.25-2005)/5.
      F1=1.-F2
      DO 135 N=1,105
         G(N)=G05(N)*F1+G10(N)*F2
135      H(N)=H05(N)*F1+H10(N)*F2
      GOTO 300
C
C       INTERPOLATE BETWEEN 2010 - 2015:
C
140   F2=(FLOAT(IY)+FLOAT(IDAY-1)/365.25-2010)/5.
      F1=1.-F2
      DO 145 N=1,105
         G(N)=G10(N)*F1+G15(N)*F2
145      H(N)=H10(N)*F1+H15(N)*F2
      GOTO 300
C
C       INTERPOLATE BETWEEN 2015 - 2020:
C
150   F2=(FLOAT(IY)+FLOAT(IDAY-1)/365.25-2015)/5.
      F1=1.-F2
      DO 155 N=1,105
         G(N)=G15(N)*F1+G20(N)*F2
155      H(N)=H15(N)*F1+H20(N)*F2
      GOTO 300
c
C   COEFFICIENTS FOR A GIVEN YEAR HAVE BEEN CALCULATED; NOW MULTIPLY
C   THEM BY SCHMIDT NORMALIZATION FACTORS:
C
300   S=1.
      DO 250 N=2,14
         MN=N*(N-1)/2+1
         S=S*FLOAT(2*N-3)/FLOAT(N-1)
         G(MN)=G(MN)*S
         H(MN)=H(MN)*S
         P=S
         DO 250 M=2,N
            AA=1.
            IF (M.EQ.2) AA=2.
            P=P*SQRT(AA*FLOAT(N-M+1)/FLOAT(N+M-2))
            MNN=MN+M-1
            G(MNN)=G(MNN)*P
250         H(MNN)=H(MNN)*P

           G_10=-G(2)
           G_11= G(3)
           H_11= H(3)
C
C  NOW CALCULATE GEO COMPONENTS OF THE UNIT VECTOR EzMAG, PARALLEL TO GEODIPOLE AXIS:
C   SIN(TETA0)*COS(LAMBDA0), SIN(TETA0)*SIN(LAMBDA0), AND COS(TETA0)
C         ST0 * CL0                ST0 * SL0                CT0
C
      SQ=G_11**2+H_11**2
      SQQ=SQRT(SQ)
      SQR=SQRT(G_10**2+SQ)
      SL0=-H_11/SQQ
      CL0=-G_11/SQQ
      ST0=SQQ/SQR
      CT0=G_10/SQR
      STCL=ST0*CL0
      STSL=ST0*SL0
      CTSL=CT0*SL0
      CTCL=CT0*CL0
c ///////////////////////////////////////////////////////////////
      DIPMOM=SQRT(G(2)**2+G(3)**2+H(3)**2)
      call t_car_to_sph(STCL,STSL,CT0,r,teta,phi)
      rlat=90.-teta*180./3.14159
      rlong=phi*180./3.14159
      print 101,STCL,STSL,CT0, rlat,rlong,DIPMOM
  101 format(1x,3F9.5,f7.2,f8.2,e15.6)
c ///////////////////////////////////////////////////////////////
C
C  NOW CALCULATE GEI COMPONENTS (S1,S2,S3) OF THE UNIT VECTOR S = EX_GSE
C    POINTING FROM THE EARTH'S CENTER TO SUN
C
      CALL SUN_08 (IY,IDAY,IHOUR,MIN,ISEC,GST,SLONG,SRASN,SDEC)
C
      S1=COS(SRASN)*COS(SDEC)
      S2=SIN(SRASN)*COS(SDEC)
      S3=SIN(SDEC)
C
C  NOW CALCULATE GEI COMPONENTS (DZ1,DZ2,DZ3) OF THE UNIT VECTOR EZGSE
C  POINTING NORTHWARD AND ORTHOGONAL TO THE ECLIPTIC PLANE, AS
C  (0,-SIN(OBLIQ),COS(OBLIQ)). FOR THE EPOCH 1978, OBLIQ = 23.44214 DEGS.
C  HERE WE USE A MORE ACCURATE TIME-DEPENDENT VALUE, DETERMINED AS:
C
      DJ=FLOAT(365*(IY-1900)+(IY-1901)/4 +IDAY)
     * -0.5+FLOAT(IHOUR*3600+MIN*60+ISEC)/86400.
      T=DJ/36525.
      OBLIQ=(23.45229-0.0130125*T)/57.2957795
      DZ1=0.
      DZ2=-SIN(OBLIQ)
      DZ3=COS(OBLIQ)
C
C  NOW OBTAIN GEI COMPONENTS OF THE UNIT VECTOR EYGSE=(DY1,DY2,DY3),
C  COMPLETING THE RIGHT-HANDED SYSTEM. THEY CAN BE FOUND FROM THE VECTOR
C  PRODUCT EZGSE x EXGSE = (DZ1,DZ2,DZ3) x (S1,S2,S3):
C
      DY1=DZ2*S3-DZ3*S2
      DY2=DZ3*S1-DZ1*S3
      DY3=DZ1*S2-DZ2*S1
C
C  NOW CALCULATE GEI COMPONENTS OF THE UNIT VECTOR X = EXGSW, DIRECTED ANTIPARALLEL
C  TO THE OBSERVED SOLAR WIND FLOW. FIRST, CALCULATE ITS COMPONENTS IN GSE:
C
      V=SQRT(VGSEX**2+VGSEY**2+VGSEZ**2)
      DX1=-VGSEX/V
      DX2=-VGSEY/V
      DX3=-VGSEZ/V
C
C  THEN IN GEI:
C
      X1=DX1*S1+DX2*DY1+DX3*DZ1
      X2=DX1*S2+DX2*DY2+DX3*DZ2
      X3=DX1*S3+DX2*DY3+DX3*DZ3
C
C  NOW CALCULATE GEI COMPONENTS (DIP1,DIP2,DIP3) OF THE UNIT VECTOR DIP = EZ_SM = EZ_MAG,
C   ALIGNED WITH THE GEODIPOLE AND POINTING NORTHWARD FROM ECLIPTIC PLANE:
C
      CGST=COS(GST)
      SGST=SIN(GST)
C
      DIP1=STCL*CGST-STSL*SGST
      DIP2=STCL*SGST+STSL*CGST
      DIP3=CT0
C
C  THIS ALLOWS US TO CALCULATE GEI COMPONENTS OF THE UNIT VECTOR Y = EYGSW
C   BY TAKING THE VECTOR PRODUCT DIP x X AND NORMALIZING IT TO UNIT LENGTH:
C
      Y1=DIP2*X3-DIP3*X2
      Y2=DIP3*X1-DIP1*X3
      Y3=DIP1*X2-DIP2*X1
      Y=SQRT(Y1*Y1+Y2*Y2+Y3*Y3)
      Y1=Y1/Y
      Y2=Y2/Y
      Y3=Y3/Y
C
C   AND GEI COMPONENTS OF THE UNIT VECTOR Z = EZGSW = EXGSW x EYGSW = X x Y:
C
      Z1=X2*Y3-X3*Y2
      Z2=X3*Y1-X1*Y3
      Z3=X1*Y2-X2*Y1
C
C   ELEMENTS OF THE MATRIX GSE TO GSW ARE THE SCALAR PRODUCTS:
C
C  E11=(EXGSE,EXGSW)  E12=(EXGSE,EYGSW)  E13=(EXGSE,EZGSW)
C  E21=(EYGSE,EXGSW)  E22=(EYGSE,EYGSW)  E23=(EYGSE,EZGSW)
C  E31=(EZGSE,EXGSW)  E32=(EZGSE,EYGSW)  E33=(EZGSE,EZGSW)
C
      E11= S1*X1 +S2*X2 +S3*X3
      E12= S1*Y1 +S2*Y2 +S3*Y3
      E13= S1*Z1 +S2*Z2 +S3*Z3
      E21=DY1*X1+DY2*X2+DY3*X3
      E22=DY1*Y1+DY2*Y2+DY3*Y3
      E23=DY1*Z1+DY2*Z2+DY3*Z3
      E31=DZ1*X1+DZ2*X2+DZ3*X3
      E32=DZ1*Y1+DZ2*Y2+DZ3*Y3
      E33=DZ1*Z1+DZ2*Z2+DZ3*Z3
C
C   GEODIPOLE TILT ANGLE IN THE GSW SYSTEM: PSI=ARCSIN(DIP,EXGSW)
C
      SPS=DIP1*X1+DIP2*X2+DIP3*X3
      CPS=SQRT(1.-SPS**2)
      PSI=ASIN(SPS)
!      print*,'      GSM: tilt=',PSI*180./3.1415927
C
C   ELEMENTS OF THE MATRIX GEO TO GSW ARE THE SCALAR PRODUCTS:
C
C   A11=(EXGEO,EXGSW), A12=(EYGEO,EXGSW), A13=(EZGEO,EXGSW),
C   A21=(EXGEO,EYGSW), A22=(EYGEO,EYGSW), A23=(EZGEO,EYGSW),
C   A31=(EXGEO,EZGSW), A32=(EYGEO,EZGSW), A33=(EZGEO,EZGSW),
C
C   ALL THE UNIT VECTORS IN BRACKETS ARE ALREADY DEFINED IN GEI:
C
C  EXGEO=(CGST,SGST,0), EYGEO=(-SGST,CGST,0), EZGEO=(0,0,1)
C  EXGSW=(X1,X2,X3),  EYGSW=(Y1,Y2,Y3),   EZGSW=(Z1,Z2,Z3)
C                                                           AND  THEREFORE:
C
      A11=X1*CGST+X2*SGST
      A12=-X1*SGST+X2*CGST
      A13=X3
      A21=Y1*CGST+Y2*SGST
      A22=-Y1*SGST+Y2*CGST
      A23=Y3
      A31=Z1*CGST+Z2*SGST
      A32=-Z1*SGST+Z2*CGST
      A33=Z3
C
C  NOW CALCULATE ELEMENTS OF THE MATRIX MAG TO SM (ONE ROTATION ABOUT THE GEODIPOLE AXIS);
C   THEY ARE FOUND AS THE SCALAR PRODUCTS: CFI=GM22=(EYSM,EYMAG)=(EYGSW,EYMAG),
C                                          SFI=GM23=(EYSM,EXMAG)=(EYGSW,EXMAG),
C    DERIVED AS FOLLOWS:
C
C IN GEO, THE VECTORS EXMAG AND EYMAG HAVE THE COMPONENTS (CT0*CL0,CT0*SL0,-ST0)
C  AND (-SL0,CL0,0), RESPECTIVELY.    HENCE, IN GEI THEIR COMPONENTS ARE:
C  EXMAG:    CT0*CL0*COS(GST)-CT0*SL0*SIN(GST)
C            CT0*CL0*SIN(GST)+CT0*SL0*COS(GST)
C            -ST0
C  EYMAG:    -SL0*COS(GST)-CL0*SIN(GST)
C            -SL0*SIN(GST)+CL0*COS(GST)
C             0
C  NOW, NOTE THAT GEI COMPONENTS OF EYSM=EYGSW WERE FOUND ABOVE AS Y1, Y2, AND Y3,
C  AND WE ONLY HAVE TO COMBINE THESE QUANTITIES INTO SCALAR PRODUCTS:
C
      EXMAGX=CT0*(CL0*CGST-SL0*SGST)
      EXMAGY=CT0*(CL0*SGST+SL0*CGST)
      EXMAGZ=-ST0
      EYMAGX=-(SL0*CGST+CL0*SGST)
      EYMAGY=-(SL0*SGST-CL0*CGST)
      CFI=Y1*EYMAGX+Y2*EYMAGY
      SFI=Y1*EXMAGX+Y2*EXMAGY+Y3*EXMAGZ
C
 10   FORMAT(//1X,
     *'**** RECALC_08 WARNS: YEAR IS OUT OF INTERVAL 1965-2025: IYEAR=',
     *I4,/,6X,'CALCULATIONS WILL BE DONE FOR IYEAR=',I4,/)
      RETURN
      END
c
c==================================================================================
C*******************************************************************
c
      SUBROUTINE SUN_08 (IYEAR,IDAY,IHOUR,MIN,ISEC,GST,SLONG,SRASN,SDEC)
C
C  CALCULATES FOUR QUANTITIES NECESSARY FOR COORDINATE TRANSFORMATIONS
C  WHICH DEPEND ON SUN POSITION (AND, HENCE, ON UNIVERSAL TIME AND SEASON)
C
C-------  INPUT PARAMETERS:
C  IYR,IDAY,IHOUR,MIN,ISEC -  YEAR, DAY, AND UNIVERSAL TIME IN HOURS, MINUTES,
C    AND SECONDS  (IDAY=1 CORRESPONDS TO JANUARY 1).
C
C-------  OUTPUT PARAMETERS:
C  GST - GREENWICH MEAN SIDEREAL TIME, SLONG - LONGITUDE ALONG ECLIPTIC
C  SRASN - RIGHT ASCENSION,  SDEC - DECLINATION  OF THE SUN (RADIANS)
C  ORIGINAL VERSION OF THIS SUBROUTINE HAS BEEN COMPILED FROM:
C  RUSSELL, C.T., COSMIC ELECTRODYNAMICS, 1971, V.2, PP.184-196.
C
C  LAST MODIFICATION:  MARCH 31, 2003 (ONLY SOME NOTATION CHANGES)
C
C     ORIGINAL VERSION WRITTEN BY:    Gilbert D. Mead
C
      DOUBLE PRECISION DJ,FDAY
      DATA RAD/57.295779513/
C
      IF(IYEAR.LT.1901.OR.IYEAR.GT.2099) RETURN
      FDAY=DFLOAT(IHOUR*3600+MIN*60+ISEC)/86400.D0
      DJ=365*(IYEAR-1900)+(IYEAR-1901)/4+IDAY-0.5D0+FDAY
      T=DJ/36525.
      VL=DMOD(279.696678+0.9856473354*DJ,360.D0)
      GST=DMOD(279.690983+.9856473354*DJ+360.*FDAY+180.,360.D0)/RAD
      G=DMOD(358.475845+0.985600267*DJ,360.D0)/RAD
      SLONG=(VL+(1.91946-0.004789*T)*SIN(G)+0.020094*SIN(2.*G))/RAD
      IF(SLONG.GT.6.2831853) SLONG=SLONG-6.2831853
      IF (SLONG.LT.0.) SLONG=SLONG+6.2831853
      OBLIQ=(23.45229-0.0130125*T)/RAD
      SOB=SIN(OBLIQ)
      SLP=SLONG-9.924E-5
C
C   THE LAST CONSTANT IS A CORRECTION FOR THE ANGULAR ABERRATION DUE TO
C   EARTH'S ORBITAL MOTION
C
      SIND=SOB*SIN(SLP)
      COSD=SQRT(1.-SIND**2)
      SC=SIND/COSD
      SDEC=ATAN(SC)
      SRASN=3.141592654-ATAN2(COS(OBLIQ)/SOB*SC,-COS(SLP)/COSD)
      RETURN
      END
C
C================================================================================
c
!     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
!                                                                       
      subroutine t_car_to_sph(x,y,z,r,teta,phi) 
!                                                                       
! ----------------------------------------------------------------------
! *   Class  : transform modules of Rocotlib Software                   
! *   Object : transforms_car_to_sph: CAR -> SPH  system                
! *   Author : P. Robert, CRPE, 1992                                    
! *   Comment: none                                                     
!                                                                       
! *   Input  :   x,y,z        cartesian coordinates                     
! *   Output :   r,teta,phi   spherical coordinates (angles in radians) 
! ----------------------------------------------------------------------
!                                                                       
!                                                                       
      teta=0. 
      phi=0. 
      sq=x**2+y**2 
      r=sqrt(sq+z**2) 
      pi=acos(-1.)
      pisd=pi/180.
!                                                                       
      if(r.lt.1.e-30) return 
!                                                                       
! *** en dessous de 1/10000 degres, on considere que teta =0 ou 180     
!     et phi indefini, mis a zero                                       
!                                                                       
      if(sq/r.gt.1.7e-6) then 
                   phi=atan2(y,x) 
                 if(phi.lt.-179.999*pisd.and.phi.gt.-180.*pisd) phi=pi
                 if(phi.lt. 0.0001*pisd.and.phi.gt.-0.0001*pisd) phi=0.
                   teta=acos(z/r) 
                   return 
                   endif 
!                                                                       
      if (z.lt.0.) then 
                   teta=acos(-1.) 
                   else 
                   teta=0. 
                   endif 
!                                                                       
      return 
      END                                           
!                                                                       
!     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
