!
! XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
! |                                                                    |
! |                      deconvolib                                    |
! |                                                                    |
! |        bibliotheque de deconvolution de formes d'ondes UBF         |
! |                                                                    |
! |  Permet le despinage et la calibration en nT de formes d'ondes     |
! |  d'ultra basse frequence (typiquement 0.1-10 Hz)                   |
! |  enregistrees dans  le repere tournant des antennes du satellite,  |
! |  et presentant  une forte composante a la frequence de spin,       |
! |  porteuse des ULF. Le spin est de l'ordre de 6 rps (0.17 Hz)       |
! |                                                                    |
! |  Bibliotheque inspiree de la geoscalclib, developpe pour  les      |
! |  satellite GEOS-1 et GEOS-2, de 1977 a 1984, et reprise pour le    |
! |  traitement des donnees GEOS au CDPP, entre 1996 et 2000.          |
! |                                                                    |
! | On suppose l'existence de la fonction complexe decrivant le gain   |
! | des antennes sous la forme gainant(f,ix,isat,fe) exprimee en V/nT  |
! | et dependant de la frequence, de la composante, du satellite et de |
! | la frequence d'echantillonnage.                                    |
! |                                                                    |
! | Extraction des modules utiles et simplifications, application a    |
! | CLUSTER ou pour d'autres donnees equivalentes:                     |
! |                                                                    |
! |         P. Robert, CNRS/CETP, Septembre 2000                       |
! |         Revision P.Ro Avril   2008 pour les RPC                    |
! |         Revision P.Ro Fevrier 2009 - menage                        |
! |         Revision P.Ro May     2011 - optimisation pour CLUSTER/CAA |
! |         Refonte  P.Ro Dec     2012 - grand menage et optimisation  |
! |                                                                    |
! XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
!
!
! XXXX7XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX+XX
!
      subroutine casinus(sig,n,fe,fs,as,phi,com)
!
!     ------------------------------------------------------------------
! *   Objet  : calcul d'une sinusoide contenue dans un signal reel
! *   Classe : mathematiques & traitement du signal
! *   Auteur : P. Robert, CRPE, 1977-1984
!     ------------------------------------------------------------------
!
      dimension sig(n)
      character*(*) com
!
!                    *********************
!
!     Le signal est compose d'un signal utile
!     superpose a une forte sinusoide pure de frequence connue.
!     Arguments d'entree:
!         sig:  tableau des valeurs du signal
!               contenant la sinusoide pure
!           n:  dimension du tableau sig
!          fe:  frequence d]echantillonnage
!          fs:  frequence de la sinusoide a soustraire
!         com:  commentaire a imprimer
!
!     Arguments de sortie:
!          as:  amplitude de la sinusoide calculee
!         phi:  phase en degres
!
!
! *** initialisations; on tien compte du fait que la frequence fs
!                      n'est pas un multiple de df=fe/nbp
!
      pi=3.141592654
      s=sin(2.*pi*fs/fe)
      c=cos(2.*pi*fs/fe)
!
      s1=0.
      c1=1.
      xs=0.
      xc=0.
!
! *** calcul du nombre de points d]integration pour le calcul
!     harmonique sur un nombre entier de periode de spin
!
      nbsp=int(float(n)*fs/fe)
!
      if(nbsp.lt.1) then
                    print 200, 'casinus: number of spin period LT 1'
                    print 200, 'computation of spin signal impossible'
                    as=0.
                    phi=0.
                    return
                    endif
!
      nbi=int((float(nbsp)*fe/fs)+0.5)
!
! *** calcul de l'amplitude et de la phase
!
      do 1 i=1,nbi
         sc=s1*c+c1*s
         c1=c1*c-s1*s
         s1=sc
         xs=xs+s1*sig(i)
         xc=xc+c1*sig(i)
    1 continue
      as=2.*sqrt(xs*xs+xc*xc)/nbi
      phi=atan2(xc,xs)
      phi=phi*180./pi

      print 100, com,as,phi
!
  100 format(1x,a,'  amplitude=',f10.6,'   phase=',f9.3,' deg.')
  200 format(1x,a)
!
      return
      end
!
! XXXX7XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX+XX
!
      subroutine casinus_D(sig,n,fe,fs,as,phi,com)
!
!     ------------------------------------------------------------------
! *   Objet  : calcul d'une sinusoide contenue dans un signal reel
! *   Classe : mathematiques & traitement du signal
! *   Auteur : P. Robert, CRPE, 1977-1984
!     ------------------------------------------------------------------
!
      dimension sig(n)
      character*(*) com
! jah, probleme de precision Solaris / Linux avec ifort (passage en double precision)
      double precision c, pi, xc, s1, c1, s, sc, xs
!
!                    *********************
!
!     Le signal est compose d'un signal utile
!     superpose a une forte sinusoide pure de frequence connue.
!     Arguments d'entree:
!         sig:  tableau des valeurs du signal
!               contenant la sinusoide pure
!           n:  dimension du tableau sig
!          fe:  frequence d]echantillonnage
!          fs:  frequence de la sinusoide a soustraire
!         com:  commentaire a imprimer
!
!     Arguments de sortie:
!          as:  amplitude de la sinusoide calculee
!         phi:  phase en degres
!
!
! *** initialisations; on tien compte du fait que la frequence fs
!                      n'est pas un multiple de df=fe/nbp
!
      pi=0.314159265358979d1
      s=dsin(2.D0*pi*dble(fs)/dble(fe))
      c=dcos(2.D0*pi*dble(fs)/dble(fe))
!
      s1=0.d0
      c1=1.d0
      xs=0.d0
      xc=0.d0
!
! *** calcul du nombre de points d]integration pour le calcul
!     harmonique sur un nombre entier de periode de spin
!
      nbsp=idint(dble(n)*dble(fs)/dble(fe))
!
      if(nbsp.lt.1) then
                    print 200, 'casinus: number of spin period LT 1'
                    print 200, 'computation of spin signal impossible'
                    as=0.
                    phi=0.
                    return
                    endif
!
                    nbi=idint((dble(nbsp)*dble(fe)/dble(fs))+0.5d0)
!
! *** calcul de l'amplitude et de la phase
!
      do 1 i=1,nbi
         sc=s1*c+c1*s
         c1=c1*c-s1*s
         s1=sc
         xs=xs+s1*dble(sig(i))
         xc=xc+c1*dble(sig(i))
    1 continue
      as=sngl(2.d0*dsqrt(xs*xs+xc*xc)/dble(nbi))
      phi=sngl(datan2(xc,xs))
      phi=sngl(dble(phi)*180.d0/pi)

!
!
      print 100, com,as,phi
!
  100 format(1x,a,'  amplitude=',f10.6,'   phase=',f9.3,' deg.')
  200 format(1x,a)
!
      return
      end
!
! XXXX7XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX+XX
!
      subroutine desinus(xio,n,fe,fs,amp,pha)
!
!     ------------------------------------------------------------------
! *   Objet  : despinage d'une forme d'onde xio (algoritme demod)
! *   Classe : depouillement specifique GEOS/UBF
! *   Auteur : P. Robert, CRPE, 1977-1984
!     ------------------------------------------------------------------
!
      dimension xio(n)
!
!                    *********************
!
! *** meme algorithme que casinus mais retire la sinusoide
!
!     Le signal est compose d'un signal utile
!     superpose a une forte sinusoide pure de frequence connue.
!     Arguments d'entree:
!         sig:  tableau des valeurs du signal
!               contenant la sinusoide pure
!           n:  dimension du tableau sig
!          fe:  frequence d]echantillonnage
!          fs:  frequence de la sinusoide a soustraire
!         com:  commentaire a imprimer
!
!     Arguments de sortie:
!          as:  amplitude de la sinusoide calculee
!         phi:  phase en degres
!
!
! *** initialisations; on tien compte du fait que la frequence fs
!                      n'est pas un multiple de df=fe/nbp
!
      pi=3.1415926
      ss=sin(2.*pi*fs/fe)
      cc=cos(2.*pi*fs/fe)
!
! *** calcul du nombre de points d integration pour le calcul
!     harmonique (sur un nombre entier de periode de spin)
!
      nbsp=int(float(n)*fs/fe)
      nbi=int((float(nbsp)*fe/fs)+0.5)
      if (nbsp.eq.0) goto 100
      if (nbi.gt.n) goto 100
!
! *** calcul harmonique de la sinusoide de spin
!
      s1=0.
      c1=1.
      zs=0.
      zc=0.
!
      do 10 i=1,nbi
         sc=s1*cc+c1*ss
         c1=c1*cc-s1*ss
         s1=sc
         zs=zs+s1*xio(i)
         zc=zc+c1*xio(i)
   10 continue
!
      amp=2.*sqrt(zs*zs+zc*zc)/float(nbi)
      pha=atan2(zc,zs)
!
      sz1=sin(pha)
      cz1=cos(pha)

!
      do 2 i=1,n
      scz1=cc*sz1+ss*cz1
      cz1 =cc*cz1-ss*sz1
      sz1 =scz1
!
! *** calcul du signal sans la composante de spin
!
      xio(i)=xio(i)-amp*sz1
    2 continue
!
      pha=pha*180./pi
      return
!
  100 continue
!
      amp=0.
      pha=0.
!
      print 3, fs,fe,n,nbsp,nbi

!
   3  format(' ***desinus: despinage des formes d''onde impossible...', &
           /,10x,'fs, fe=     ',2f10.3, &
           /,10x,'n, nbsp,nbi=',3(i6,4x))
!
      return
      end
!
! XXXX7XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX+XX
!
      subroutine desinus_D(xio,n,fe,fs,amp,pha)
!
!     ------------------------------------------------------------------
! *   Objet  : despinage d'une forme d'onde xio (algoritme demod)
! *   Classe : depouillement specifique GEOS/UBF
! *   Auteur : P. Robert, CRPE, 1977-1984
!     ------------------------------------------------------------------
!
      dimension xio(n)
! jah, probleme de precision Solaris / Linux avec ifort (passage en double precision)
      double precision pi, ss, cc, s1, c1, zs, zc, sc, dbleAmp, dblePha, &
           sz1, cz1, scz1
!lm      complex ji
!
!                    *********************
!
! *** meme algorithme que casinus mais retire la sinusoide
!
!     Le signal est compose d'un signal utile
!     superpose a une forte sinusoide pure de frequence connue.
!     Arguments d'entree:
!         sig:  tableau des valeurs du signal
!               contenant la sinusoide pure
!           n:  dimension du tableau sig
!          fe:  frequence d]echantillonnage
!          fs:  frequence de la sinusoide a soustraire
!         com:  commentaire a imprimer
!
!     Arguments de sortie:
!          as:  amplitude de la sinusoide calculee
!         phi:  phase en degres
!
!
! *** initialisations; on tien compte du fait que la frequence fs
!                      n'est pas un multiple de df=fe/nbp
!
      pi=0.314159265358979d1
      ss=dsin(2.d0*pi*dble(fs)/dble(fe))
      cc=dcos(2.d0*pi*dble(fs)/dble(fe))
!
! *** calcul du nombre de points d integration pour le calcul
!     harmonique (sur un nombre entier de periode de spin)
!
      nbsp=idint(dble(n)*dble(fs)/dble(fe))
      nbi=idint((dble(nbsp)*dble(fe)/dble(fs))+0.5d0)
      if (nbsp.eq.0) goto 100
      if (nbi.gt.n) goto 100
!
! *** calcul harmonique de la sinusoide de spin
!
      s1=0.d0
      c1=1.d0
      zs=0.d0
      zc=0.d0
!
      do 10 i=1,nbi
         sc=s1*cc+c1*ss
         c1=c1*cc-s1*ss
         s1=sc
         zs=zs+s1*dble(xio(i))
         zc=zc+c1*dble(xio(i))
   10 continue
!
      dbleAmp=2.d0*dsqrt(zs*zs+zc*zc)/dble(nbi)
      dblePha=datan2(zc,zs)
!
      sz1=dsin(dblePha)
      cz1=dcos(dblePha)

!
      do 2 i=1,n
      scz1=cc*sz1+ss*cz1
      cz1 =cc*cz1-ss*sz1
      sz1 =scz1
!
! *** calcul du signal sans la composante de spin
!
      xio(i)=sngl(dble(xio(i))-dbleAmp*sz1)
    2 continue
!
      pha=sngl(dblePha*180.d0/pi)
      amp=sngl(dbleAmp)
      return
!
  100 continue
!
      amp=0.
      pha=0.
!
      print 3, fs,fe,n,nbsp,nbi

!
   3  format(' ***desinus: despinage des formes d''onde impossible...', &
           /,10x,'fs, fe=     ',2f10.3, &
           /,10x,'n, nbsp,nbi=',3(i6,4x))
!
      return
      end
!
! XXXX7XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX+XX
!
      subroutine deconvo_R(xio,tra,TFcor,nbp,ix,DSS,DCS,M_Kern)

      real   xio(nbp),tra(nbp)
      real DSS(M_Kern),DCS(M_Kern)
      complex TFcor(nbp,3)
      complex ctra
!
!     ------------------------------------------------------------------
! *   Objet  : deconvolution des formes d'onde x y z
! *   Classe : depouillement specifique GEOS/UBF
! *   Auteur : P. Robert, CRPE, 1977-1984
!              Revision P. Robert Janv 2010 F90 et optimisation
!     ------------------------------------------------------------------
!
! *** centrage, apodisation legere (trapeze) puis
!     fft-correction des gains d'antenne-correction des retards
!     d'echantillonnage (inutile ici)-filtrage-fft inverse.
!     Le signal d'entree en volts est suppose debarrasse de la forte
!     composante a la frequence de spin.


! *** centrage
!     --------

      xio(:)= xio(:) -SUM(xio)/float(nbp)

! *** apodisation Gaussienne

      edge= 1.e-3
      sigma= float(nbp/2)/sqrt(-alog(edge))

      do i=1,nbp
         xio(i)= xio(i)*exp(-((float(i-nbp/2)/sigma)**2))
      end do

! *** passage en frequence
!     --------------------

      tra(:)=0.
      call fftpat_XY(xio,tra,nbp,DSS,DCS,M_Kern, 1)

! *** correction des gains d'antenne et filtrage
!     ------------------------------------------

      do i=1,nbp
         ctra=cmplx(xio(i),tra(i))*TFcor(i,ix)
         xio(i)=real(ctra)
         tra(i)=aimag(ctra)
      enddo

! *** retour au domaine temps
!     -----------------------

      call fftpat_XY(xio,tra,nbp,DSS,DCS,M_Kern,-1)

      return
      end
!
! XXXX7XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX+XX
!
  subroutine deconvo_C3(Z,TFcor,N,DSS,DCS,M)

  complex (kind=4), dimension(3,N) :: Z,TFcor
  real    (kind=4), dimension(M)   :: DSS,DCS
  complex (kind=4)                 :: U,S,T

!     ---------------------------------------------------------------0--
! *   Objet  : deconvolution optimisee par D-FFT et I-FFT incluses
! *   Classe : traitement du signal
! *   Auteur : P. Robert, LPP , 2012
!     ---------------------------------------------------------------0--
!
! FFT directe

  DO L=1,M
     LE=2**(M+1-L)
     LE1=LE/2
     U=cmplx(1.,0.)
     S=cmplx(DCS(L),-DSS(L))

     DO J=1,LE1
        DO I=J,N,LE
           IP=I+LE1

           T=Z(1,I)+Z(1,IP)
           Z(1,IP)=(Z(1,I)-Z(1,IP))*U
           Z(1,I)=T

           T=Z(2,I)+Z(2,IP)
           Z(2,IP)=(Z(2,I)-Z(2,IP))*U
           Z(2,I)=T

           T=Z(3,I)+Z(3,IP)
           Z(3,IP)=(Z(3,I)-Z(3,IP))*U
           Z(3,I)=T
        ENDDO
        U=U*S
     ENDDO
  ENDDO

  NV2=N/2
  NM1=N-1
  J=1

  DO I=1,NM1
     IF (I < J) then

        T=Z(1,J)
        Z(1,J)=Z(1,I)
        Z(1,I)=T

        T=Z(2,J)
        Z(2,J)=Z(2,I)
        Z(2,I)=T

        T=Z(3,J)
        Z(3,J)=Z(3,I)
        Z(3,I)=T
     ENDIF

     K=NV2
     do while (K < J)
        J=J-K
        K=K/2
     enddo
     J=J+K
  ENDDO

! *   Normalisation et correction du filtre

     do i=1,N
     Z(1,i)=Z(1,i)*TFcor(1,i)
     Z(2,i)=Z(2,i)*TFcor(2,i)
     Z(3,i)=Z(3,i)*TFcor(3,i)
     enddo
!
!FFT inverse


  DO L=1,M
     LE=2**(M+1-L)
     LE1=LE/2
     U=cmplx(1.,0.)
     S=cmplx(DCS(L),DSS(L))

     DO J=1,LE1
        DO I=J,N,LE
           IP=I+LE1

           T=Z(1,I)+Z(1,IP)
           Z(1,IP)=(Z(1,I)-Z(1,IP))*U
           Z(1,I)=T

           T=Z(2,I)+Z(2,IP)
           Z(2,IP)=(Z(2,I)-Z(2,IP))*U
           Z(2,I)=T

           T=Z(3,I)+Z(3,IP)
           Z(3,IP)=(Z(3,I)-Z(3,IP))*U
           Z(3,I)=T
        ENDDO
        U=U*S
     ENDDO
  ENDDO

  NV2=N/2
  NM1=N-1
  J=1

  DO I=1,NM1
     IF (I < J) then

        T=Z(1,J)
        Z(1,J)=Z(1,I)
        Z(1,I)=T

        T=Z(2,J)
        Z(2,J)=Z(2,I)
        Z(2,I)=T

        T=Z(3,J)
        Z(3,J)=Z(3,I)
        Z(3,I)=T
     ENDIF

     K=NV2
     do while (K < J)
        J=J-K
        K=K/2
     enddo
     J=J+K
  ENDDO
  RETURN
  END
!
! XXXX7XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX+XX
!
  subroutine deconvo_C3_sd(Z,TFcor,N,DSS,DCS,M)

  complex (kind=4), dimension(3,N) :: Z,TFcor
  real    (kind=4), dimension(M)   :: DSS,DCS
  complex (kind=4)                 :: U,S,T

!     ---------------------------------------------------------------0--
! *   Objet  : deconvolution optimisee par D-FFT et I-FFT incluses
! *            avec correction des retards d'echantillonage (GEOS)
! *   Classe : traitement du signal
! *   Auteur : P. Robert, LPP , 2012
!     ---------------------------------------------------------------0--
!
! FFT directe

  DO L=1,M
     LE=2**(M+1-L)
     LE1=LE/2
     U=cmplx(1.,0.)
     S=cmplx(DCS(L),-DSS(L))

     DO J=1,LE1
        DO I=J,N,LE
           IP=I+LE1

           T=Z(1,I)+Z(1,IP)
           Z(1,IP)=(Z(1,I)-Z(1,IP))*U
           Z(1,I)=T

           T=Z(2,I)+Z(2,IP)
           Z(2,IP)=(Z(2,I)-Z(2,IP))*U
           Z(2,I)=T

           T=Z(3,I)+Z(3,IP)
           Z(3,IP)=(Z(3,I)-Z(3,IP))*U
           Z(3,I)=T
        ENDDO
        U=U*S
     ENDDO
  ENDDO

  NV2=N/2
  NM1=N-1
  J=1

  DO I=1,NM1
     IF (I < J) then

        T=Z(1,J)
        Z(1,J)=Z(1,I)
        Z(1,I)=T

        T=Z(2,J)
        Z(2,J)=Z(2,I)
        Z(2,I)=T

        T=Z(3,J)
        Z(3,J)=Z(3,I)
        Z(3,I)=T
     ENDIF

     K=NV2
     do while (K < J)
        J=J-K
        K=K/2
     enddo
     J=J+K
  ENDDO

! *   Normalisation et correction du filtre

     do i=1,N
     Z(1,i)=Z(1,i)*TFcor(1,i)
     Z(2,i)=Z(2,i)*TFcor(2,i)
     Z(3,i)=Z(3,i)*TFcor(3,i)
     enddo
!
!FFT inverse


  DO L=1,M
     LE=2**(M+1-L)
     LE1=LE/2
     U=cmplx(1.,0.)
     S=cmplx(DCS(L),DSS(L))

     DO J=1,LE1
        DO I=J,N,LE
           IP=I+LE1

           T=Z(1,I)+Z(1,IP)
           Z(1,IP)=(Z(1,I)-Z(1,IP))*U
           Z(1,I)=T

           T=Z(2,I)+Z(2,IP)
           Z(2,IP)=(Z(2,I)-Z(2,IP))*U
           Z(2,I)=T

           T=Z(3,I)+Z(3,IP)
           Z(3,IP)=(Z(3,I)-Z(3,IP))*U
           Z(3,I)=T
        ENDDO
        U=U*S
     ENDDO
  ENDDO

  NV2=N/2
  NM1=N-1
  J=1

  DO I=1,NM1
     IF (I < J) then

        T=Z(1,J)
        Z(1,J)=Z(1,I)
        Z(1,I)=T

        T=Z(2,J)
        Z(2,J)=Z(2,I)
        Z(2,I)=T

        T=Z(3,J)
        Z(3,J)=Z(3,I)
        Z(3,I)=T
     ENDIF

     K=NV2
     do while (K < J)
        J=J-K
        K=K/2
     enddo
     J=J+K
  ENDDO
  RETURN
  END
!
! XXXX7XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX+XX
!
      subroutine fftpat_XY(X,Y,N,DSS,DCS,M,IND)

      DIMENSION X(N),Y(N)
      DIMENSION DSS(M),DCS(M)

!     ---------------------------------------------------------------0--
!
! *   Objet  : fft directe ou inverse sans table de sinus
!              Modification P.Robert, decembre 2010, passage a F90
!              Externalisation de la table des sinus pour optimisation
! *   Classe : traitement du signal
! *   Auteur : P. Robert, CRPE, 1984, C/T
!     Input  : X(N), signal reel d'entree, Y(N) imaginaire
!                  N   Nombre de points, sous la forme 2**M
!                  ind=1 pour dft,-1 pour idft
!     Output : X(N),Y(N), spectre reel et imag
!                normalisation par n si ind=1
!
!     ---------------------------------------------------------------0--
!
! *   DEPUIS COOLEY-TUQUEY + CAPELLINI, 1984

      DO 20 L=1,M
         LE=2**(M+1-L)
         LE1=LE/2
         U1=1.
         U2=0.

         SS=DSS(L)
         CS=DCS(L)
         IF(IND.EQ.1) SS=-SS

         DO 18 J=1,LE1
            DO 15 I=J,N,LE
               IP=I+LE1

               T1=X(I)+X(IP)
               T2=Y(I)+Y(IP)
               T3=X(I)-X(IP)
               T4=Y(I)-Y(IP)

               X(IP)=T3*U1-T4*U2
               Y(IP)=T4*U1+T3*U2
               X(I)=T1
               Y(I)=T2
   15       continue

            U3=U1*CS-U2*SS
            U2=U2*CS+U1*SS
            U1=U3
   18    Continue
   20    Continue

      NV2=N/2
      NM1=N-1
      J=1
!
      DO 30 I=1,NM1
         IF (I.LT.J) then
            T1=X(J)
            T2=Y(J)

            X(J)=X(I)
            Y(J)=Y(I)
            X(I)=T1
            Y(I)=T2
         endif

         K=NV2

         do while (K.LT.J)
            J=J-K
            K=K/2
         enddo

         J=J+K
   30 Continue

! *   Normalisation si TF directe

      IF(IND.EQ.1) THEN
                   X(:)=X(:)/Float(N)
                   Y(:)=Y(:)/Float(N)
      ENDIF

      RETURN
      END
!
! XXXX7XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX+XX
!

!
      subroutine lissage(x,nbp,nlis,iano)

      real x(nbp)
!
!     ---------------------------------------------------------------0--
! *   Objet  : lissage d'un tableau sur nlis points
! *   Classe : traitement du signal
! *   Auteur : P. Robert, CETP, 2001
!     ---------------------------------------------------------------0--
!
      iano=0

      if(nlis.gt.nbp/2) then
                        print 100, nlis,nbp
                        iano=1
                        return
                        endif

      if(nlis.le.1) then
                    print 200, nlis,nbp
                    iano=2
                    return
                    endif
!
! *** chaque point prends la valeur moyenne des nlis/2 de chaque cote

      do 10 i=1,nbp
      xsom=0.
!
! *** on reduit nlis pour les nlis/2 premiers et derniers points
!
      if(i.le.    nlis/2) then
                          nlisv=2*(i-1)+1
                          else
                          nlisv=nlis
                          endif
      if(i.gt.nbp-nlis/2) nlisv=2*(nbp-i)+1

! *   ij1=i+1-1-nlisv/2
! *   ij2=i+nlisv-1-nlisv/2

      do 20 j=1,nlisv
      ij=i+j-1-nlisv/2
      xsom=xsom+x(ij)
   20 continue
      xmoy=xsom/float(nlisv)
      x(i)=xmoy
   10 continue

  100 format(1x,'*** lissage: nlis >nbp/2, lissage impossible', &
                ' nlis=',i6,' nbp=',i6)
  200 format(1x,'*** lissage: nlis <2, lissage impossible', &
                ' nlis=',i6,' nbp=',i6)

      return
      end
!
! XXXX7XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX+XX
!
      subroutine modpha(s,rm,rp)
!
!     ------------------------------------------------------------------
! *   Objet  : calcule le module et la phase (degre) d un nb. complexe
! *   Classe : depouillement specifique GEOS/UBF
! *   Auteur : P. Robert, CRPE, 1977-1984
!     ------------------------------------------------------------------
!
      complex s
!
!                    *********************
!
!
      epsilum=1.e-36
      pisd=acos(-1.)/180.
      rm=cabs(s)

      si=aimag(s)+epsilum
      co=real(s)+epsilum

      rp=atan2(si,co)
      rp=rp/pisd

      return
      end
!
! XXXX7XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX+XX
!
     subroutine subtrend(xfo,work,nbp,nlis,iano)
!
!     ------------------------------------------------------------------
! *   Objet  : lissage d'une courbe et soustraction a l'original
! *   Classe : depouillement specifique CUSP/Current_Loop
! *   Auteur : P. Robert, CETP, 2003
!     ------------------------------------------------------------------
!
      dimension xfo(nbp),work(nbp)
!
!
! *** on met xfo dans work
!
      do 10 i=1,nbp
      work(i)=xfo(i)
   10 continue
!
! *** lissage de work
!
      call lissage(work,nbp,nlis,iano)
!
! *** soustraction du tableau lisse a l'original
!
      do 20 i=1,nbp
      xfo(i)=xfo(i)-work(i)
   20 continue

      return
      end
!
!     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
!
     subroutine test_blk_cont(nor_index,imsres,ierr)

!     ---------------------------------------------------------------0--
! *   Objet  : test sur la continuite des blocks
! *   Classe : depouillement specifique CLUSTER
! *   Auteur : P. Robert, LPP , 2010
!     ---------------------------------------------------------------0--

      character(len=*) :: nor_index
      save msdayp,jd1,icall

      data icall /0/

!     ------------------------------------------------------------------
! *** test on block continuity
!     ------------------------------------------------------------------

      icall=icall+1
      ierr=0

      call decode_datiso(nor_index,iyy,imm,idd,ih,im,is,ims,imc)
      call cjd2000(iyy,imm,idd,jd00)
      if (icall == 1) then
                  jd1=jd00
                  msdayp= (jd00-jd1)*86400000 +ih*3600000 +im*60000  &
                                              +is*1000 +ims
                  write(*,*)
                  write(*,*) 'first block read : ',trim(nor_index)
                  return
      endif

! *   test time difference with previous bloc

      msday= (jd00-jd1)*86400000 +ih*3600000 +im*60000  &
                                 +is*1000 +ims
      idiff= msday-msdayp
      msdayp=msday

! * backward time

      if (idiff < 0) then
                   write(*,*) '*** backward time at ',nor_index
                   write(*,*) '    diff=',idiff,' ms, i.e. ', idiff/60000, 'mn'
                   ierr=1
                   return
      endif

! * data gap

      if (abs(idiff-imsres) > imsres/2) then
                  write(*,*) '*** data gap at  : ',nor_index
                  write(*,*) '    diff=',idiff,' ms rather than ',imsres
                  write(*,*) '    i.e.=',idiff/60000, 'mn'
                  if(imsres > 1) then
                        write(*,*) '    vectors missing: ',idiff/imsres -1
                                 else
                        write(*,*) '    vectors missing: undetermined because time res < 1 msec.'
                  endif
                  ierr=2
                  return
      endif

      return
      end
!
! XXXX7XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX+XX
!
      subroutine test_blk_cont_no_cal(nor_index,ext_index,imsres,ierr)

!     ---------------------------------------------------------------0--
! *   Objet  : test sur la continuite des blocks/saut des calibrations
! *   Classe : depouillement specifique CLUSTER
! *   Auteur : P. Robert, LPP , 2010
!     ---------------------------------------------------------------0--

      character(len=*) :: nor_index,ext_index
      save msdayp,jd1,icall

      data icall /0/

      icall=icall+1
      ierr=0

!     first, check that block data does not contains cal sequence

!R. Piberne : change here the cal status
!if(ext_index(1:1) /= '0') then
      if(ext_index(8:8) /= '0') then
                      ierr=3
                      return
      endif


      call decode_datiso(nor_index,iyy,imm,idd,ih,im,is,ims,imc)
      call cjd2000(iyy,imm,idd,jd00)
      if (icall == 1) then
                  jd1=jd00
                  msdayp= (jd00-jd1)*86400000 +ih*3600000 +im*60000  &
                                              +is*1000 +ims
                  write(*,*)
                  write(*,*) 'first block read : ',trim(nor_index)
                  return
      endif

! *   test time difference with previous bloc

      msday= (jd00-jd1)*86400000 +ih*3600000 +im*60000  &
                                 +is*1000 +ims
      idiff= msday-msdayp
      msdayp=msday

! * backward time

      if (idiff < 0) then
                   write(*,*) '*** backward time at ',nor_index
                   write(*,*) '    diff=',idiff,' ms, i.e. ', idiff/60000, 'mn'
                   ierr=1
                   return
      endif

! * data gap

      if (abs(idiff-imsres) > imsres/2) then
                  write(*,*) '*** data gap at  : ',nor_index
                  write(*,*) '    diff=',idiff,' ms rather than ',imsres
                  write(*,*) '    i.e.=',idiff/60000, 'mn'
                  write(*,*) '    vectors missing: ',idiff/imsres -1
                  ierr=2
                  return
      endif

      return
      end
!
!     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
!

      subroutine w_com(pr_out,com)

!     ---------------------------------------------------------------0--
! *   Objet  : write on stdo comment only if pr_out is true
! *   Classe : depouillement specifique CLUSTER
! *   Auteur : P. Robert, LPP , 2010
!     ---------------------------------------------------------------0--

      logical pr_out
      character(len=*) :: com

      if (pr_out) then
          write(*,*) com
      endif

      return
      end
!
! XXXX7XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX+XX
!
      subroutine trotfix(wave,nbp,fe,fspin,rotdeg)
!
!     ------------------------------------------------------------------
! *   Objet  : passage d''un repere en rotation a un repere fixe
! *   Classe : depouillement specifique GEOS/UBF
! *   Auteur : P. Robert, CRPE, 1977-2008
!
!     ------------------------------------------------------------------
!
!     passage de xio,yio a un repere fixe
!     (celui du satellite immobilise a t tel que x soit dans un  plan
!     defini par rotdeg, par exemple celui du  meridien geographique
!     ou celui qui contiens la direction du Soleil)
!
      real, dimension (3,*) :: wave
!
!                    *********************
!
!
      pi=acos(-1.)
      pi2=2.*pi
      rotrad=rotdeg*pi/180.
!
      do 10 i=1,nbp
!
      tau=real(i-1)/fe
      depift= mod(pi2*fspin*tau,pi2)
      phicr=rotrad-depift
      sinphi=sin(phicr)
      cosphi=cos(phicr)

      xo=  cosphi*wave(1,i) +sinphi*wave(2,i)
      yo= -sinphi*wave(1,i) +cosphi*wave(2,i)
      wave(1,i)=xo
      wave(2,i)=yo
   10 continue
!
      return
      end
!
! XXXX7XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX+XX
!
      subroutine CAPODI(X,NBP,IAPOD,NOM)

      real (kind=4), dimension(NBP) :: X
      real (kind=8) D1_tmp
      character(len=*) :: NOM
!
!     ---------------------------------------------------------------0--
! *   Objet  : CALCUL UNE FENETRE D'APODISISATION. IAPOD=1,8
! *   Classe : TRAITEMENT DU SIGNAL
! *   Auteur : P. ROBERT, 1977-1985 ; version f90 P.R. mars 2015
!     ---------------------------------------------------------------0--
!
      PI2N=6.283185318/float(NBP)

      select case (IAPOD)

    case(1)

!     FENETRE TRAPEZE
      NOM='Trapeze'

      do I=1,NBP
         X(I)=1.
      enddo

      I=NBP/16
      do J=1,I
         X(j) = float(j-1)/float(i-1)
         jj=NBP -j +1
         X(jj)= float(j-1)/float(i-1)
      enddo
      return

    case(2)

!     FENETRE RECTANGLE ARONDI
      NOM='Rounded Rectangle'

      do I=1,NBP
         X(I)=1.
      enddo

      I=NBP/8
      do J=1,I
         X(J) = SIN(3.1415927*float(J)/(2.*float(I)))
         JJ=NBP -J +1
         X(JJ)= SIN(3.1415927*float(J)/(2.*float(I)))
      enddo
      return

    case(3)

!     FENETRE TRAPEZE ARONDI
      NOM='Rounded Trapeze'

      do I=1,NBP
         X(I)=1.
      enddo

      I=NBP/4
      do J=1,I
         X(J) = (SIN(3.1415927*float(J)/(2.*float(I))))**2
         JJ=NBP -J +1
         X(JJ)= (SIN(3.1415927*float(J)/(2.*float(I))))**2
      enddo
      return

    case(4)

!     FENETRE DE RIESZ
      NOM='Riesz'

      do I=1,NBP
         RN=float(I-1)
         X(I)=1.-(float(2*NBP)/RN -1.)**2
      enddo
      return

    case(5)

!     FENETRE DE HANN (HANNING)
      NOM='Hanning'

      do I=1,NBP
         RN=float(I-1)
         X(I)=0.5*(1. -COS(PI2N*RN))
      enddo
      return

    case(6)

!     FENETRE DE KAISER-BESSEL
      Nom='Kaiser-Bessel'

      do I=1,NBP
         RN=float(I-1)
         X(I)=0.40243-0.49804*COS(PI2N*RN)+0.09831*COS(PI2N*2.*RN) &
             -0.00122*COS(PI2N*3.*RN)
!     fac=2.4849
      enddo
      return

    case(7)

!     FENETRE DE BLACKMAN-HARRIS
      NOM='Blackman-Harris'

      do I=1,NBP
         RN=float(I-1)
         X(I)=0.35875-0.48829*COS(PI2N*RN)+0.14128*COS(PI2N*2.*RN) &
             -0.01168*COS(PI2N*3.*RN)
      enddo
      return

    case(8)

!     FENETRE GAUSSIENNE
      NOM='Gauss'

      edge= 1.e-3
      sigma= float(NBP/2)/sqrt(-alog(edge))

      do I=1,NBP
!       le -0.5 est pour symétriser la gaussienne
!       avec le sommet entre N/2 et N/2 +1
        D1_tmp=(dble(I-NBP/2)-0.5D0)/dble(sigma)
        X(I)= sngl(exp(-(D1_tmp**2D0)))
      enddo
      return

    case default
         write(*,*)
         write(*,'(a)') ' -> CAPODI: NOTHING DONE...'
  end select

      END

!     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      SUBROUTINE csurf(x,nbp,nom,surf)
!
!     ---------------------------------------------------------------0--
! *   Objet  : calcule la surface du tableau X sur nbp points
! *   Classe : TRAITEMENT DU SIGNAL
! *   Auteur : P. ROBERT, 1977-1985
!     ---------------------------------------------------------------0--
!
      real (kind=4), dimension(nbp) :: x
      character(len=*) :: nom

      sum=0.
      sum2=0.

      do i=1,nbp
      sum=sum+x(i)
      sum2=sum2+x(i)**2
      enddo

      surf=sum/float(nbp)
      surf2=sum2/float(nbp)

      print*, 'normalisation factor ',nom,' ',surf, 1./surf,  &
                                              surf2, 1./surf2
      return
      end

!     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
