
!     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

!     subroutines pour calculer le mouvement d'un satellite autour 
!     d'un corps central de masse >> celle du satellite par les lois
!     de Kepler (sans autres perturbations).
!     Peut etre utilise pour calculer le mouvement de la terre autour 
!     du soleil, ou celui d'un satellite autour de la Terre.
!
!     P. Robert, CRPE, ~1985, revu LPP novembre 2010

!     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      subroutine co_axes(apo,per,a,b,c,e)

      real*8 apo,per, a,b,c,e

!     ------------------------------------------------------------------
! *   Objet  : calcule grand axe et excentricite de l'ellipse
! *   Auteur : P. Robert, LPP, Novembre 2010
!     ------------------------------------------------------------------
!     ---------------------------------------------------------------0--
!     calcule grand axe et excentricite de l'ellipse
!     a partir de l'apogee et du perigee
!     ---------------------------------------------------------------0--

!     demi grand axe de l'orbite (km) :
      a= (apo+per)/2.d0

!     excentricite :
      c= (apo-per)/2.d0
      e= c/a

!     demi petit axe
      b=sqrt(a*a -c*c)

      return
      end

!     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      subroutine co_period(a, GM, GT)

      real*8 a,pi,work

!     ------------------------------------------------------------------
! *   Objet  : calcul de la periode d'un satellite autour d'un corps central
! *   Auteur :  P. Robert, LPP, Novembre 2010
!     ------------------------------------------------------------------
!     ---------------------------------------------------------------0--
!     calcul de la periode d'un satellite autour d'un corps central
!     (3eme loi de Kepler)

!     in : a,  grand axe de l'ellipse, en km (real*8)
!          GM, masse du corps central, terre pour les sat terrestres
!     out: GT, periode de revolution
!     ---------------------------------------------------------------0--

!     contante de gravitation en m3.kg-1.s-2
      G= 6.67428e-11 ! http://physics.nist.gov/cgi-bin/cuu
      G= 6.67421e-11 ! http://fr.wikipedia.org/wiki/Constantes_physiques
      G= 6.67310e-11 ! web
      G= 6.67260e-11 ! NLR Plasma Formulary (ce qui marche le mieux !)
      G= 6.67259e-11 ! http://www.jacquesfortier.com/zweb/jf/ConstanteG.html
      pi=acos(-1.d0)

      work= 4.d0*pi*pi*a*a*a*1.d9/(dble(GM)*dble(G))
      GT=sngl(sqrt(work))

      return
      end

!     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      subroutine co_apex_time(GT,e, Tapex)

      real*8 e,pi

!     ------------------------------------------------------------------
! *   Objet  : calcul du temp de passage au premier sommet secondaire (ou apex)
! *   Auteur : P. Robert, LPP, Novembre 2010
!     ------------------------------------------------------------------
!     ---------------------------------------------------------------0--
!     calcul du temp de passage au premier sommet secondaire (ou apex)
!     soit pour AE=90 (t=0 est pris au perigee).
!     Input  : GT, periode de revolution
!               e, excentricite
!     Output : Tapex, temps de passage a l'apex (meme unite que GT)
!     ---------------------------------------------------------------0--

!     Anomalie excentrique AE et equation de Kepler: AE -e*sin(AE)=AM
!     avec AM= 2.*pi*time/GT

      pi=acos(-1.d0)
      Tapex=sngl( (0.5d0 -e/pi)*dble(GT)/2.d0)

      return
      end

!     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      subroutine co_anom_excen(e,time,GT,AE)

      real*8 e,AED,deltaD,delmiD,pisdd

!     ------------------------------------------------------------------
! *   Objet  : calcul de l'anomalie excentrique par resolution numerique
! *   Auteur : P. Robert, LPP, Novembre 2010
!     ------------------------------------------------------------------
!     ---------------------------------------------------------------0--
!     calcul de l'anomalie excentrique par resolution numerique 
!     de l'equation de Kepler

!     in : e, excentricite de l'ellipse (real*8)
!          time, temps depuis le perigee
!          GT, periode de revolution su satellite (meme unite que time)
!     out: AE, anomalie excentrique de l'equation de Kepler (radians)
!     ---------------------------------------------------------------0--

      pi=acos(-1.)
      pisd=pi/180.

!     anomalie moyenne

      AM= 2.*pi*time/GT

!     Anomalie excentrique AE et equation de Kepler: AE -e*sin(AE)=AM

!  1) resolution numerique grossiere a 1 degre pres

      delmin=1.e30
      AE=-1e30

      do i= 1,360
      AE=float(i-1)*pisd
      delta= abs(AE -sngl(e)*sin(AE)-AM)
      if(delta.lt.delmin) then
                          delmin=delta
                          AE1=AE
      endif
      enddo

!  2) resolution a 1/10 000 de degre autour de cette valeur grossiere

      delmiD= 1.d30
      AED= -1d30
      pisdd=acos(-1.d0)/180.d0

      do i= 0,20000
      AED= dble(AE1) + dble(i-10000)*pisdd/10000.d0
      deltaD= abs(AED -e*sin(AED)-dble(AM))
      if(deltaD.lt.delmiD) then
                           delmiD=deltaD
                           AE2=sngl(AED)
      endif
      enddo

      AE=AE2
      
      return
      end

!     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      subroutine co_anom_vraie(AE,e,a,AV,r)

      real*8 e,a,r
      real*8 DcosAE,DsinAE,DcosAV,DsinAV

!     ------------------------------------------------------------------
! *   Objet  : calcul de l'anomalie vraie donnant la position angulaire
! *   Auteur : P. Robert, LPP, Novembre 2010
!     ------------------------------------------------------------------
!     ---------------------------------------------------------------0--
!     calcul de l'anomalie vraie donnant la position angulaire
!     depuis le foyer  de l'ellipse

!     in : AE, anomalie excentrique, calculee par la resolution 
!              de l'equation de Kepler (radians)
!          e , excentricite de l'ellipse (real*8)
!          a , demi grand axe de l'ellipse (real*8)
!     out: AV, anomalie vraie de l'equation de Kepler (radians)
!          r , distance radiale (meme unite que a, et real*8)
!
!     rappel: AV est l'angle polaire, dans le plan de l'ellipse, 
!             compté depuis le perigee
!     ---------------------------------------------------------------0--


      DcosAE=cos(dble(AE))
      DsinAE=sin(dble(AE))

      r=a*(1.d0 -e*DcosAE)

      DcosAV= (DcosAE-e)/(1.d0-e*DcosAE)
      DsinAV= sqrt(1.d0 -e*e)*DsinAE/(1.d0 -e*DcosAE)

      AV= sngl( atan2(DsinAV,DcosAV) )

      return
      end
           
!     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      subroutine posgei(r,AV,a1,a2,a3,b1,b2,b3,sfac,x,y,z)

      real*8 r
	  
!     ------------------------------------------------------------------
! *   Objet  : calcul la position dans le GEI depuis l'anomalie vraie
! *   Auteur : P. Robert, LPP, Novembre 2010
!     ------------------------------------------------------------------

      sinAV=sin(AV)
      cosAV=cos(AV)

      x=sfac*sngl(r)*(a1*cosAV +b1*sinAV)
      y=sfac*sngl(r)*(a2*cosAV +b2*sinAV)
      z=sfac*sngl(r)*(a3*cosAV +b3*sinAV)

      return
      end 
!
!     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      subroutine U_cross_V(Ux,Uy,Uz,Vx,Vy,Vz,Wx,Wy,Wz,sina,cosa)

      real*8 Ux,Uy,Uz,Vx,Vy,Vz,UVx,UVy,UVz
      real*8 rmodUV, rmodW,dpro

!     ------------------------------------------------------------------
! *   Objet  : calcule le produit vectoriel de U et V en double precision
! *   Auteur : P. Robert, LPP, Novembre 2010
!     ------------------------------------------------------------------
!     ---------------------------------------------------------------0--
!     calcule le produit vectoriel de U et V en double precision
!     retourne aussi sinus et cosinus de l'angle (simple prec)
!     attention: le sinus est toujours positif
!     ---------------------------------------------------------------0--

      rmodUV=sqrt( (Ux**2 +Uy**2 +Uz**2)*(Vx**2 +Vy**2 +Vz**2) )

      UVx= Uy*Vz - Uz*Vy
      UVy= Uz*Vx - Ux*Vz
      UVz= Ux*Vy - Uy*Vx

      rmodW=sqrt(UVx**2 +UVy**2 +UVz**2)

      Wx=sngl(UVx/rmodW)
      Wy=sngl(UVy/rmodW)
      Wz=sngl(UVz/rmodW)

      dpro= Ux*Vx +Uy*Vy + Uz*Vz
      sina=sngl(rmodW/rmodUV)
      cosa=sngl(dpro/rmodUV)

      return
      end
!
!     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
!
