!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
!
! Laboratoire de Physique des Plasmas
! Modules and subroutines to read CLUSTER/STAFF-SC RFF WaveForm and
! VecTime files.
!
! Author   Patrick  ROBERT, LPP, 2011 January 23
! Updated Rodrigue PIBERNE, February 2011
! Updated Patrick  ROBERT,  March    2011
! Updated Patrick  ROBERT,  July     2011
!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  module rff_param_def

!----------------------------------------------------------------------!
! Object : modules for RFF Parameter definition
! Author : P. Robert , LPP, 2011 Jan. 23
! Revis. : P. Robert , LPP, 2011 July    06
!----------------------------------------------------------------------!

  integer, parameter :: par_len= 120 ! max length of the parameters
  integer, parameter :: txt_len=  80 ! max length of a TXT type line
  integer, parameter :: txt_lin=  50 ! max number of lines for TXT type

    TYPE Mandatory_param

       character(len=par_len) :: FILE_NAME
       character(len=par_len) :: FILE_CLASS
       character(len=par_len) :: FILE_FORMAT_VERSION
       character(len=par_len) :: FILE_CREATION_DATE

       character(len=par_len) :: MISSION_NAME
       character(len=par_len) :: OBSERVATORY_NAME
       integer                :: OBSERVATORY_NUMBER
       character(len=par_len) :: EXPERIMENT_NAME
       character(len=par_len) :: EXPERIMENT_MODE
       character(len=par_len) :: INSTRUMENT_TYPE
       character(len=par_len) :: MEASUREMENT_TYPE

       character(len=par_len) :: INDEX_LABEL
       character(len=par_len) :: INDEX_TYPE
       character(len=par_len) :: INDEX_UNITS
       character(len=par_len) :: INDEX_FORMAT
       character(len=par_len) :: INDEX_FORM
       integer                :: INDEX_DIMENSION
       character(len=par_len) :: INDEX_PROPERTIES

       character(len=par_len) :: INDEX_EXTENSION_LABEL
       character(len=par_len) :: INDEX_EXTENSION_TYPE
       character(len=par_len) :: INDEX_EXTENSION_UNITS
       character(len=par_len) :: INDEX_EXTENSION_FORMAT
       integer                :: INDEX_EXTENSION_LENGTH

       character(len=par_len) :: DATA_LABEL
       character(len=par_len) :: DATA_TYPE
       character(len=par_len) :: DATA_UNITS
       character(len=par_len) :: DATA_FORMAT
       character(len=par_len) :: DATA_FORM
       integer, dimension(2)  :: DATA_DIMENSION
       character(len=par_len) :: DATA_REPRESENTATION
       character(len=par_len) :: DATA_COORDINATE_SYSTEM
       character(len=par_len) :: DATA_FILL_VALUE

       integer                :: BLOCK_NUMBER
       character(len=par_len) :: BLOCK_FIRST_INDEX
       character(len=par_len) :: BLOCK_LAST_INDEX

    END TYPE Mandatory_param

!--------0-----------------------------------------------------------0--

    TYPE Optional_param

       real(kind=8)           :: TIME_RESOLUTION
       real(kind=8)           :: FREQUENCY_RESOLUTION

       character(len=32)      :: TIME_SPAN_FROM
       character(len=32)      :: TIME_SPAN_TO

       character(len=par_len) :: TITLE
       character(len=par_len) :: SUB_TITLE
       character(len=par_len) :: DISCIPLINE_NAME
       character(len=par_len) :: EXPERIMENT_PI_NAME
       character(len=par_len) :: EXPERIMENT_PI_MAIL

       character(len=txt_len), dimension(txt_lin) :: MISSION_DESCRIPTION
       character(len=txt_len), dimension(txt_lin) :: EXPERIMENT_DESCRIPTION
       character(len=txt_len), dimension(txt_lin) :: INDEX_DESCRIPTION
       character(len=txt_len), dimension(txt_lin) :: INDEX_EXTENSION_DESCRIP
       character(len=txt_len), dimension(txt_lin) :: DATA_DESCRIPTION
       character(len=txt_len), dimension(txt_lin) :: BLOCK_DESCRIPTION
       character(len=txt_len), dimension(txt_lin) :: FILE_ANOMALIES
       character(len=txt_len), dimension(txt_lin) :: HISTORY

       integer           :: MI_DE
       integer           :: EX_DE
       integer           :: IN_DE
       integer           :: IE_DE
       integer           :: DA_DE
       integer           :: BL_DE
       integer           :: FI_AN
       integer           :: BL_HI

    END TYPE Optional_param

!--------0-----------------------------------------------------------0--

    TYPE Constant_data

       character(len=par_len) :: TED_VERSION
       character(len=par_len) :: TCOR_OPTION

       real(kind=8)      :: SAMPLE_RATE
       real(kind=4)      :: VOLT_RANGE_MIN
       real(kind=4)      :: VOLT_RANGE_MAX

       integer           :: TM_RANGE_MIN
       integer           :: TM_RANGE_MAX

       real(kind=4)      :: MISALIGNMENT_MATRIX_L1_C1
       real(kind=4)      :: MISALIGNMENT_MATRIX_L1_C2
       real(kind=4)      :: MISALIGNMENT_MATRIX_L1_C3
       real(kind=4)      :: MISALIGNMENT_MATRIX_L2_C1
       real(kind=4)      :: MISALIGNMENT_MATRIX_L2_C2
       real(kind=4)      :: MISALIGNMENT_MATRIX_L2_C3
       real(kind=4)      :: MISALIGNMENT_MATRIX_L3_C1
       real(kind=4)      :: MISALIGNMENT_MATRIX_L3_C2
       real(kind=4)      :: MISALIGNMENT_MATRIX_L3_C3

       character(len=32) :: CONSTANT_TIME_MEASUREMENT
       character(len=32) :: SPECTRA_WEIGHTING

       real(kind=8)      :: SPIN_PERIOD
       real(kind=4)      :: SPIN_GEI_RIGHT_ASCENSION
       real(kind=4)      :: SPIN_GEI_DECLINATION
       real(kind=4)      :: MASS_CENTER_X
       real(kind=4)      :: MASS_CENTER_Y
       real(kind=4)      :: MASS_CENTER_Z
       real(kind=4)      :: EULER_ANGLE_FIRST
       real(kind=4)      :: EULER_ANGLE_SECOND

       real(kind=4)      :: FREQUENCY_FILTER_MIN
       real(kind=4)      :: FREQUENCY_FILTER_MAX
       real(kind=4)      :: FREQUENCY_CUT_OFF
       real(kind=4)      :: FREQUENCY_DETREND

       integer           :: CALIB_KERNEL_SIZE
       integer           :: CALIB_SHIFT_SIZE

       integer           :: SPECTRA_KERNEL_SIZE
       integer           :: SPECTRA_SHIFT_SIZE

    END TYPE Constant_data


!   variables used to read metadata

    TYPE(Mandatory_param) :: manda_param
    TYPE(Optional_param)  :: optio_param
    TYPE(Constant_data)   :: const_data

  END module RFF_param_def

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  module rff_data_def

!----------------------------------------------------------------------!
! Object : modules for RFF indexed data definition
! Author : R. Piberne, LPP, 2011 Feb. 17
!----------------------------------------------------------------------!

    use rff_param_def

    implicit none

    character(len=27), dimension(:),     allocatable :: data_index
    integer,           dimension(:,:),   allocatable :: I_data_vector
    integer,           dimension(:,:,:), allocatable :: I_data_matrix
    real(kind=4),      dimension(:,:),   allocatable :: R_data_vector
    real(kind=4),      dimension(:,:,:), allocatable :: R_data_matrix
    real(kind=4),      dimension(:),     allocatable :: phase

    character(len=27), dimension(:),     allocatable :: data_index2
    integer,           dimension(:,:),   allocatable :: I_data_vector2
    integer,           dimension(:,:,:), allocatable :: I_data_matrix2
    real(kind=4),      dimension(:,:),   allocatable :: R_data_vector2
    real(kind=4),      dimension(:,:,:), allocatable :: R_data_matrix2
    real(kind=4),      dimension(:),     allocatable :: phase2

    character(len=par_len), dimension(:),allocatable :: status
    character(len=par_len), dimension(:),allocatable :: status2


  end module rff_data_def

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_allocate_data_arrays

!----------------------------------------------------------------------!
! Object : Allocate data arrays from mandatory parameters
! Author : R. Piberne, LPP, 2011 Feb. 17
! Revis. : P. Robert, 2011 March 09
!----------------------------------------------------------------------!

  use rff_param_def
  use rff_data_def

  implicit none

  character(len=255) :: data_format
  integer, dimension(2):: data_dim
  integer :: nbvec, ierr1, ierr2, ierr3, ierr4, i

! ------------------------------------------


  write(*,*) ('-',i=1,72)
  write(*,*) 'allocate arrays for data...'

  nbvec=manda_param%BLOCK_NUMBER
  data_dim=manda_param%DATA_DIMENSION
  data_format=manda_param%DATA_FORMAT

  ierr1=0
  ierr2=0
  ierr3=0
  ierr4=0

! VecTime file
! ------------

  IF(manda_param%FILE_CLASS == 'VecTime') THEN
      IF(manda_param%DATA_TYPE == 'INT') THEN
         if (allocated(I_data_vector)) deallocate(I_data_vector, stat=ierr1)
         allocate(I_data_vector(data_dim(1),nbvec),stat=ierr2)
         write(*,*) '    I_data_vector allocated to ',data_dim(1),nbvec
      ENDIF

      IF(manda_param%DATA_TYPE == 'FLT') THEN
         if (allocated(R_data_vector)) deallocate(R_data_vector, stat=ierr1)
         allocate(R_data_vector(data_dim(1),nbvec),stat=ierr2)
         write(*,*) '    R_data_vector allocated to ',data_dim(1),nbvec
      ENDIF

  ENDIF

! WaveForm
! --------

  IF(manda_param%FILE_CLASS == 'WaveForm' .or. manda_param%FILE_CLASS == 'Spectrogram') THEN
      IF(manda_param%DATA_TYPE == 'INT') THEN
         if (allocated(I_data_matrix)) deallocate(I_data_matrix, stat=ierr3)
         allocate(I_data_matrix(data_dim(1),data_dim(2),nbvec),stat=ierr4)
         write(*,*) '    I_data_matrix allocated to ',data_dim(1),data_dim(2),nbvec
      ENDIF

      IF(manda_param%DATA_TYPE == 'FLT') THEN
         if (allocated(R_data_matrix)) deallocate(R_data_matrix, stat=ierr3)
         allocate(R_data_matrix(data_dim(1),data_dim(2),nbvec),stat=ierr4)
         write(*,*) '    R_data_matrix allocated to ',data_dim(1),data_dim(2),nbvec
      ENDIF

  ENDIF

  IF (ierr1 /= 0) STOP "*** lib_rw_rff/rff_allocate_data_arrays: 1-deallocate failed ***"
  IF (ierr2 /= 0) STOP "*** lib_rw_rff/rff_allocate_data_arrays: 2-Not enough memory ***"

  IF (ierr3 /= 0) STOP "*** lib_rw_rff/rff_allocate_data_arrays: 3-deallocate failed ***"
  IF (ierr4 /= 0) STOP "*** lib_rw_rff/rff_allocate_data_arrays: 4-Not enough memory ***"

! Both files
! ----------

  if(allocated(data_index)) deallocate(data_index, stat=ierr1)
  allocate(data_index(nbvec),stat=ierr2)

  if(allocated(phase)) deallocate(phase, stat=ierr1)
  allocate(phase(nbvec),stat=ierr2)

  if(allocated(status)) deallocate(status, stat=ierr1)
  allocate(status(nbvec),stat=ierr2)

  IF (ierr1 /= 0) STOP "*** lib_rw_rff/rff_allocate_data_arrays: 5-deallocate failed ***"
  IF (ierr2 /= 0) STOP "*** lib_rw_rff/rff_allocate_data_arrays: 6-Not enough memory ***"

  write(*,*) '    data_index  allocated to ',nbvec
  write(*,*) '    phase       allocated to ',nbvec
  write(*,*) '    status      allocated to ',nbvec

  write(*,*) 'done !'

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_get_current_date(datiso)

! ------------------------------------------------------------------
! * Object: donne la date ISO au moment du call
! * Author: P. Robert , LPP, 2011 Mar. 08
! ------------------------------------------------------------------
!
  implicit none
  character(len=8)  :: date
  character(len=10) :: time
  character(len=5)  :: zone
  character(len=*) :: datiso
  integer, dimension(8) :: values

  if(len(datiso).lt.24) then
        print*, '*** error in gdatiso'
        print*, '*** ABORTED !'
        write(*,*) 'gdatiso : *** ABORTED ! DATISO < 24 CHARACTERS ***'
  stop 'lib_rw_rff.o/gdatiso            : *** ERROR !! Program aborted !'
  endif

  call date_and_time(date, time, zone, values)
  datiso=date(1:4)//'-'//date(5:6)//'-'//date(7:8)//'T'// &
         time(1:2)//':'//time(3:4)//':'//time(5:10)//'Z'

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_set_default_init

!----------------------------------------------------------------------!
! Object : set all parameters to zero or 'undefined'(default)
! Author : R. Piberne, LPP, 2012 Mar. 08 -Rev. P. Robert, Jan 2021
!----------------------------------------------------------------------!

  use rff_param_def

  call rff_set_default_manda_param
  call rff_set_default_optio_param
  call rff_set_default_const_data

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_set_default_manda_param

!----------------------------------------------------------------------!
! Object : Set default mandatory parameters
! Author : P. Robert , ScientiDev, January 2021
! Comment: called before reading file
!----------------------------------------------------------------------!

  use rff_param_def

  manda_param%FILE_NAME               = 'undefined'
  manda_param%FILE_CLASS              = 'undefined'
  manda_param%FILE_FORMAT_VERSION     = 'undefined'
  manda_param%FILE_CREATION_DATE      = 'undefined'

  manda_param%MISSION_NAME            = 'undefined'
  manda_param%OBSERVATORY_NAME        = 'undefined'
  manda_param%OBSERVATORY_NUMBER      = 0
  manda_param%EXPERIMENT_NAME         = 'undefined'
  manda_param%EXPERIMENT_MODE         = 'undefined'
  manda_param%INSTRUMENT_TYPE         = 'undefined'
  manda_param%MEASUREMENT_TYPE        = 'undefined'

  manda_param%INDEX_LABEL             = 'undefined'
  manda_param%INDEX_TYPE              = 'undefined'
  manda_param%INDEX_UNITS             = 'undefined'
  manda_param%INDEX_FORMAT            = 'undefined'
  manda_param%INDEX_FORM              = 'undefined'
  manda_param%INDEX_DIMENSION         = 0
  manda_param%INDEX_PROPERTIES        = 'undefined'

  manda_param%INDEX_EXTENSION_LABEL   = 'undefined'
  manda_param%INDEX_EXTENSION_TYPE    = 'undefined'
  manda_param%INDEX_EXTENSION_UNITS   = 'undefined'
  manda_param%INDEX_EXTENSION_FORMAT  = 'undefined'
  manda_param%INDEX_EXTENSION_LENGTH  = 0

  manda_param%DATA_LABEL              = 'undefined'
  manda_param%DATA_TYPE               = 'undefined'
  manda_param%DATA_UNITS              = 'undefined'
  manda_param%DATA_FORMAT             = 'undefined'
  manda_param%DATA_FORM               = 'undefined'
  manda_param%DATA_DIMENSION(1)       = 0
  manda_param%DATA_DIMENSION(2)       = 0
  manda_param%DATA_REPRESENTATION     = 'undefined'
  manda_param%DATA_COORDINATE_SYSTEM  = 'undefined'
  manda_param%DATA_FILL_VALUE         = 'undefined'

  manda_param%BLOCK_NUMBER            = 0
  manda_param%BLOCK_FIRST_INDEX       = 'undefined'
  manda_param%BLOCK_LAST_INDEX        = 'undefined'

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_set_default_optio_param

!----------------------------------------------------------------------!
! Object : Set default optional parameters
! Author : P. Robert , ScientiDev, January 2021
! Comment: called before reading file
!----------------------------------------------------------------------!

  use rff_param_def

  optio_param%TIME_RESOLUTION      = 1.D30
  optio_param%FREQUENCY_RESOLUTION = 1.e30
  optio_param%TIME_SPAN_FROM       = 'undefined'
  optio_param%TIME_SPAN_TO         = 'undefined'
  optio_param%TITLE                = 'undefined'
  optio_param%SUB_TITLE            = 'undefined'
  optio_param%DISCIPLINE_NAME      = 'undefined'
  optio_param%EXPERIMENT_PI_NAME   = 'undefined'
  optio_param%EXPERIMENT_PI_MAIL   = 'undefined'

  optio_param%MISSION_DESCRIPTION(1)     = 'undefined'
  optio_param%EXPERIMENT_DESCRIPTION(1)  = 'undefined'
  optio_param%INDEX_DESCRIPTION(1)       = 'undefined'
  optio_param%INDEX_EXTENSION_DESCRIP(1) = 'undefined'
  optio_param%DATA_DESCRIPTION(1)        = 'undefined'
  optio_param%BLOCK_DESCRIPTION(1)       = 'undefined'
  optio_param%FILE_ANOMALIES(1)          = 'None'
  optio_param%HISTORY(1)                 = 'None'

  MI_DE = 1
  EX_DE = 1
  IN_DE = 1
  IE_DE = 1
  DA_DE = 1
  BL_DE = 1
  FI_AN = 1
  BL_HI = 1

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_set_default_const_data

!----------------------------------------------------------------------!
! Object : Set default constant data
! Author : P. Robert , ScientiDev, January 2021
! Comment: called before reading file
!----------------------------------------------------------------------!

  use rff_param_def


  const_data%TED_VERSION = 'undefined'
  const_data%TCOR_OPTION = 'undefined'

  const_data%SAMPLE_RATE    = 1.d30
  const_data%VOLT_RANGE_MIN = 1.e30
  const_data%VOLT_RANGE_MAX = 1.e30

  const_data%TM_RANGE_MIN   = 999999999
  const_data%TM_RANGE_MAX   = 999999999

  const_data%MISALIGNMENT_MATRIX_L1_C1 = 1.e30
  const_data%MISALIGNMENT_MATRIX_L1_C2 = 1.e30
  const_data%MISALIGNMENT_MATRIX_L1_C3 = 1.e30
  const_data%MISALIGNMENT_MATRIX_L2_C1 = 1.e30
  const_data%MISALIGNMENT_MATRIX_L2_C2 = 1.e30
  const_data%MISALIGNMENT_MATRIX_L2_C3 = 1.e30
  const_data%MISALIGNMENT_MATRIX_L3_C1 = 1.e30
  const_data%MISALIGNMENT_MATRIX_L3_C2 = 1.e30
  const_data%MISALIGNMENT_MATRIX_L3_C3 = 1.e30

  const_data%CONSTANT_TIME_MEASUREMENT = 'undefined'
  const_data%SPECTRA_WEIGHTING         = 'undefined'

  const_data%SPIN_PERIOD               = 1.e30
  const_data%SPIN_GEI_RIGHT_ASCENSION  = 1.e30
  const_data%SPIN_GEI_DECLINATION      = 1.e30
  const_data%MASS_CENTER_X             = 1.e30
  const_data%MASS_CENTER_Y             = 1.e30
  const_data%MASS_CENTER_Z             = 1.e30
  const_data%EULER_ANGLE_FIRST         = 1.e30
  const_data%EULER_ANGLE_SECOND        = 1.e30

  const_data%FREQUENCY_FILTER_MIN = 1.e30
  const_data%FREQUENCY_FILTER_MAX = 1.e30
  const_data%FREQUENCY_CUT_OFF    = 1.e30
  const_data%FREQUENCY_DETREND    = 1.e30

  const_data%CALIB_KERNEL_SIZE    = 999999999
  const_data%CALIB_SHIFT_SIZE     = 999999999

  const_data%SPECTRA_KERNEL_SIZE  = 999999999
  const_data%SPECTRA_SHIFT_SIZE   = 999999999

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_set_default_DATA_DESCRIPTION

!----------------------------------------------------------------------!
! Object : Set DATA_DESCRIPTION parameter value to default
! Author : P. Robert , LPP, 2012 Mar. 28
! Comment: should be call before writing file
!----------------------------------------------------------------------!

  use rff_param_def

  optio_param%DA_DE=1

  optio_param%DATA_DESCRIPTION(1)='None'

  IF(manda_param%FILE_CLASS == 'WaveForm') THEN

  optio_param%DATA_DESCRIPTION( 1)='Class of this file is "WaveForm", indexed by time. WaveForm class is also'
  optio_param%DATA_DESCRIPTION( 2)='called MatTime. This is a temporal series of data blocks. Each block begins'
  optio_param%DATA_DESCRIPTION( 3)='with the INDEX value (for example ISO epoch) and INDEX_EXTENSION values. '
  optio_param%DATA_DESCRIPTION( 4)='The rest of the block is a series of data values, over several lines, '
  optio_param%DATA_DESCRIPTION( 5)='according to the DATA_FORMAT value. The number of values per line is the '
  optio_param%DATA_DESCRIPTION( 6)='first  dimension of the matrix, the number of lines is  the second dimension'
  optio_param%DATA_DESCRIPTION( 7)='of the matrix. In case of WaveForm class rather than MatTime, each line of'
  optio_param%DATA_DESCRIPTION( 8)='the  matrix is a  time-dependant vector. Sample rate, given in CONSTANT_DATA,'
  optio_param%DATA_DESCRIPTION( 9)='allows time  interpolation inside a block. Time resolution, given in'
  optio_param%DATA_DESCRIPTION(10)='OPTIONAL_PARAMETERS, can also be used, and must be consistant with sample rate.'

  optio_param%DA_DE=10

  ENDIF

  IF(manda_param%FILE_CLASS == 'VecTime') THEN
  optio_param%DATA_DESCRIPTION(1)='Class of this file is "VecTime", meaning a data vector dependant of the time.'
  optio_param%DATA_DESCRIPTION(2)='DATA_FORM and INDEX_FORM must be "Vector" and "Scalar". This is a temporal '
  optio_param%DATA_DESCRIPTION(3)='series of data blocks. Each block begins with the INDEX value giving the '
  optio_param%DATA_DESCRIPTION(4)='time (for example in ISO epoch) and INDEX_EXTENSION values. The rest of the'
  optio_param%DATA_DESCRIPTION(5)='block is a series of values, corresponding  to the vector components, '
  optio_param%DATA_DESCRIPTION(6)='and according to the DATA_FORMAT value.  Sample rate, which can be found in'
  optio_param%DATA_DESCRIPTION(7)='CONSTANT_DATA, must be consistant with the value of time_resolution.'

  optio_param%DA_DE=7

  ENDIF

  IF(manda_param%FILE_CLASS == 'Spectrogram') THEN

  optio_param%DATA_DESCRIPTION( 1)='Class of this file is "Spectrogram", indexed by time. Spectrogram class is also'
  optio_param%DATA_DESCRIPTION( 2)='called MatTime. This is a temporal series of data blocks. Each block begins'
  optio_param%DATA_DESCRIPTION( 3)='with the INDEX value (for example ISO epoch) and INDEX_EXTENSION values.'
  optio_param%DATA_DESCRIPTION( 4)='The rest of the block is a series of data values, over several lines,'
  optio_param%DATA_DESCRIPTION( 5)='according to the DATA_FORMAT value. The number of values per line is the first'
  optio_param%DATA_DESCRIPTION( 6)='dimension of the matrix, the number of lines is  the second dimension of the'
  optio_param%DATA_DESCRIPTION( 7)='matrix. In case of Spectrogram class rather than MatTime, each line of'
  optio_param%DATA_DESCRIPTION( 8)='the  matrix is a frequency-dependant vector. Sample rate given in CONSTANT_DATA,'
  optio_param%DATA_DESCRIPTION( 9)='correspond to the initial WaveForm or Vectime data, whereas the time_resolution'
  optio_param%DATA_DESCRIPTION(10)='given in the OPTIONAL_PARAMETERS correspond to the time period between two'
  optio_param%DATA_DESCRIPTION(11)='spectra. Frequency resolution between two rays is given OPTIONAL_PARAMETERS,'
  optio_param%DATA_DESCRIPTION(12)='and start from zero. Spectra can be overlapped.'

  optio_param%DA_DE=12
  ENDIF

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_set_default_BLOCK_DESCRIPTION

!----------------------------------------------------------------------!
! Object : Set BLOCK_DESCRIPTION parameter value to default
! Author : P. Robert , LPP, 2012 Mar. 28
! Comment: should be call before writing file
!----------------------------------------------------------------------!

  use rff_param_def

  optio_param%BLOCK_DESCRIPTION(1)='A data block is composed of the index, the index extension and the data.'
  optio_param%BLOCK_DESCRIPTION(2)='A block can be entirely read or written by a single operation, by using'
  optio_param%BLOCK_DESCRIPTION(3)='a format composed with the concatenation of the INDEX_FORMAT, the'
  optio_param%BLOCK_DESCRIPTION(4)='INDEX_EXTENSION_FORMAT, and the DATA_FORMAT.'

  optio_param%BL_DE=4

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_set_default_INDEX_DESCRIPTION

!----------------------------------------------------------------------!
! Object : Set INDEX_DESCRIPTION parameter value to default
! Author : P. Robert , LPP, 2012 Mar. 28
! Comment: should be call before writing file
!----------------------------------------------------------------------!

  use rff_param_def

  optio_param%IN_DE=1

  optio_param%INDEX_DESCRIPTION(1)='None'

  IF(manda_param%INDEX_LABEL == 'Time' .and. manda_param%INDEX_UNITS == 'ISO_TIME' ) THEN

  optio_param%INDEX_DESCRIPTION(1)='Time is given in ISO format, accepting any digits for second field as'
  optio_param%INDEX_DESCRIPTION(2)='"2001-02-18T19:16:17.550934Z"; "T" and "Z" separators are required.'
  optio_param%IN_DE=2

  ENDIF

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_set_default_INDEX_EXTENSION_DESCRIP

!----------------------------------------------------------------------!
! Object : Set INDEX_EXTENSION_DESCRIP parameter value to default
! Author : P. Robert , LPP, 2012 Mar. 28
! Comment: should be call before writing file
!----------------------------------------------------------------------!

  use rff_param_def

  character(len=120) :: car

  optio_param%INDEX_EXTENSION_DESCRIP(1)='The extension of the index field is used to add some auxiliary data'
  optio_param%INDEX_EXTENSION_DESCRIP(2)='varying with the same rate as the index itself.'
  optio_param%INDEX_EXTENSION_DESCRIP(3)='The index extension field must contain only a limited series of scalar'
  optio_param%INDEX_EXTENSION_DESCRIP(4)='data; label, type and units can be different, but in accordance with the'
  optio_param%INDEX_EXTENSION_DESCRIP(5)='given format.'

  optio_param%IE_DE=5

  IF(manda_param% MISSION_NAME == 'CLUSTER' .and. &
     manda_param%EXPERIMENT_NAME == 'STAFF-SC') THEN
     car=manda_param%INDEX_EXTENSION_FORMAT
     car=adjustl(car)
     optio_param%INDEX_EXTENSION_DESCRIP(6)='Here, extended index contains a status word of ' &
                                             //car(3:4)//' characters, and the'
     optio_param%INDEX_EXTENSION_DESCRIP(7)='spin phase. Note that S/C HK used to calculate the phase are not'
     optio_param%INDEX_EXTENSION_DESCRIP(8)='corrected by TCOR, so the maximum error induced is 0.2 degrees.'
     optio_param%IE_DE=8
  ENDIF

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_update_history(credate,com)

!----------------------------------------------------------------------!
! Object : Update  HISTORY parameter in optional_parameters
! Author : P. Robert , LPP, 2012 Mar. 28
! Comment: should be call before writing file
!----------------------------------------------------------------------!

  use rff_param_def

  character(len=*)   :: com, credate
  character(len=80)  :: work
  character(len=64)  :: give_RPC_version

! ex: call rff_update_history(credate,'RPC_waveform_to_vectime')

  i=optio_param%BL_HI +1

  IF(i > 1) THEN
      work=optio_param%HISTORY(i-1)
      j=LEN_TRIM(work)
      if(work(j:j) == '}') work(j:j)=' '
      optio_param%HISTORY(i-1)=work(1:80)
  ENDIF

  optio_param%BL_HI= i
  optio_param%HISTORY(i)=credate(1:24)//' : '//trim(com)//' - '//TRIM(give_RPC_version())

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_R_file(ifc,file)

!----------------------------------------------------------------------!
! Object : Read rff file, metadata & data, WaveForm or VecTime
! Author : R. Piberne, LPP, 2011 Feb. 17
! Revis. : P. Robert, 2011 Mar. 09
!----------------------------------------------------------------------!

  use rff_param_def
  use rff_data_def

  character(len=*) :: file

! ------------------------------------------

  call rff_R_metadata(ifc,file)
! call rff_W_manda_param(6)
! call rff_W_optio_param(6)

  call rff_R_const_data(ifc)
! call rff_W_const_data(6)

  call rff_R_indexed_data(ifc)
  call rff_R_tail(ifc)

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_R_manda_param(ifc)

!----------------------------------------------------------------------!
! Object : Read  mandatory parameter in the rff file
! Author : P. Robert , LPP, 2011 Jan. 23
!----------------------------------------------------------------------!

  use rff_param_def

  character(len=120) :: get_paramC
  integer            :: get_paramI
  integer            :: get_paramI2

! ----------------------------------------------------------------------------
! set default values before reading

  call rff_set_default_manda_param

! ----------------------------------------------------------------------------

  manda_param%FILE_NAME              = TRIM(get_paramC('FILE_NAME             ',ifc))
  manda_param%FILE_CLASS             = TRIM(get_paramC('FILE_CLASS            ',ifc))
  manda_param%FILE_FORMAT_VERSION    = TRIM(get_paramC('FILE_FORMAT_VERSION   ',ifc))
  manda_param%FILE_CREATION_DATE     = TRIM(get_paramC('FILE_CREATION_DATE    ',ifc))

  manda_param%MISSION_NAME           = TRIM(get_paramC('MISSION_NAME          ',ifc))
  manda_param%OBSERVATORY_NAME       = TRIM(get_paramC('OBSERVATORY_NAME      ',ifc))
  manda_param%OBSERVATORY_NUMBER     =      get_paramI('OBSERVATORY_NUMBER    ',ifc)
  manda_param%EXPERIMENT_NAME        = TRIM(get_paramC('EXPERIMENT_NAME       ',ifc))
  manda_param%EXPERIMENT_MODE        = TRIM(get_paramC('EXPERIMENT_MODE       ',ifc))
  manda_param%INSTRUMENT_TYPE        = TRIM(get_paramC('INSTRUMENT_TYPE       ',ifc))
  manda_param%MEASUREMENT_TYPE       = TRIM(get_paramC('MEASUREMENT_TYPE      ',ifc))

  manda_param%INDEX_LABEL            = TRIM(get_paramC('INDEX_LABEL           ',ifc))
  manda_param%INDEX_TYPE             = TRIM(get_paramC('INDEX_TYPE            ',ifc))
  manda_param%INDEX_UNITS            = TRIM(get_paramC('INDEX_UNITS           ',ifc))
  manda_param%INDEX_FORMAT           = TRIM(get_paramC('INDEX_FORMAT          ',ifc))
  manda_param%INDEX_FORM             = TRIM(get_paramC('INDEX_FORM            ',ifc))
  manda_param%INDEX_DIMENSION        =      get_paramI('INDEX_DIMENSION       ',ifc)
  manda_param%INDEX_PROPERTIES       = TRIM(get_paramC('INDEX_PROPERTIES      ',ifc))

  manda_param%INDEX_EXTENSION_LABEL  = TRIM(get_paramC('INDEX_EXTENSION_LABEL ',ifc))
  manda_param%INDEX_EXTENSION_TYPE   = TRIM(get_paramC('INDEX_EXTENSION_TYPE  ',ifc))
  manda_param%INDEX_EXTENSION_UNITS  = TRIM(get_paramC('INDEX_EXTENSION_UNITS ',ifc))
  manda_param%INDEX_EXTENSION_FORMAT = TRIM(get_paramC('INDEX_EXTENSION_FORMAT',ifc))
  manda_param%INDEX_EXTENSION_LENGTH =      get_paramI('INDEX_EXTENSION_LENGTH',ifc)

  manda_param%DATA_LABEL             = TRIM(get_paramC('DATA_LABEL            ',ifc))
  manda_param%DATA_TYPE              = TRIM(get_paramC('DATA_TYPE             ',ifc))
  manda_param%DATA_UNITS             = TRIM(get_paramC('DATA_UNITS            ',ifc))
  manda_param%DATA_FORMAT            = TRIM(get_paramC('DATA_FORMAT           ',ifc))
  manda_param%DATA_FORM              = TRIM(get_paramC('DATA_FORM             ',ifc))
  manda_param%DATA_DIMENSION(1)      =      get_paramI('DATA_DIMENSION        ',ifc)
  manda_param%DATA_DIMENSION(2)      =     get_paramI2('DATA_DIMENSION        ',ifc)
  manda_param%DATA_REPRESENTATION    = TRIM(get_paramC('DATA_REPRESENTATION   ',ifc))
  manda_param%DATA_COORDINATE_SYSTEM = TRIM(get_paramC('DATA_COORDINATE_SYSTEM',ifc))
  manda_param%DATA_FILL_VALUE        = TRIM(get_paramC('DATA_FILL_VALUE       ',ifc))

  manda_param%BLOCK_NUMBER           =      get_paramI('BLOCK_NUMBER          ',ifc)
  manda_param%BLOCK_FIRST_INDEX      = TRIM(get_paramC('BLOCK_FIRST_INDEX     ',ifc))
  manda_param%BLOCK_LAST_INDEX       = TRIM(get_paramC('BLOCK_LAST_INDEX      ',ifc))

! on met les autres parametres par defaut

  call rff_set_default_DATA_DESCRIPTION
  call rff_set_default_BLOCK_DESCRIPTION
  call rff_set_default_INDEX_DESCRIPTION
  call rff_set_default_INDEX_EXTENSION_DESCRIP

  RETURN
  END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_R_optio_param(ifc)

!----------------------------------------------------------------------!
! Object : Read  optional parameters in the rff file
! Author : P. Robert , LPP, 2011 Jan. 23
!----------------------------------------------------------------------!

  use rff_param_def

  character(len=120)   :: get_paramC
  real(kind=8)         :: get_paramD

  interface
    function get_paramT(param,ifc,nbli)
!     use of txt_len and txt_lin unpossible here, we put the litteral values
      character(len=80), dimension(50) :: get_paramT
      character(len=*)   :: param
    end function get_paramT
  end interface

! ----------------------------------------------------------------------------
! set default values before reading

  call rff_set_default_optio_param

! ----------------------------------------------------------------------------

  optio_param%TIME_RESOLUTION      =      get_paramD('TIME_RESOLUTION ',ifc)
  if(manda_param%FILE_CLASS == 'Spectrogram') then
  optio_param%FREQUENCY_RESOLUTION =      get_paramD('FREQUENCY_RESOLUTION ',ifc)
  endif

  optio_param%TIME_SPAN_FROM       = TRIM(get_paramC('TIME_SPAN_FROM     ',ifc))
  optio_param%TIME_SPAN_TO         = TRIM(get_paramC('TIME_SPAN_TO       ',ifc))
  optio_param%TITLE                = TRIM(get_paramC('TITLE              ',ifc))
  optio_param%SUB_TITLE            = TRIM(get_paramC('SUB_TITLE          ',ifc))
  optio_param%DISCIPLINE_NAME      = TRIM(get_paramC('DISCIPLINE_NAME    ',ifc))
  optio_param%EXPERIMENT_PI_NAME   = TRIM(get_paramC('EXPERIMENT_PI_NAME ',ifc))
  optio_param%EXPERIMENT_PI_MAIL   = TRIM(get_paramC('EXPERIMENT_PI_MAIL ',ifc))

  optio_param%MISSION_DESCRIPTION    = get_paramT('MISSION_DESCRIPTION    ',ifc,nbli)
  optio_param%MI_DE= nbli
  optio_param%EXPERIMENT_DESCRIPTION = get_paramT('EXPERIMENT_DESCRIPTION ',ifc,nbli)
  optio_param%EX_DE= nbli
  optio_param%INDEX_DESCRIPTION      = get_paramT('INDEX_DESCRIPTION      ',ifc,nbli)
  optio_param%IN_DE= nbli
  optio_param%INDEX_EXTENSION_DESCRIP= get_paramT('INDEX_EXTENSION_DESCRIP',ifc,nbli)
  optio_param%IE_DE= nbli
  optio_param%DATA_DESCRIPTION       = get_paramT('DATA_DESCRIPTION       ',ifc,nbli)
  optio_param%DA_DE= nbli
  optio_param%BLOCK_DESCRIPTION      = get_paramT('BLOCK_DESCRIPTION      ',ifc,nbli)
  optio_param%BL_DE= nbli
  optio_param%FILE_ANOMALIES         = get_paramT('FILE_ANOMALIES         ',ifc,nbli)
  optio_param%FI_AN= nbli
  optio_param%HISTORY                = get_paramT('HISTORY                ',ifc,nbli)
  optio_param%BL_HI= nbli

  RETURN
  END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_R_const_data(ifc)

!----------------------------------------------------------------------!
! Object : Read  constant data in the rff file
! Author : P. Robert , LPP, 2011 Jan. 23, Rev. Nov 2020
!----------------------------------------------------------------------!

  use rff_param_def

  implicit none

  character(len=120)  :: get_paramC
  integer             :: get_paramI, ifc
  real(kind=4)        :: get_paramR
  real(kind=8)        :: get_paramD


! ----------------------------------------------------------------------
! set default values before reading

  call rff_set_default_const_data

! ----------------------------------------------------------------------


  if(manda_param%MISSION_NAME == 'CLUSTER' .AND. &
     manda_param%EXPERIMENT_NAME == 'STAFF-SC') then
    const_data%TED_VERSION          = TRIM(get_paramC('TED_VERSION',ifc))
    const_data%TCOR_OPTION          = TRIM(get_paramC('TCOR_OPTION',ifc))
    const_data%MISALIGNMENT_MATRIX_L1_C1 = get_paramR('MISALIGNMENT_MATRIX_L1_C1',ifc)
    const_data%MISALIGNMENT_MATRIX_L1_C2 = get_paramR('MISALIGNMENT_MATRIX_L1_C2',ifc)
    const_data%MISALIGNMENT_MATRIX_L1_C3 = get_paramR('MISALIGNMENT_MATRIX_L1_C3',ifc)
    const_data%MISALIGNMENT_MATRIX_L2_C1 = get_paramR('MISALIGNMENT_MATRIX_L2_C1',ifc)
    const_data%MISALIGNMENT_MATRIX_L2_C2 = get_paramR('MISALIGNMENT_MATRIX_L2_C2',ifc)
    const_data%MISALIGNMENT_MATRIX_L2_C3 = get_paramR('MISALIGNMENT_MATRIX_L2_C3',ifc)
    const_data%MISALIGNMENT_MATRIX_L3_C1 = get_paramR('MISALIGNMENT_MATRIX_L3_C1',ifc)
    const_data%MISALIGNMENT_MATRIX_L3_C2 = get_paramR('MISALIGNMENT_MATRIX_L3_C2',ifc)
    const_data%MISALIGNMENT_MATRIX_L3_C3 = get_paramR('MISALIGNMENT_MATRIX_L3_C3',ifc)

    const_data%SPIN_GEI_RIGHT_ASCENSION  = get_paramR('SPIN_GEI_RIGHT_ASCENSION',ifc)
    const_data%SPIN_GEI_DECLINATION      = get_paramR('SPIN_GEI_DECLINATION',ifc)
    const_data%MASS_CENTER_X             = get_paramR('MASS_CENTER_X',ifc)
    const_data%MASS_CENTER_Y             = get_paramR('MASS_CENTER_Y',ifc)
    const_data%MASS_CENTER_Z             = get_paramR('MASS_CENTER_Z',ifc)
    const_data%EULER_ANGLE_FIRST         = get_paramR('EULER_ANGLE_FIRST ',ifc)
    const_data%EULER_ANGLE_SECOND        = get_paramR('EULER_ANGLE_SECOND',ifc)
    const_data%CONSTANT_TIME_MEASUREMENT = TRIM(get_paramC('CONSTANT_TIME_MEASUREMENT',ifc))

  endif

  if(manda_param%DATA_UNITS(1:2) == 'TM') then
    const_data%VOLT_RANGE_MIN            = get_paramR('VOLT_RANGE_MIN',ifc)
    const_data%VOLT_RANGE_MAX            = get_paramR('VOLT_RANGE_MAX',ifc)
    const_data%TM_RANGE_MIN              = get_paramI('TM_RANGE_MIN',ifc)
    const_data%TM_RANGE_MAX              = get_paramI('TM_RANGE_MAX',ifc)
  endif



  if(manda_param%DATA_UNITS(1:2) == 'nT') then

     const_data%FREQUENCY_FILTER_MIN = get_paramR('FREQUENCY_FILTER_MIN',ifc)
     const_data%FREQUENCY_FILTER_MAX = get_paramR('FREQUENCY_FILTER_MAX',ifc)
     const_data%FREQUENCY_CUT_OFF    = get_paramR('FREQUENCY_CUT_OFF',ifc)
     const_data%FREQUENCY_DETREND    = get_paramR('FREQUENCY_DETREND',ifc)
     const_data%CALIB_KERNEL_SIZE    = get_paramI('CALIB_KERNEL_SIZE',ifc)
     const_data%CALIB_SHIFT_SIZE     = get_paramI('CALIB_SHIFT_SIZE',ifc)

  endif


  const_data%SAMPLE_RATE              = get_paramD('SAMPLE_RATE',ifc)
  const_data%SPIN_PERIOD              = get_paramD('SPIN_PERIOD',ifc)


  if(manda_param%file_class == 'Spectrogram') then

     const_data%SPECTRA_KERNEL_SIZE  = get_paramI('SPECTRA_KERNEL_SIZE',ifc)
     const_data%SPECTRA_SHIFT_SIZE   = get_paramI('SPECTRA_SHIFT_SIZE',ifc)
     const_data%SPECTRA_WEIGHTING    = TRIM(get_paramC('SPECTRA_WEIGHTING',ifc))

  endif

  RETURN
  END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_R_metadata(ifc,file)

!----------------------------------------------------------------------!
! Object : Open RFF file, read header, metadata & constant data
! Author : R. Piberne, LPP, 2011 Feb. 17
!----------------------------------------------------------------------!

  use rff_param_def

  implicit none

  character(len=*) :: file
  character(len=255) :: io_emsg
  character(len=255) :: Fbasename
  integer            :: ifc, err

! -----------------------------------


! 1) open file
!    ---------

  write(*,*)
  write(*,*) 'Opening file ',TRIM(file)
  close(ifc)
  OPEN(ifc,file=file,status='old', form='formatted',iostat=err, iomsg=io_emsg)

  if(err /= 0) then
               write(*,*) '*** unable to open file : ',TRIM(file)
               write(*,*) '    in subroutine rff_R_metadata, on unit ',ifc
               write(*,*) '    err= ',err
               write(*,*) trim(io_emsg)
               write(*,*) '    program aborted !!!'
       stop 'lib_rw_rff.o/rff_R_metadata     : *** ERROR !! Program aborted !'
  endif


! 2) read metadata
!    -------------

  write(*,*)
  write(*,*) 'reading mandatory parameters...'

  call rff_R_manda_param(ifc)

  if (TRIM(Fbasename(file)) /= manda_param%file_name) then
     write(*,*) '*  WARNING !!!,      open file is ',TRIM(Fbasename(file))
     write(*,*) '   while manda_param%file_name is ',TRIM(manda_param%file_name)
  endif

  write(*,*)
  write(*,*) 'reading optional parameters...'

  call rff_R_optio_param(ifc)

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_R_indexed_data(ifc)

!----------------------------------------------------------------------!
! Object : Read  indexed data for waveform and vectime RFF files
! Author : R. Piberne, LPP, 2011 Feb. 17
! Revis. : P. Robert, 2011 Mar. 09, Jan 2021
!----------------------------------------------------------------------!

  use rff_param_def
  use rff_data_def

  implicit none

  character(len=255) :: ligne
  character(len=par_len) :: data_format, index_extension_format, ind_ext,work_fmt
  integer :: get_pos, ifc, ierr, i, ieof, ipos1, ipos2,n_type,b_number,iwork,ipha

! ------------------------------------------

  write(*,*) ('-',i=1,72)
  write(*,*) 'Reading indexed data'

  call rff_allocate_data_arrays

! warning to input format containing "," unstead of "1x"

  call rff_format_W_to_R(manda_param%DATA_FORMAT, data_format)
  call rff_format_W_to_R(manda_param%INDEX_EXTENSION_FORMAT, index_extension_format)

  b_number=manda_param%BLOCK_NUMBER
  write(*,*)
  write(*,*) 'Reading indexed data, nb. block to read=', b_number
  write(*,*) 'Please wait...'

  n_type = INDEX(manda_param%INDEX_EXTENSION_TYPE, ';')

  ierr=get_pos('START INDEXED_DATA',ifc)

  IF(ierr /= 0) THEN
        write(*,*) '* WARNING ! keyword START INDEXED_DATA not found'
                ELSE
        write(*,*) '    keyword START INDEXED_DATA found, OK'
  ENDIF


! possible blank line after the START INDEXED_DATA
  READ(ifc,*) ligne
  IF(ligne(1:10) /= '          ') THEN
      BACKSPACE ifc
      write(*,*) '    no blank line aftet START INDEXED_DATA, OK'
                                  ELSE
      write(*,*) '    blank line founf after START INDEXED_DATA, ignored'
  ENDIF

! reading all blocks

  DO i=1,b_number
     READ(ifc,'(a)', iostat=ieof) ligne
     IF(ieof /= 0) EXIT
     IF(i == 1) THEN
        !find where the ISO time ends
        ipos1=INDEX(ligne,'Z')
        write(*,*) '    data(index(1)  =',ligne(1:ipos1)
        write(*,*) '    position of "Z" separator in index :', ipos1
     ENDIF

!    save ISO time

     data_index(i)=ligne(1:ipos1)

!    save extented index
     IF (manda_param%INDEX_EXTENSION_LENGTH == 0) THEN
        ind_ext=' '
        status(i)=' '
        phase(i)=-999.99
        ipos2=ipos1+2 ! position a partir de laquelle on vas lire les donnees
     ELSE
        ipos2=ipos1 +2 +manda_param%INDEX_EXTENSION_LENGTH +1 ! car virgule entre les champs
        ind_ext=ligne(ipos1+2:ipos2-2)

!       quand le format d'ecriture a une virgule, il ne marche pas en lecture
!       on remplace la virgure par un "1x"
        iwork=index(index_extension_format,'","')
        if(iwork.gt.2) then
                       work_fmt=index_extension_format(1:iwork-1)//"1x"//index_extension_format(iwork+3:)
                       else
                       work_fmt=index_extension_format
        endif
! 
!       test si il y a une phase à lire

        ipha=index(manda_param%INDEX_EXTENSION_LABEL,'Phase')
        
!       exception !!!

        if(manda_param%MISSION_NAME == 'GEOS' .and. manda_param%EXPERIMENT_MODE == 'POSITIONS') ipha=1
        
        IF (ipha > 0) THEN
            read(ind_ext,work_fmt,iostat=ierr) status(i),phase(i)
                                                    ELSE
            read(ind_ext,work_fmt,iostat=ierr) status(i)
        ENDIF
     ENDIF

     IF(ierr /= 0) THEN
                   write(*,*) '*** error RFF_R_indexed_data'
                   write(*,*) '    error reading file at block # ',i
                   write(*,*) '    FILE_CLASS =',trim(manda_param%FILE_CLASS)
                   write(*,*) '    DATA_TYPE  =',trim(manda_param%DATA_TYPE)
                   write(*,*) '    data_format=',trim(data_format)
                   write(*,*) '    ind_e_label=',trim(manda_param%INDEX_EXTENSION_LABEL)
                   write(*,*) '    error in reading ind_ext:'
                   write(*,*) '    ind_ext    =',trim(ind_ext)
                   write(*,*) '    work_fmt   =',trim(work_fmt)
                   write(*,*) '*** error RFF_R_indexed_data'
                   write(*,*) '    program aborted !!!'
     stop 'lib_rw_rff.o/rff_R_indexed_data : *** ERROR !! Program aborted !'
     ENDIF

! authorized classes

     IF((manda_param%FILE_CLASS == 'WaveForm') .or. &
        (manda_param%FILE_CLASS == 'Spectrogram') .or. &
        (manda_param%FILE_CLASS == 'Image')) THEN
       !save matrix
       IF(manda_param%DATA_TYPE == 'INT') THEN
!         read(ifc,data_format,iostat=ierr) I_data_matrix(:,:,i)
          read(ifc,*,iostat=ierr) I_data_matrix(:,:,i)
       ENDIF

       IF(manda_param%DATA_TYPE == 'FLT') THEN
!         read(ifc,data_format,iostat=ierr) R_data_matrix(:,:,i)
          read(ifc,*,iostat=ierr) R_data_matrix(:,:,i)
       ENDIF

     ENDIF

     IF(manda_param%FILE_CLASS == 'VecTime') THEN
       !save vector
       IF(manda_param%DATA_TYPE == 'INT') THEN
!         read(ligne(ipos2:255),data_format,iostat=ierr) I_data_vector(:,i)
          read(ligne(ipos2:255),*,iostat=ierr) I_data_vector(:,i)
       ENDIF

       IF(manda_param%DATA_TYPE == 'FLT') THEN
!         read(ligne(ipos2:255),data_format,iostat=ierr) R_data_vector(:,i)
          read(ligne(ipos2:255),*,iostat=ierr) R_data_vector(:,i)
       ENDIF

     ENDIF


     IF(ierr /= 0) THEN
                   write(*,*) '*** error RFF_R_indexed_data'
                   write(*,*) '    error reading file at block # ',i
                   write(*,*) '    FILE_CLASS= ',trim(manda_param%FILE_CLASS)
                   write(*,*) '    DATA_TYPE = ',trim(manda_param%DATA_TYPE)
                   write(*,*) '    data_format=',trim(data_format)
                   write(*,*) '    ligne(ipos2:255)=',trim(ligne(ipos2:255))
                   write(*,*) '*** error RFF_R_indexed_data'
                   write(*,*) '    program aborted !!!'
     stop 'lib_rw_rff.o/rff_R_indexed_data : *** ERROR !! Program aborted !'
     ENDIF

  ENDDO

  write(*,*) '    data_index(fin)=',data_index(b_number)

! check index array

  if(manda_param%BLOCK_FIRST_INDEX /= data_index(1) ) then
      write(*,*) '*** error RFF_R_indexed_data'
      write(*,*) '    BLOCK_FIRST_INDEX=',trim(manda_param%BLOCK_FIRST_INDEX)
      write(*,*) '    data_index(1)    =',data_index(1)
      write(*,*)
      write(*,*) '    these two values should be identical !!!'
      write(*,*) '    Check INDEX_FORMAT and DATA_FORMAT in rff file'
      write(*,*) '*** error RFF_R_indexed_data'
      write(*,*) '    program aborted !!!'
      stop 'lib_rw_rff.o/rff_R_indexed_data : *** ERROR !! Program aborted !'
  endif


  if(manda_param%BLOCK_LAST_INDEX /= data_index(b_number) ) then
      write(*,*) '*** error RFF_R_indexed_data'
      write(*,*) '    BLOCK_LAST_INDEX =',trim((manda_param%BLOCK_LAST_INDEX))
      write(*,*) '    data_index(fin)  =',data_index(b_number)
      write(*,*)
      write(*,*) '    these two values should be identical !!!'
      write(*,*) '    Check INDEX_FORMAT & DATA_FORMAT in rff file'
      write(*,*) '*** error RFF_R_indexed_data'
      write(*,*) '    program aborted !!!'
      stop 'lib_rw_rff.o/rff_R_indexed_data : *** ERROR !! Program aborted !'
  endif


  IF(ieof /= 0) THEN
                write(*,*)
                write(*,*) '*** error RFF_R_indexed_data'
                write(*,*) '          EOF encountered at # bloc ',i
                write(*,*) '          last 4 index are:'
                write(*,*) '          ', i-4, trim(data_index(i-4))
                write(*,*) '          ', i-3, trim(data_index(i-3))
                write(*,*) '          ', i-2, trim(data_index(i-2))
                write(*,*) '          ', i-1, trim(data_index(i-1))
                write(*,*) '*** error RFF_R_indexed_data'
                write(*,*) '    program aborted !!!'
     stop 'lib_rw_rff.o/rff_R_indexed_data : *** ERROR !! Program aborted !'
  ENDIF

  write(*,*) 'done !'

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_R_tail(ifc)

!----------------------------------------------------------------------!
! Object : Read  tail of a RFF file
! Author : P. Robert , LPP, 2011 Mar. 08
!----------------------------------------------------------------------!

  integer :: get_pos, ierr


  write(*,*) ('-',i=1,72)
  write(*,*) 'Reading normal tail file...'

  write(*,*) '    Looking for END INDEXED_DATA...'
  ierr=get_pos('END INDEXED_DATA',ifc)
  IF(ierr /= 0) THEN
        write(*,*) '* WARNING ! keyword END INDEXED_DATA not found'
                ELSE
        write(*,*) '    OK'
  ENDIF

  write(*,*) '    Looking for END DATA...'
  ierr=get_pos('END DATA',ifc)
  IF(ierr /= 0) THEN
        write(*,*) '* WARNING ! keyword END DATA not found'
                ELSE
        write(*,*) '    OK'
  ENDIF

  write(*,*) '    Looking for END ROPROC_FORMAT_FILE...'
  ierr=get_pos('END ROPROC_FORMAT_FILE',ifc)
  IF(ierr /= 0) THEN
        write(*,*) '* WARNING ! keyword END ROPROC_FORMAT_FILE not found'
                ELSE
        write(*,*) '    OK'
  ENDIF

  close(ifc)
  write(*,*) 'Done ! closing unit file ',ifc

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_W_file(ifc,file)

!----------------------------------------------------------------------!
! Object : Write rff file, metadata & data, WaveForm or VecTime
! Author : P. Robert , LPP, 2011 Mar. 09
!----------------------------------------------------------------------!

  use rff_param_def
  use rff_data_def

  character(len=*) :: file

  call rff_W_metadata(ifc,file)
  call rff_W_const_data(ifc)
  call rff_W_indexed_data(ifc)
  call rff_W_tail(ifc)

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_W_manda_param(ifc)

!----------------------------------------------------------------------!
! Object : Write mandatory parameters
! Author : P. Robert , LPP, 2011 Jan. 23
!----------------------------------------------------------------------!

  use rff_param_def

  character(len=80)       :: fmt1,fmt2,fmt3,fmt4,fmt5,fmt6
  character(len=par_len) :: work

! -----------------------------------

  fmt1='(80a)'
  fmt2='("PAR ",a25," (STR): ",a)'
  fmt3='("PAR ",a25," (INT): ",i1,1x,i5)'
  fmt4='("PAR ",a25," (INT): ",a)'
  fmt5='(i4.4,"-",i2.2,"-",i2.2,"T",i2.2,":",i2.2,":",i2.2,".000Z")'
  fmt6='("PAR ",a25," (INT): ",i5,1x,i5)'

  call rff_get_current_date(work)
  manda_param%FILE_CREATION_DATE=TRIM(work)


  write(ifc,fmt1) '#',('-',i=1,79)
  write(ifc,fmt1) 'START MANDATORY_PARAMETERS'
  write(ifc,fmt1)

  write(ifc,fmt2) 'FILE_NAME                ', TRIM(manda_param%FILE_NAME)
  write(ifc,fmt2) 'FILE_CLASS               ', TRIM(manda_param%FILE_CLASS)
  write(ifc,fmt2) 'FILE_FORMAT_VERSION      ', TRIM(manda_param%FILE_FORMAT_VERSION)
  write(ifc,fmt2) 'FILE_CREATION_DATE       ', TRIM(manda_param%FILE_CREATION_DATE)
  write(ifc,fmt1)
  write(ifc,fmt2) 'MISSION_NAME             ', TRIM(manda_param%MISSION_NAME)
  write(ifc,fmt2) 'OBSERVATORY_NAME         ', TRIM(manda_param%OBSERVATORY_NAME)
  write(ifc,fmt3) 'OBSERVATORY_NUMBER       ',      manda_param%OBSERVATORY_NUMBER
  write(ifc,fmt2) 'EXPERIMENT_NAME          ', TRIM(manda_param%EXPERIMENT_NAME)
  write(ifc,fmt2) 'EXPERIMENT_MODE          ', TRIM(manda_param%EXPERIMENT_MODE)
  write(ifc,fmt2) 'INSTRUMENT_TYPE          ', TRIM(manda_param%INSTRUMENT_TYPE)
  write(ifc,fmt2) 'MEASUREMENT_TYPE         ', TRIM(manda_param%MEASUREMENT_TYPE)
  write(ifc,fmt1)
  write(ifc,fmt2) 'INDEX_LABEL              ', TRIM(manda_param%INDEX_LABEL)
  write(ifc,fmt2) 'INDEX_TYPE               ', TRIM(manda_param%INDEX_TYPE)
  write(ifc,fmt2) 'INDEX_UNITS              ', TRIM(manda_param%INDEX_UNITS)
  write(ifc,fmt2) 'INDEX_FORMAT             ', TRIM(manda_param%INDEX_FORMAT)
  write(ifc,fmt2) 'INDEX_FORM               ', TRIM(manda_param%INDEX_FORM)
  write(ifc,fmt3) 'INDEX_DIMENSION          ',      manda_param%INDEX_DIMENSION
  write(ifc,fmt2) 'INDEX_PROPERTIES         ', TRIM(manda_param%INDEX_PROPERTIES)
  write(ifc,fmt1)
  write(ifc,fmt2) 'INDEX_EXTENSION_LABEL    ', TRIM(manda_param%INDEX_EXTENSION_LABEL)
  write(ifc,fmt2) 'INDEX_EXTENSION_TYPE     ', TRIM(manda_param%INDEX_EXTENSION_TYPE)
  write(ifc,fmt2) 'INDEX_EXTENSION_UNITS    ', TRIM(manda_param%INDEX_EXTENSION_UNITS)
  write(ifc,fmt2) 'INDEX_EXTENSION_FORMAT   ', TRIM(manda_param%INDEX_EXTENSION_FORMAT)

  write(work,*) manda_param%INDEX_EXTENSION_LENGTH
  write(ifc,fmt4) 'INDEX_EXTENSION_LENGTH   ', TRIM(ADJUSTL(work))
  write(ifc,fmt1)
  write(ifc,fmt2) 'DATA_LABEL               ', TRIM(manda_param%DATA_LABEL)
  write(ifc,fmt2) 'DATA_TYPE                ', TRIM(manda_param%DATA_TYPE)
  write(ifc,fmt2) 'DATA_UNITS               ', TRIM(manda_param%DATA_UNITS)
  write(ifc,fmt2) 'DATA_FORMAT              ', TRIM(manda_param%DATA_FORMAT)
  write(ifc,fmt2) 'DATA_FORM                ', TRIM(manda_param%DATA_FORM)

  IF(manda_param%DATA_FORM == 'Vector') THEN
  write(ifc,fmt3) 'DATA_DIMENSION           ', manda_param%DATA_DIMENSION(1)
                                        ELSE
  write(ifc,fmt6) 'DATA_DIMENSION           ', manda_param%DATA_DIMENSION
  ENDIF

  write(ifc,fmt2) 'DATA_REPRESENTATION      ', TRIM(manda_param%DATA_REPRESENTATION)
  write(ifc,fmt2) 'DATA_COORDINATE_SYSTEM   ', TRIM(manda_param%DATA_COORDINATE_SYSTEM)
  write(ifc,fmt2) 'DATA_FILL_VALUE          ', TRIM(manda_param%DATA_FILL_VALUE)
  write(ifc,fmt1)

  write(work,*) manda_param%BLOCK_NUMBER
  write(ifc,fmt4) 'BLOCK_NUMBER             ', TRIM(ADJUSTL(work))
  write(ifc,fmt2) 'BLOCK_FIRST_INDEX        ', TRIM(manda_param%BLOCK_FIRST_INDEX)
  write(ifc,fmt2) 'BLOCK_LAST_INDEX         ', TRIM(manda_param%BLOCK_LAST_INDEX)

  write(ifc,fmt1)
  write(ifc,fmt1) 'END MANDATORY_PARAMETERS'

  RETURN
  END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_W_optio_param(ifc)

!----------------------------------------------------------------------!
! Object : Write optional parameters
! Author : P. Robert , LPP, 2011 Jan. 23
!----------------------------------------------------------------------!

  use rff_param_def

  character(len=32)     :: fmt1,fmt2,fmt3,fmt4
  character(len=80)     :: work

! -----------------------------------

  fmt1='(80a)'
  fmt2='("PAR ",a25," (STR): ",a)'
  fmt3='("PAR ",a25," (TXT): ",a)'
  fmt4='("PAR ",a25," (DBL): ",e17.10)'

  write(ifc,fmt1) '#',('-',i=1,79)
  write(ifc,fmt1) 'START OPTIONAL_PARAMETERS'
  write(ifc,fmt1)

  if(optio_param%TIME_RESOLUTION  < 1.D30) &
       write(ifc,fmt4) 'TIME_RESOLUTION          ', optio_param%TIME_RESOLUTION

  if(manda_param%FILE_CLASS == 'Spectrogram') then
  if(optio_param%FREQUENCY_RESOLUTION > 0.) &
       write(ifc,fmt4) 'FREQUENCY_RESOLUTION     ', optio_param%FREQUENCY_RESOLUTION
  endif

  if(optio_param%TIME_SPAN_FROM     /= 'undefined') &
     write(ifc,fmt2) 'TIME_SPAN_FROM           ', TRIM(optio_param%TIME_SPAN_FROM)
  if(optio_param%TIME_SPAN_TO       /= 'undefined') &
     write(ifc,fmt2) 'TIME_SPAN_TO             ', TRIM(optio_param%TIME_SPAN_TO)
  if(optio_param%TITLE              /= 'undefined') &
     write(ifc,fmt2) 'TITLE                    ', TRIM(optio_param%TITLE)
  if(optio_param%SUB_TITLE          /= 'undefined') &
     write(ifc,fmt2) 'SUB_TITLE                ', TRIM(optio_param%SUB_TITLE)
  if(optio_param%DISCIPLINE_NAME    /= 'undefined') &
     write(ifc,fmt2) 'DISCIPLINE_NAME          ', TRIM(optio_param%DISCIPLINE_NAME)
  if(optio_param%EXPERIMENT_PI_NAME /= 'undefined') &
     write(ifc,fmt2) 'EXPERIMENT_PI_NAME       ', TRIM(optio_param%EXPERIMENT_PI_NAME)
  if(optio_param%EXPERIMENT_PI_MAIL /= 'undefined') &
     write(ifc,fmt2) 'EXPERIMENT_PI_MAIL       ', TRIM(optio_param%EXPERIMENT_PI_MAIL)

  write(ifc,fmt1)
  write(ifc,fmt3) 'MISSION_DESCRIPTION      ', '{'
  IF(optio_param%MI_DE == 0) then
     optio_param%MI_DE=1
     write(ifc,fmt1) 'None.}'
                             else
     DO i=1, optio_param%MI_DE -1
         write(ifc,fmt1) TRIM(optio_param%MISSION_DESCRIPTION(i))
     ENDDO
     last=len_trim(optio_param%MISSION_DESCRIPTION(optio_param%MI_DE))
     work=optio_param%MISSION_DESCRIPTION(optio_param%MI_DE)
     if(work(last:last) == '}') then
         write(ifc,fmt1) TRIM(optio_param%MISSION_DESCRIPTION(optio_param%MI_DE))
                                else
         write(ifc,fmt1) TRIM(optio_param%MISSION_DESCRIPTION(optio_param%MI_DE)),'}'
     endif
  ENDIF

  write(ifc,fmt1)
  write(ifc,fmt3) 'EXPERIMENT_DESCRIPTION   ', '{'
  IF(optio_param%EX_DE == 0) then
     optio_param%EX_DE=1
     write(ifc,fmt1) 'None.}'
                             else
     DO i=1, optio_param%EX_DE -1
         write(ifc,fmt1) TRIM(optio_param%EXPERIMENT_DESCRIPTION(i))
     ENDDO
     last=len_trim(optio_param%EXPERIMENT_DESCRIPTION(optio_param%EX_DE))
     work=optio_param%EXPERIMENT_DESCRIPTION(optio_param%EX_DE)
     if(work(last:last) == '}') then
         write(ifc,fmt1) TRIM(optio_param%EXPERIMENT_DESCRIPTION(optio_param%EX_DE))
                                else
         write(ifc,fmt1) TRIM(optio_param%EXPERIMENT_DESCRIPTION(optio_param%EX_DE)),'}'
     endif
  ENDIF

  write(ifc,fmt1)
  write(ifc,fmt3) 'INDEX_DESCRIPTION        ', '{'
  IF(optio_param%IN_DE == 0) then
     optio_param%IN_DE=1
     write(ifc,fmt1) 'None.}'
                             else
     DO i=1, optio_param%IN_DE -1
         write(ifc,fmt1) TRIM(optio_param%INDEX_DESCRIPTION(i))
     ENDDO
     last=len_trim(optio_param%INDEX_DESCRIPTION(optio_param%IN_DE))
     work=optio_param%INDEX_DESCRIPTION(optio_param%IN_DE)
     if(work(last:last) == '}') then
         write(ifc,fmt1) TRIM(optio_param%INDEX_DESCRIPTION(optio_param%IN_DE))
                                else
         write(ifc,fmt1) TRIM(optio_param%INDEX_DESCRIPTION(optio_param%IN_DE)),'}'
     endif
  ENDIF

  write(ifc,fmt1)
  write(ifc,fmt3) 'INDEX_EXTENSION_DESCRIP  ', '{'
  IF(optio_param%IE_DE == 0) then
     optio_param%IE_DE=1
     write(ifc,fmt1) 'None.}'
                             else
     DO i=1, optio_param%IE_DE -1
         write(ifc,fmt1) TRIM(optio_param%INDEX_EXTENSION_DESCRIP(i))
     ENDDO
     last=len_trim(optio_param%INDEX_EXTENSION_DESCRIP(optio_param%IE_DE))
     work=optio_param%INDEX_EXTENSION_DESCRIP(optio_param%IE_DE)
     if(work(last:last) == '}') then
         write(ifc,fmt1) TRIM(optio_param%INDEX_EXTENSION_DESCRIP(optio_param%IE_DE))
                                else
         write(ifc,fmt1) TRIM(optio_param%INDEX_EXTENSION_DESCRIP(optio_param%IE_DE)),'}'
     endif
  ENDIF

  write(ifc,fmt1)
  write(ifc,fmt3) 'DATA_DESCRIPTION         ', '{'
  IF(optio_param%DA_DE == 0) then
     optio_param%DA_DE=1
     write(ifc,fmt1) 'None.}'
                             else
     DO i=1, optio_param%DA_DE -1
         write(ifc,fmt1) TRIM(optio_param%DATA_DESCRIPTION(i))
     ENDDO
     last=len_trim(optio_param%DATA_DESCRIPTION(optio_param%DA_DE))
     work=optio_param%DATA_DESCRIPTION(optio_param%DA_DE)
     if(work(last:last) == '}') then
         write(ifc,fmt1) TRIM(optio_param%DATA_DESCRIPTION(optio_param%DA_DE))
                                else
         write(ifc,fmt1) TRIM(optio_param%DATA_DESCRIPTION(optio_param%DA_DE)),'}'
     endif
  ENDIF

  write(ifc,fmt1)
  write(ifc,fmt3) 'BLOCK_DESCRIPTION        ', '{'
  IF(optio_param%BL_DE == 0) then
     optio_param%BL_DE=1
     write(ifc,fmt1) 'None.}'
                             else
     DO i=1, optio_param%BL_DE -1
         write(ifc,fmt1) TRIM(optio_param%BLOCK_DESCRIPTION(i))
     ENDDO
     last=len_trim(optio_param%BLOCK_DESCRIPTION(optio_param%BL_DE))
     work=optio_param%BLOCK_DESCRIPTION(optio_param%BL_DE)
     if(work(last:last) == '}') then
         write(ifc,fmt1) TRIM(optio_param%BLOCK_DESCRIPTION(optio_param%BL_DE))
                                else
         write(ifc,fmt1) TRIM(optio_param%BLOCK_DESCRIPTION(optio_param%BL_DE)),'}'
     endif
  ENDIF

  write(ifc,fmt1)
  write(ifc,fmt3) 'FILE_ANOMALIES           ', '{'
  IF(optio_param%FI_AN == 0) then
     optio_param%FI_AN=1
     write(ifc,fmt1) 'None.}'
                             else
     DO i=1, optio_param%FI_AN -1
         write(ifc,fmt1) TRIM(optio_param%FILE_ANOMALIES(i))
     ENDDO
     last=len_trim(optio_param%FILE_ANOMALIES(optio_param%FI_AN))
     work=optio_param%FILE_ANOMALIES(optio_param%FI_AN)
     if(work(last:last) == '}') then
         write(ifc,fmt1) TRIM(optio_param%FILE_ANOMALIES(optio_param%FI_AN))
                                else
         write(ifc,fmt1) TRIM(optio_param%FILE_ANOMALIES(optio_param%FI_AN)),'}'
     endif
  ENDIF


  write(ifc,fmt1)
  write(ifc,fmt3) 'HISTORY                  ', '{'
  IF(optio_param%BL_HI == 0) then
     optio_param%BL_HI=1
     write(ifc,fmt1) 'None.}'
                             else
     DO i=1, optio_param%BL_HI -1
         write(ifc,fmt1) TRIM(optio_param%HISTORY(i))
     ENDDO
     last=len_trim(optio_param%HISTORY(optio_param%BL_HI))
     work=optio_param%HISTORY(optio_param%BL_HI)
     if(work(last:last) == '}') then
         write(ifc,fmt1) TRIM(optio_param%HISTORY(optio_param%BL_HI))
                                else
         write(ifc,fmt1) TRIM(optio_param%HISTORY(optio_param%BL_HI)),'}'
     endif
  ENDIF

  write(ifc,fmt1)
  write(ifc,fmt1) 'END OPTIONAL_PARAMETERS'

  RETURN
  END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_W_const_data(ifc)

!----------------------------------------------------------------------!
! Object : Write constant data
! Author : P. Robert , LPP, 2011 Jan. 23
!----------------------------------------------------------------------!

  use rff_param_def

  character(len=10) uSR,uVR,uTM,uMI,uCT,uSP,uSG,uMC,uEA,uTS

! -----------------------------------

100 format('VAR ',a25,' (STR), u=',a10,': ',a)
101 format('VAR ',a25,' (DBL), u=',a10,': ',e15.7)
102 format('VAR ',a25,' (FLT), u=',a10,': ',f6.2)
103 format('VAR ',a25,' (FLT), u=',a10,': ',f7.4)
104 format('VAR ',a25,' (INT), u=',a10,': ',i5)
105 format(80a)

  uSR='Hz'
  uVR='Volts'
  uTM='TM_counts'
  uMI='None'
  uCT='ISO_TIME'
  uSP='second'
  uSG='degree'
  uMC='mm' ! a revoir
  uEA='degree'
  uTS='ISO_TIME'

  write(ifc,105) 'END METADATA'
  write(ifc,105) '#',('-',i=1,79)
  write(ifc,105)
  write(ifc,105) 'START DATA'
  write(ifc,105)

  write(ifc,105) '#',('-',i=1,79)
  write(ifc,105) 'START CONSTANT_DATA'
  write(ifc,105)

  if(const_data%TED_VERSION     /= 'undefined') &
     write(ifc,100) 'TED_VERSION              ', uMI, TRIM(AdjustL(const_data%TED_VERSION))
  if(const_data%TCOR_OPTION     /= 'undefined') &
     write(ifc,100) 'TCOR_OPTION              ', uMI, TRIM(AdjustL(const_data%TCOR_OPTION))


  if(const_data%SAMPLE_RATE     < 1.d30 ) &
     write(ifc,101) 'SAMPLE_RATE              ', uSR, const_data%SAMPLE_RATE
  if(const_data%VOLT_RANGE_MIN  < 1.e30 ) &
     write(ifc,102) 'VOLT_RANGE_MIN           ', uVR, const_data%VOLT_RANGE_MIN
  if(const_data%VOLT_RANGE_MAX  < 1.e30 ) &
     write(ifc,102) 'VOLT_RANGE_MAX           ', uVR, const_data%VOLT_RANGE_MAX
  if(const_data%TM_RANGE_MIN    < 999999999 ) &
     write(ifc,104) 'TM_RANGE_MIN             ', uTM, const_data%TM_RANGE_MIN
  if(const_data%TM_RANGE_MAX    < 999999999 ) &
     write(ifc,104) 'TM_RANGE_MAX             ', uTM, const_data%TM_RANGE_MAX



  if(const_data%MISALIGNMENT_MATRIX_L1_C1     < 1.e30 ) &
     write(ifc,105)
  if(const_data%MISALIGNMENT_MATRIX_L1_C1     < 1.e30 ) &
     write(ifc,103) 'MISALIGNMENT_MATRIX_L1_C1', uMI, const_data%MISALIGNMENT_MATRIX_L1_C1
  if(const_data%MISALIGNMENT_MATRIX_L1_C2     < 1.e30 ) &
     write(ifc,103) 'MISALIGNMENT_MATRIX_L1_C2', uMI, const_data%MISALIGNMENT_MATRIX_L1_C2
  if(const_data%MISALIGNMENT_MATRIX_L1_C3     < 1.e30 ) &
     write(ifc,103) 'MISALIGNMENT_MATRIX_L1_C3', uMI, const_data%MISALIGNMENT_MATRIX_L1_C3
  if(const_data%MISALIGNMENT_MATRIX_L2_C1     < 1.e30 ) &
     write(ifc,103) 'MISALIGNMENT_MATRIX_L2_C1', uMI, const_data%MISALIGNMENT_MATRIX_L2_C1
  if(const_data%MISALIGNMENT_MATRIX_L2_C2     < 1.e30 ) &
     write(ifc,103) 'MISALIGNMENT_MATRIX_L2_C2', uMI, const_data%MISALIGNMENT_MATRIX_L2_C2
  if(const_data%MISALIGNMENT_MATRIX_L2_C3     < 1.e30 ) &
     write(ifc,103) 'MISALIGNMENT_MATRIX_L2_C3', uMI, const_data%MISALIGNMENT_MATRIX_L2_C3
  if(const_data%MISALIGNMENT_MATRIX_L3_C1     < 1.e30 ) &
     write(ifc,103) 'MISALIGNMENT_MATRIX_L3_C1', uMI, const_data%MISALIGNMENT_MATRIX_L3_C1
  if(const_data%MISALIGNMENT_MATRIX_L3_C2     < 1.e30 ) &
     write(ifc,103) 'MISALIGNMENT_MATRIX_L3_C2', uMI, const_data%MISALIGNMENT_MATRIX_L3_C2
  if(const_data%MISALIGNMENT_MATRIX_L3_C3     < 1.e30 ) &
     write(ifc,103) 'MISALIGNMENT_MATRIX_L3_C3', uMI, const_data%MISALIGNMENT_MATRIX_L3_C3
  if(const_data%MISALIGNMENT_MATRIX_L3_C3     < 1.e30 ) &
     write(ifc,105)

  if(const_data%CONSTANT_TIME_MEASUREMENT     /= 'undefined') &
     write(ifc,100) 'CONSTANT_TIME_MEASUREMENT', uCT, TRIM(AdjustL(const_data%CONSTANT_TIME_MEASUREMENT))

  if(const_data%SPIN_PERIOD              < 1.d30 ) &
     write(ifc,101) 'SPIN_PERIOD              ', uSR, const_data%SPIN_PERIOD
  if(const_data%SPIN_GEI_RIGHT_ASCENSION < 1.e30) &
     write(ifc,102) 'SPIN_GEI_RIGHT_ASCENSION ', uSG, const_data%SPIN_GEI_RIGHT_ASCENSION
  if(const_data%SPIN_GEI_DECLINATION     < 1.e30) &
     write(ifc,102) 'SPIN_GEI_DECLINATION     ', usG, const_data%SPIN_GEI_DECLINATION
  if(const_data%MASS_CENTER_X            < 1.e30) &
     write(ifc,103) 'MASS_CENTER_X            ', uMC, const_data%MASS_CENTER_X
  if(const_data%MASS_CENTER_Y            < 1.e30) &
     write(ifc,103) 'MASS_CENTER_Y            ', uMC, const_data%MASS_CENTER_Y
  if(const_data%MASS_CENTER_Z            < 1.e30) &
     write(ifc,103) 'MASS_CENTER_Z            ', uMC, const_data%MASS_CENTER_Z
  if(const_data%EULER_ANGLE_FIRST        < 1.e30) &
     write(ifc,102) 'EULER_ANGLE_FIRST        ', uEA, const_data%EULER_ANGLE_FIRST
  if(const_data%EULER_ANGLE_SECOND       < 1.e30) &
     write(ifc,102) 'EULER_ANGLE_SECOND       ', uEA, const_data%EULER_ANGLE_SECOND


  if (const_data%FREQUENCY_FILTER_MIN < 1.e30 .or. &
      const_data%FREQUENCY_FILTER_MAX < 1.e30 .or. &
      const_data%FREQUENCY_CUT_OFF    < 1.e30 .or. &
      const_data%FREQUENCY_DETREND    < 1.e30 .or. &
      const_data%CALIB_KERNEL_SIZE    < 999999999  .or. &
      const_data%CALIB_SHIFT_SIZE     < 999999999 ) then
     write(ifc,105)
     write(ifc,102) 'FREQUENCY_FILTER_MIN     ', uSR, const_data%FREQUENCY_FILTER_MIN
     write(ifc,102) 'FREQUENCY_FILTER_MAX     ', uSR, const_data%FREQUENCY_FILTER_MAX
     write(ifc,102) 'FREQUENCY_CUT_OFF        ', uSR, const_data%FREQUENCY_CUT_OFF
     write(ifc,102) 'FREQUENCY_DETREND        ', uSR, const_data%FREQUENCY_DETREND
     write(ifc,104) 'CALIB_KERNEL_SIZE        ', uMI, const_data%CALIB_KERNEL_SIZE
     write(ifc,104) 'CALIB_SHIFT_SIZE         ', uMI, const_data%CALIB_SHIFT_SIZE
  endif

  if (const_data%SPECTRA_KERNEL_SIZE < 999999999  .or. &
      const_data%SPECTRA_SHIFT_SIZE  < 999999999) then
     write(ifc,105)
     write(ifc,104) 'SPECTRA_KERNEL_SIZE      ',uMI,const_data%SPECTRA_KERNEL_SIZE
     write(ifc,104) 'SPECTRA_SHIFT_SIZE       ',uMI,const_data%SPECTRA_SHIFT_SIZE
     write(ifc,100) 'SPECTRA_WEIGHTING        ',uMI,const_data%SPECTRA_WEIGHTING
  endif

  write(ifc,105)
  write(ifc,105) 'END CONSTANT_DATA'

  RETURN
  END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_W_metadata(ifc,file)

!----------------------------------------------------------------------!
! Object : Open RFF file, Write header, metadata & constant data
! Author : P. Robert , LPP, 2011 Mar. 08
!----------------------------------------------------------------------!

  use rff_param_def
  use rff_data_def

  implicit none

  character(len=*)   :: file
  character(len=255) :: Fbasename
  integer            :: ifc,i, err

! -----------------------------------


! 1) open file and write header
!    --------------------------

  write(*,*)
  write(*,*) 'Opening file ',TRIM(file)
  close(ifc)
  OPEN(ifc,file=file, form='formatted',iostat=err)

  if(err /= 0) then
               write(*,*) '*** unable to open file :',TRIM(file)
               write(*,*) '    in subroutine rff_W_metadata'
               write(*,*) '    program aborted !!!'
       stop 'lib_rw_rff.o/rff_W_metadata     : *** ERROR !! Program aborted !'
  endif

  write(*,*)
  write(*,*) 'Writing RFF file header'

  manda_param%file_name=TRIM(Fbasename(file))

1 format(80a)

  write(ifc,1) 'START ROPROC_FORMAT_FILE'
  write(ifc,1)
  write(ifc,1) '#',('=',i=1,79)
  write(ifc,1) '# The Roproc_Format_File is a general format mainly used for data from'
  write(ifc,1) '# spatial missions (magnetometer, waveform units, S/C trajectory ...).'
  write(ifc,1) '# For readability, all lines starting with character "#" are comments,'
  write(ifc,1) '# while empty or blank lines are ignored. Others lines are keywords,'
  write(ifc,1) '# parameters variables or data.'
  write(ifc,1) '# This file contents Metadata (description of the data), Constant Data'
  write(ifc,1) '# (one value per file, or a few, but in limited number) and indexed data'
  write(ifc,1) '# (data versus time or other INDEX).'
  write(ifc,1) '# Any group of data begins with a "START" keyword, and stops by a "END".'
  write(ifc,1) '# Metadata are given as parameters series (one value per parameter).'
  write(ifc,1) '# Data are described by Metadata parameters.'
  write(ifc,1) '# Examples of different files are given in the document:'
  write(ifc,1) '# "The Roproc Format File, a dedicated file format for vectorial data'
  write(ifc,1) '# processing"'
  write(ifc,1) '# Author: P. Robert, CNRS/LPP (formerly CETP)'
  write(ifc,1) '# V 1.0, 1996-2000    (for archive of old mission data)'
  write(ifc,1) '# V 1.1, October 2001 (for CLUSTER/STAFF-SC NBR & HBR wave data)'
  write(ifc,1) '# V 1.2, January 2002 (for GEOS wave data)'
  write(ifc,1) '# V 1.3, August  2003 (for CUSP and any kind of wave data)'
  write(ifc,1) '# V 1.4, January 2003 (for general titles management)'
  write(ifc,1) '# V 2.0, March   2004 (for compatibility with Roproc Vector format)'
  write(ifc,1) '# V 2.1, May     2004 (to be coherent with Cluster Exchange Format)'
  write(ifc,1) '# V 2.2, March   2007 (some useful upgrades)'
  write(ifc,1) '# Any comment or suggestion: Patrick.Robert@lpp.polytechnique.fr'
  write(ifc,1) '#',('=',i=1,79)
  write(ifc,1)
  write(ifc,1) 'START METADATA'
  write(ifc,1)

! 2) write metadata
!    --------------

  write(*,*) 'Writing mandatory parameters...'

  call rff_W_manda_param(ifc)

  write(*,*) 'Writing optional  parameters...'

  call rff_W_optio_param(ifc)

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_W_indexed_data(ifc)

!--------------------------------------------------------------------0--
! Object : Write indexed data for waveform and vectime RFF files
! Author : P. Robert , LPP, 2011 March 09
!--------------------------------------------------------------------0--

  use rff_param_def
  use rff_data_def

  character(len=255) :: fmt_block_header, fmt_block_data
  character(len= 32) :: fmt_tot
  character(len=255) :: ind_nor, ind_ext, datavec
  character(len=par_len) :: fortmp

! -----------------------------

  IF(manda_param%INDEX_EXTENSION_FORMAT /= 'none') THEN
     fmt_block_header= '('//TRIM(manda_param%INDEX_FORMAT)//',1x,'// &
                  TRIM(manda_param%INDEX_EXTENSION_FORMAT)//')'
                                                   ELSE
     fmt_block_header= '('//TRIM(manda_param%INDEX_FORMAT)//')'
  ENDIF

  fmt_block_data=manda_param%DATA_FORMAT

  write(*,*) ('-',i=1,72)
  write(*,*) 'Writing indexed data, nb. blocks=',manda_param%BLOCK_NUMBER
  write(*,*) '    fmt_block_header=', TRIM(fmt_block_header)
  write(*,*) '    fmt_block_data  =', TRIM(fmt_block_data  )
  write(*,*) 'Please wait...'
  
  write(*,*)
  if(manda_param%BLOCK_NUMBER <1 ) STOP '***  rff_W_indexed_data : no block to write'

1 format(80a)

  write(ifc,1) '#',('-',i=1,79)
  write(ifc,1)   'START INDEXED_DATA'

  IF(manda_param%FILE_CLASS == 'WaveForm') THEN
     DO i=1,manda_param%BLOCK_NUMBER
          if(allocated(status) .and. allocated(phase)) then
             write(ifc,fmt_block_header) data_index(i),status(i),phase(i)
                                                        else
             if(allocated(status)) then
                write(ifc,fmt_block_header) data_index(i),status(i)
                                   else
                write(ifc,fmt_block_header) data_index(i)
             endif
          endif
          IF(manda_param%DATA_TYPE == 'INT') THEN
             write(ifc,fmt_block_data)   I_data_matrix(:,:,i)
          ENDIF
          IF(manda_param%DATA_TYPE == 'FLT') THEN
             write(ifc,fmt_block_data)   R_data_matrix(:,:,i)
          ENDIF
     ENDDO
  ENDIF

  IF(manda_param%FILE_CLASS == 'Spectrogram') THEN
     DO i=1,manda_param%BLOCK_NUMBER
        write(ifc,fmt_block_header) data_index(i)
          IF(manda_param%DATA_TYPE == 'INT') THEN
             write(ifc,fmt_block_data)   I_data_matrix(:,:,i)
          ENDIF
          IF(manda_param%DATA_TYPE == 'FLT') THEN
             write(ifc,fmt_block_data)   R_data_matrix(:,:,i)
          ENDIF
     ENDDO
  ENDIF

! ----------------------------------------------------------------
  IF(manda_param%FILE_CLASS == 'VecTime') THEN

     DO i=1,manda_param%BLOCK_NUMBER

        IF(manda_param%DATA_TYPE == 'INT') THEN
           write(datavec,manda_param%DATA_FORMAT) I_data_vector(:,i)
        ENDIF

        IF(manda_param%DATA_TYPE == 'FLT') THEN
           write(datavec,manda_param%DATA_FORMAT) R_data_vector(:,i)
        ENDIF

        write(ind_nor,manda_param%INDEX_FORMAT) data_index(i)

        IF(manda_param%INDEX_EXTENSION_LENGTH /= 0) THEN
       !    on met toujours des "," a l'ecriture comme separateur si on a un extended index     
            fmt_tot='(a,",",a,",",a)'
                    
            call test_pha(fortmp,ipha)
        
            if(ipha > 0) then
                 write(ind_ext,fortmp) status(i),phase(i)
                         else
                 write(ind_ext,fortmp) status(i)
            endif
                                                     ELSE
        !   pas d'index etendu              
            ind_ext=' '
            fmt_tot='(a,a,a)'
        ENDIF

        write(ifc,fmt_tot) TRIM(ind_nor), TRIM(ind_ext), TRIM(datavec)
     END DO
  ENDIF
! ----------------------------------------------------------------

  write(*,*) 'Done !'

  return
  end
  
!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine test_pha(fortmp,ipha)
  
  use rff_param_def
  use rff_data_def
  
  character(len=par_len) :: fortmp
  
! test si il y a une phase à ecrire
  ipha=index(manda_param%INDEX_EXTENSION_LABEL,'Phase')
        
! exception !!!
  if(manda_param%MISSION_NAME == 'GEOS' .and. manda_param%EXPERIMENT_MODE == 'POSITIONS') ipha=1
        
  IF (ipha > 0) THEN
      ! on a une phase à écrire
      
      ! test si on a une virgule dans le format d'ecriture, separant la phase
        ivir=index(manda_param%INDEX_EXTENSION_FORMAT,'","')
        if(ivir == 0) ivir=index(manda_param%INDEX_EXTENSION_FORMAT,"','")
        
        if(ivir > 0) then
              fortmp=manda_param%INDEX_EXTENSION_FORMAT
                     else
            ! si pas de virgule, il faut absolument qu'on ai un blanc
              iblc=index(manda_param%INDEX_EXTENSION_FORMAT,'1x')
              if(iblc == 0) iblc=index(manda_param%INDEX_EXTENSION_FORMAT,'1X')
              
              if(iblc > 0) then
                     fortmp=manda_param%INDEX_EXTENSION_FORMAT(1:iblc-1)//"','"//manda_param%INDEX_EXTENSION_FORMAT(iblc+2:)
                           else
                    ! on laisse comme ça...
                     fortmp=manda_param%INDEX_EXTENSION_FORMAT
              endif
         endif
               ELSE
       fortmp=manda_param%INDEX_EXTENSION_FORMAT    
  ENDIF                  

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  subroutine rff_W_tail(ifc)

!----------------------------------------------------------------------!
! Object : Write tail of a RFF file
! Author : P. Robert , LPP, 2011 March 08
!----------------------------------------------------------------------!


  write(*,*) 'Writing normal tail file...'

1 format(80a)

  write(ifc,1) 'END INDEXED_DATA'
  write(ifc,1) 'END DATA'
  write(ifc,1) 'END ROPROC_FORMAT_FILE'

  close(ifc)
  write(*,*) 'Done ! closing unit file ',ifc

  return
  end
 !
!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
!
  subroutine rff_format_W_to_R(format_W,format_R)

  character(len=*) :: format_W,format_R
  character(len=255) work

!----------------------------------------------------------------------!
! Object : convert a Write format in a Read format
! Author : P. Robert , LPP, 2011 October 15
!----------------------------------------------------------------------!

  work=format_W
  ig1=1

  do while (ig1 /= 0)
      ig1=index(work, ',",",')
      if (ig1 /= 0) then
                    work(ig1:ig1+4)=', 1x,'
                    else
                    cycle
      endif
  enddo

  format_R=TRIM(work)

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
!
  subroutine rff_format_R_to_W(format_R,format_W)

  character(len=*) :: format_R,format_W
  character(len=255) work

!----------------------------------------------------------------------!
! Object : convert a Read format in a Write format
! Author : P. Robert , LPP, 2012 Aug. 30
!----------------------------------------------------------------------!

  work=format_R
  ig1=1

  do while (ig1 /= 0)
      ig1=index(work, ',1x')
      if (ig1 /= 0) then
                    work(ig1+3:255)=' '//work(ig1+3:254)
                    work(ig1:ig1+4)=',",",'
                    else
                    cycle
      endif
  enddo

  format_W=TRIM(work)

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  function get_pos(param,ifc)

!----------------------------------------------------------------------!
! Object : Read rff file until searched Character parameter
!         return 0 if successfull
! Author : P. Robert , LPP, 2011 Jan. 23
!----------------------------------------------------------------------!

  integer            :: get_pos
  character(len=*)   :: param
  character(len=255) :: com

  integer eof

! -----------------------------------

  com='xxx'
  ll=LEN_TRIM(param)
  get_pos=1

  DO WHILE (com(1:ll) /= param(1:ll))
     READ(ifc,'(a)',iostat=eof) com
     IF(com(1:18) == 'START INDEXED_DATA') EXIT
     IF(eof /= 0) EXIT
  ENDDO

  IF(com(1:ll) == param(1:ll)) THEN
                               get_pos=0
                               RETURN
  ENDIF

  IF(com(1:18) == 'START INDEXED_DATA') THEN
               write(*,*) '*** Keyword ',param(1:ll),' not found'
               write(*,*) '    file rewinded for next use'
               get_pos=1
               REWIND ifc
               RETURN
  ENDIF

  IF(eof /= 0) THEN
               write(*,*) '*** Keyword ',param(1:ll),' not found'
               write(*,*) '*** Keyword  START INDEXED_DATA not found'
               write(*,*) '*** RFF file not correct'
               write(*,*) '*** eof reached on RFF file'
               write(*,*) '    Program aborted !!!'
       stop 'lib_rw_rff.o/get_pos            : *** ERROR !! Program aborted !'
  ENDIF


  RETURN
  END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  function get_paramC(param,ifc)

!----------------------------------------------------------------------!
! Object : Read rff file until searched Character parameter & return it
! Author : P. Robert , LPP, 2011 Jan. 23
!----------------------------------------------------------------------!

  character(len=120) :: get_paramC
  character(len=*)   :: param
  character(len=255) :: com

  integer eof

! -----------------------------------

  com='xxx'
  ll=LEN_TRIM(param)

  DO WHILE (com(5:5+ll) /= param(1:ll))
     READ(ifc,'(a)',iostat=eof) com
     IF(com(1:18) == 'START INDEXED_DATA') EXIT
     IF(eof /= 0) EXIT
  ENDDO

! si les param ne sont pas dans l'odre, deuxieme chance avec un rewind
  IF(com(1:18) == 'START INDEXED_DATA') REWIND ifc

  DO WHILE (com(5:5+ll) /= param(1:ll))
     READ(ifc,'(a)',iostat=eof) com
     IF(com(1:18) == 'START INDEXED_DATA') EXIT
     IF(eof /= 0) EXIT
  ENDDO

  IF(com(1:18) == 'START INDEXED_DATA') THEN
               write(*,*) '*   Keyword ',param(1:ll),' not found'
               write(*,*) '    Value set to undefined'
               write(*,*) '    file rewinded for next use'
               get_paramC='undefined'
               REWIND ifc
               RETURN
  ENDIF

  IF(eof /= 0) THEN
               write(*,*) '*** Keyword ',param(1:ll),' not found'
               write(*,*) '*** Keyword  START INDEXED_DATA not found'
               write(*,*) '*** RFF file not correct'
               write(*,*) '*** eof reached on RFF file'
               write(*,*) '    Program aborted !!!'
       stop 'lib_rw_rff.o/get_paramC         : *** ERROR !! Program aborted !'
  ENDIF

  i=INDEX(com,':')
  IF (i /= 0) THEN
              k=LEN_TRIM(com)
              get_paramC=com(i+2:k)
              ELSE
              write(*,*) '*** RFF PARAMETER not found, program aborted'
              write(*,*) '*** ',TRIM(param),' cannot be found'
              write(*,*) '      Program aborted !!!'
       stop 'lib_rw_rff.o/get_paramC         : *** ERROR !! Program aborted !'
  ENDIF

  RETURN
  END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  function get_paramI(param,ifc)

!----------------------------------------------------------------------!
! Object : Read rff file until searched Integer   parameter & return it
! Author : P. Robert , LPP, 2011 Jan. 23
!----------------------------------------------------------------------!

  integer            :: get_paramI
  character(len=*)   :: param
  character(len=255) :: com

  integer eof

! -----------------------------------

  com='xxx'
  ll=LEN_TRIM(param)

  DO WHILE (com(5:5+ll) /= param(1:ll))
     READ(ifc,'(a)',iostat=eof) com
     IF(com(1:18) == 'START INDEXED_DATA') EXIT
     IF(eof /= 0) EXIT
  ENDDO

! si les param ne sont pas dans l'odre, deuxieme chance avec un rewind
  IF(com(1:18) == 'START INDEXED_DATA') REWIND ifc

  DO WHILE (com(5:5+ll) /= param(1:ll))
     READ(ifc,'(a)',iostat=eof) com
     IF(com(1:18) == 'START INDEXED_DATA') EXIT
     IF(eof /= 0) EXIT
  ENDDO

  IF(com(1:18) == 'START INDEXED_DATA') THEN
               write(*,*) '*   Keyword ',param(1:ll),' not found'
               write(*,*) '    Value set to 999999999'
               write(*,*) '    file rewinded for next use'
               get_paramI= 999999999
               REWIND ifc
               RETURN
  ENDIF

  IF(eof /= 0) THEN
               write(*,*) '*** Keyword ',param(1:ll),' not found'
               write(*,*) '*** Keyword  START INDEXED_DATA not found'
               write(*,*) '*** RFF file not correct'
               write(*,*) '*** eof reached on RFF file'
               write(*,*) '    Program aborted !!!'
       stop 'lib_rw_rff.o/get_paramI         : *** ERROR !! Program aborted !'
  ENDIF

  i=INDEX(com,':')
  IF (i /= 0) THEN
              k=LEN_TRIM(com)
              READ(com(i+2:k),*) ival
              get_paramI=ival
              ELSE
              write(*,*) '*** RFF PARAMETER not found, program aborted'
              write(*,*) '*** ',TRIM(param),' cannot be found'
              write(*,*) '      Program aborted !!!'
       stop 'lib_rw_rff.o/get_paramI         : *** ERROR !! Program aborted !'
  ENDIF

  RETURN
  END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  function get_paramI2(param,ifc)

!----------------------------------------------------------------------!
! Object : Read rff file until searched 2nd Int.  parameter & return it
! Author : P. Robert , LPP, 2011 Jan. 23
!----------------------------------------------------------------------!

  integer:: get_paramI2
  character(len=*)   :: param
  character(len=255) :: com

  integer eof

! -----------------------------------

  com='xxx'
  ll=LEN_TRIM(param)
  backspace ifc

  DO WHILE (com(5:5+ll) /= param(1:ll))
     READ(ifc,'(a)',iostat=eof) com
     IF(com(1:18) == 'START INDEXED_DATA') EXIT
     IF(eof /= 0) EXIT
  ENDDO

! si les param ne sont pas dans l'odre, deuxieme chance avec un rewind
  IF(com(1:18) == 'START INDEXED_DATA') REWIND ifc

  DO WHILE (com(5:5+ll) /= param(1:ll))
     READ(ifc,'(a)',iostat=eof) com
     IF(com(1:18) == 'START INDEXED_DATA') EXIT
     IF(eof /= 0) EXIT
  ENDDO

  IF(com(1:18) == 'START INDEXED_DATA') THEN
               write(*,*) '*   Keyword ',param(1:ll),' not found'
               write(*,*) '    Value set to 999999999'
               write(*,*) '    file rewinded for next use'
               get_paramI2= 999999999
               REWIND ifc
               RETURN
  ENDIF

  IF(eof /= 0) THEN
               write(*,*) '*** Keyword ',param(1:ll),' not found'
               write(*,*) '*** Keyword  START INDEXED_DATA not found'
               write(*,*) '*** RFF file not correct'
               write(*,*) '*** eof reached on RFF file'
               write(*,*) '    Program aborted !!!'
       stop 'lib_rw_rff.o/get_paramI2        : *** ERROR !! Program aborted !'
  ENDIF

  i=INDEX(com,':')
  IF (i /= 0) THEN
                k=LEN_TRIM(com)
                READ(com(i+2:k),*,iostat=ierr) ival1,ival
                IF(ierr == 0) THEN
                                get_paramI2=ival
                              ELSE
                                get_paramI2=1
                ENDIF
              ELSE
              write(*,*) '*** RFF PARAMETER not found, program aborted'
              write(*,*) '*** ',TRIM(param),' cannot be found'
              write(*,*) '      Program aborted !!!'
       stop 'lib_rw_rff.o/get_paramI2        : *** ERROR !! Program aborted !'
  ENDIF

  RETURN
  END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  function get_paramR(param,ifc)

!----------------------------------------------------------------------!
! Object : Read rff file until searched Real SNGL parameter & return it
! Author : P. Robert , LPP, 2011 Jan. 23
!----------------------------------------------------------------------!

  real(kind=4)       :: get_paramR
  character(len=*)   :: param
  character(len=255) :: com

  integer eof

! -----------------------------------

  com='xxx'
  ll=LEN_TRIM(param)

  DO WHILE (com(5:5+ll) /= param(1:ll))
     READ(ifc,'(a)',iostat=eof) com
     IF(com(1:18) == 'START INDEXED_DATA') EXIT
     IF(eof /= 0) EXIT
  ENDDO

! si les param ne sont pas dans l'odre, deuxieme chance avec un rewind
  IF(com(1:18) == 'START INDEXED_DATA') REWIND ifc

  DO WHILE (com(5:5+ll) /= param(1:ll))
     READ(ifc,'(a)',iostat=eof) com
     IF(com(1:18) == 'START INDEXED_DATA') EXIT
     IF(eof /= 0) EXIT
  ENDDO

  IF(com(1:18) == 'START INDEXED_DATA') THEN
               write(*,*) '*   Keyword ',param(1:ll),' not found'
               write(*,*) '    Value set to 1.e30'
               write(*,*) '    file rewinded for next use'
               get_paramR= 1.e30
               REWIND ifc
               RETURN
  ENDIF

  IF(eof /= 0) THEN
               write(*,*) '*** Keyword ',param(1:ll),' not found'
               write(*,*) '*** Keyword  START INDEXED_DATA not found'
               write(*,*) '*** RFF file not correct'
               write(*,*) '*** eof reached on RFF file'
               write(*,*) '    Program aborted !!!'
       stop 'lib_rw_rff.o/get_paramR         : *** ERROR !! Program aborted !'
  ENDIF

  i=INDEX(com,':')
  IF (i /= 0) THEN
                k=LEN_TRIM(com)
                READ(com(i+2:k),*) rval
                get_paramR=rval
              ELSE
                write(*,*) '*** RFF PARAMETER not found, program aborted'
                write(*,*) '*** ',TRIM(param),' cannot be found'
                write(*,*) '      Program aborted !!!'
       stop 'lib_rw_rff.o/get_paramR         : *** ERROR !! Program aborted !'
    ENDIF

  RETURN
  END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  function get_paramD(param,ifc)

!----------------------------------------------------------------------!
! Object : Read rff file until searched Real DBLE parameter & return it
! Author : P. Robert , LPP, 2011 Jan. 23
!----------------------------------------------------------------------!

  real(kind=8)       :: get_paramD, dval
  character(len=*)   :: param
  character(len=255) :: com

  integer eof

! -----------------------------------

  com='xxx'
  ll=LEN_TRIM(param)

  DO WHILE (com(5:5+ll) /= param(1:ll))
     READ(ifc,'(a)',iostat=eof) com
     IF(com(1:18) == 'START INDEXED_DATA') EXIT
     IF(eof /= 0) EXIT
  ENDDO

! si les param ne sont pas dans l'ordre, deuxieme chance avec un rewind
  IF(com(1:18) == 'START INDEXED_DATA') REWIND ifc

  DO WHILE (com(5:5+ll) /= param(1:ll))
     READ(ifc,'(a)',iostat=eof) com
     IF(com(1:18) == 'START INDEXED_DATA') EXIT
     IF(eof /= 0) EXIT
  ENDDO

  IF(com(1:18) == 'START INDEXED_DATA') THEN
               write(*,*) '*   Keyword ',param(1:ll),' not found'
               write(*,*) '    Value set to 1.D30'
               write(*,*) '    file rewinded for next use'
               get_paramD=1.D30
               REWIND ifc
               RETURN
  ENDIF

  IF(eof /= 0) THEN
               write(*,*) '*** Keyword ',param(1:ll),' not found'
               write(*,*) '*** Keyword  START INDEXED_DATA not found'
               write(*,*) '*** RFF file not correct'
               write(*,*) '*** eof reached on RFF file'
               write(*,*) '    Program aborted !!!'
       stop 'lib_rw_rff.o/get_paramD         : *** ERROR !! Program aborted !'
  ENDIF

  i=INDEX(com,':')
  IF (i /= 0) THEN
                k=LEN_TRIM(com)
                READ(com(i+2:k),*) dval
                get_paramD=dval
              ELSE
                write(*,*) '*** RFF PARAMETER not found, program aborted'
                write(*,*) '*** ',TRIM(param),' cannot be found'
                write(*,*) '      Program aborted !!!'
       stop 'lib_rw_rff.o/get_paramD         : *** ERROR !! Program aborted !'
  ENDIF

  RETURN
  END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  function get_paramT(param,ifc,nbli)

!----------------------------------------------------------------------!
! Object : Read rff file until searched Character parameter & return it
! Author : P. Robert , LPP, 2011 Jan. 23
!----------------------------------------------------------------------!

  character(len=80), dimension(50)  :: get_paramT
  character(len=*)   :: param
  character(len=255) :: com

  integer eof

! -----------------------------------

  com='xxx'
  ll=LEN_TRIM(param)

  DO WHILE (com(5:5+ll) /= param(1:ll))
     READ(ifc,'(a)',iostat=eof) com
     IF(com(1:18) == 'START INDEXED_DATA') EXIT
     IF(eof /= 0) EXIT
  ENDDO

! si les param ne sont pas dans l'odre, deuxieme chance avec un rewind
  IF(com(1:18) == 'START INDEXED_DATA') REWIND ifc

  DO WHILE (com(5:5+ll) /= param(1:ll))
     READ(ifc,'(a)',iostat=eof) com
     IF(com(1:18) == 'START INDEXED_DATA') EXIT
     IF(eof /= 0) EXIT
  ENDDO

  IF(com(1:18) == 'START INDEXED_DATA') THEN
               write(*,*) '*   Keyword ',param(1:ll),' not found'
               write(*,*) '    Value set to undefined'
               write(*,*) '    file rewinded for next use'
               get_paramT='undefined'
               nbli=0
               REWIND ifc
               RETURN
  ENDIF

  ll=LEN_TRIM(com)
  nbli=0
  get_paramT=' '

  DO WHILE ((com(ll:ll) /= '}') .and. (nbli <= 50))
     READ(ifc,'(a)',iostat=eof) com
     IF(eof /= 0) EXIT
     ll=LEN_TRIM(com)
     nbli=nbli+1
     get_paramT(nbli)=com(1:80)
  ENDDO

  IF(eof /= 0) THEN
               write(*,*) '*** Keyword ',param(1:ll),' not found'
               write(*,*) '*** Keyword  START INDEXED_DATA not found'
               write(*,*) '*** RFF file not correct'
               write(*,*) '*** eof reached on RFF file'
               write(*,*) '    Program aborted !!!'
       stop 'lib_rw_rff.o/get_paramT         : *** ERROR !! Program aborted !'
  ENDIF

  RETURN
  END

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  function Fbasename(path)

!----------------------------------------------------------------------!
! Object : Return file name of a path
! Author : P. Robert , LPP, 2011 March 08
!----------------------------------------------------------------------!

  character(len=*)   :: path
  character(len=255) :: ligne, Fbasename

! ------------------------------------------

  ligne=path

  np=LEN_TRIM(path)

! recherche du dernier "/"

  do i=1,np
     ns=INDEX(ligne,"/")
     ligne=ligne(ns+1:np)
  enddo

  Fbasename=ligne

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  function Fdirname(path)

!----------------------------------------------------------------------!
! Object : Return directory name of a path
! Author : P. Robert , LPP, 2011 March 08
!----------------------------------------------------------------------!

  character(len=*)   :: path
  character(len=255) :: Fdirname, Fbasename

! ------------------------------------------

  np=LEN_TRIM(path)
  nf=LEN_TRIM(Fbasename(path))
  Fdirname=path(1:np-nf-1)

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX


  function give_RPC_version()

!----------------------------------------------------------------------!
! Object : Return RPC_version
! Author : P. Robert , LPP, 2014 Jan.
!----------------------------------------------------------------------!

  character(len=64) :: give_RPC_version
  give_RPC_version= 'RPC_V5.1, February 2021'

  return
  end

!XXXXXXXX0XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

! -----------------------------------------------
MODULE String_Functions  ! by David Frank  dave_frank@hotmail.com
IMPLICIT NONE            ! http://home.earthlink.net/~dave_gemini/strings.f90

! Copy (generic) char array to string or string to char array
! Clen           returns same as LEN      unless last non-blank char = null
! Clen_trim      returns same as LEN_TRIM    "              "
! Ctrim          returns same as TRIM        "              "
! Count_Items    in string that are blank or comma separated
! Reduce_Blanks  in string to 1 blank between items, last char not blank
! Replace_Text   in all occurances in string with replacement string
! Spack          pack string's chars == extract string's chars
! Tally          occurances in string of text arg
! Translate      text arg via indexed code table
! Upper/Lower    case the text arg

INTERFACE Copy    ! generic
   MODULE PROCEDURE copy_a2s, copy_s2a
END INTERFACE Copy

CONTAINS
! ------------------------
PURE FUNCTION Copy_a2s(a)  RESULT (s)    ! copy char array to string
CHARACTER,INTENT(IN) :: a(:)
CHARACTER(SIZE(a)) :: s
INTEGER :: i
DO i = 1,SIZE(a)
   s(i:i) = a(i)
END DO
END FUNCTION Copy_a2s

! ------------------------
PURE FUNCTION Copy_s2a(s)  RESULT (a)   ! copy s(1:Clen(s)) to char array
CHARACTER(*),INTENT(IN) :: s
CHARACTER :: a(LEN(s))
INTEGER :: i
DO i = 1,LEN(s)
   a(i) = s(i:i)
END DO
END FUNCTION Copy_s2a

! ------------------------
PURE INTEGER FUNCTION Clen(s)      ! returns same result as LEN unless:
CHARACTER(*),INTENT(IN) :: s       ! last non-blank char is null
INTEGER :: i
Clen = LEN(s)
i = LEN_TRIM(s)
IF (s(i:i) == CHAR(0)) Clen = i-1  ! len of C string
END FUNCTION Clen

! ------------------------
PURE INTEGER FUNCTION Clen_trim(s) ! returns same result as LEN_TRIM unless:
CHARACTER(*),INTENT(IN) :: s       ! last char non-blank is null, if true:
INTEGER :: i                       ! then len of C string is returned, note:
                                   ! Ctrim is only user of this function
i = LEN_TRIM(s) ; Clen_trim = i
IF (s(i:i) == CHAR(0)) Clen_trim = Clen(s)   ! len of C string
END FUNCTION Clen_trim

! ----------------
FUNCTION Ctrim(s1)  RESULT(s2)     ! returns same result as TRIM unless:
CHARACTER(*),INTENT(IN)  :: s1     ! last non-blank char is null in which
CHARACTER(Clen_trim(s1)) :: s2     ! case trailing blanks prior to null
s2 = s1                            ! are output
END FUNCTION Ctrim

! --------------------
INTEGER FUNCTION Count_Items(s1)  ! in string or C string that are blank or comma separated
CHARACTER(*) :: s1
CHARACTER(Clen(s1)) :: s
INTEGER :: i, k

s = s1                            ! remove possible last char null
k = 0  ; IF (s /= ' ') k = 1      ! string has at least 1 item
DO i = 1,LEN_TRIM(s)-1
   IF (s(i:i) /= ' '.AND.s(i:i) /= ',' &
                    .AND.s(i+1:i+1) == ' '.OR.s(i+1:i+1) == ',') k = k+1
END DO
Count_Items = k
END FUNCTION Count_Items

! --------------------
FUNCTION Reduce_Blanks(s)  RESULT (outs)
CHARACTER(*)      :: s
CHARACTER(LEN_TRIM(s)) :: outs
INTEGER           :: i, k, n

n = 0  ; k = LEN_TRIM(s)          ! k=index last non-blank (may be null)
DO i = 1,k-1                      ! dont process last char yet
   n = n+1 ; outs(n:n) = s(i:i)
   IF (s(i:i+1) == '  ') n = n-1  ! backup/discard consecutive output blank
END DO
n = n+1  ; outs(n:n)  = s(k:k)    ! last non-blank char output (may be null)
IF (n < k) outs(n+1:) = ' '       ! pad trailing blanks
END FUNCTION Reduce_Blanks

! ------------------
FUNCTION Replace_Text (s,text,rep)  RESULT(outs)
CHARACTER(*)        :: s,text,rep
CHARACTER(LEN(s)+100) :: outs     ! provide outs with extra 100 char len
INTEGER             :: i, nt, nr

outs = s ; nt = LEN_TRIM(text) ; nr = LEN_TRIM(rep)
DO
   i = INDEX(outs,text(:nt)) ; IF (i == 0) EXIT
   outs = outs(:i-1) // rep(:nr) // outs(i+nt:)
END DO
END FUNCTION Replace_Text

! ---------------------------------
FUNCTION Spack (s,ex)  RESULT (outs)
CHARACTER(*) :: s,ex
CHARACTER(LEN(s)) :: outs
CHARACTER :: aex(LEN(ex))   ! array of ex chars to extract
INTEGER   :: i, n

n = 0  ;  aex = Copy(ex)
DO i = 1,LEN(s)
   IF (.NOT.ANY(s(i:i) == aex)) CYCLE   ! dont pack char
   n = n+1 ; outs(n:n) = s(i:i)
END DO
outs(n+1:) = ' '     ! pad with trailing blanks
END FUNCTION Spack

! --------------------
INTEGER FUNCTION Tally (s,text)
CHARACTER(*) :: s, text
INTEGER :: i, nt

Tally = 0 ; nt = LEN_TRIM(text)
DO i = 1,LEN(s)-nt+1
   IF (s(i:i+nt-1) == text(:nt)) Tally = Tally+1
END DO
END FUNCTION Tally

! ---------------------------------
FUNCTION Translate(s1,codes)  RESULT (s2)
CHARACTER(*)       :: s1, codes(2)
CHARACTER(LEN(s1)) :: s2
CHARACTER          :: ch
INTEGER            :: i, j

DO i = 1,LEN(s1)
   ch = s1(i:i)
   j = INDEX(codes(1),ch) ; IF (j > 0) ch = codes(2)(j:j)
   s2(i:i) = ch
END DO
END FUNCTION Translate

! ---------------------------------
FUNCTION Upper(s1)  RESULT (s2)
CHARACTER(*)       :: s1
CHARACTER(LEN(s1)) :: s2
CHARACTER          :: ch
INTEGER,PARAMETER  :: DUC = ICHAR('A') - ICHAR('a')
INTEGER            :: i

DO i = 1,LEN(s1)
   ch = s1(i:i)
   IF (ch >= 'a'.AND.ch <= 'z') ch = CHAR(ICHAR(ch)+DUC)
   s2(i:i) = ch
END DO
END FUNCTION Upper

! ---------------------------------
FUNCTION Lower(s1)  RESULT (s2)
CHARACTER(*)       :: s1
CHARACTER(LEN(s1)) :: s2
CHARACTER          :: ch
INTEGER,PARAMETER  :: DUC = ICHAR('A') - ICHAR('a')
INTEGER            :: i

DO i = 1,LEN(s1)
   ch = s1(i:i)
   IF (ch >= 'A'.AND.ch <= 'Z') ch = CHAR(ICHAR(ch)-DUC)
   s2(i:i) = ch
END DO
END FUNCTION Lower

END MODULE String_Functions
