!XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
  
  program vectime_calibration_GEOULF

!--------------------------------------------------------------------0--
! Laboratoire de Physique des Plasmas                                  
!
! vectime_calibration_GEOULF : Read input RFF file VecTime type, 
!                              calibrate data with continuous method 
!                              and create new VecTime    
!                                                                      
! Input : input & output RFF file names, data are in input RFF file    
! Output: Run report on stdout, and new output RFF file for the data   
!                                                                      
! Written by: Patrick Robert,  ScientiDev, January 2019
!             Derived from cali_CLUSTA.f90 (continuous calibration) 
!             P. Robert , LPP, 2009-2012  
!        
! Object: calibrate a GEOS/ULF VTL1 file, create calibrated VTL2  
! Author: P. Robert , ScientiDev, January 2019
!--------------------------------------------------------------------0--
 
 
! Variables declaration

  use rff_param_def
  use rff_data_def
  
  implicit none
  
  character(len=255) :: in_file_name, ou_file_name
  character(len=120) :: in_data_format, ou_data_format, ou_block_fmt
  character(len=100) :: C1_tmp, C3_tmp,C4_tmp

  character(len= 27)     :: nor_index
  character(len=par_len) :: ext_index

  character(len=1),       dimension(3) :: sensor=(/'X','Y','Z'/)
  character(len=par_len), dimension(:),allocatable ::  data_exind
  character(len=32)  :: apod
  character(len=24)  :: credate
  character(len=255) :: work
  character(len=64)  :: R_ind_ext_for, for_R_exti
  character(len=27)  :: first_index, last_index
  character(len=1)   :: sensrot

  
  integer :: in_file_unit=1, ou_file_unit=2, tmp_file_unit=3
  integer :: i, ii, i1, i2, j,jj,kk,itim,idv=8,ipv=5
  integer :: i_win, ig, le1, block_size,ncomp
  integer :: ierr , fill_val_int, get_pos
  integer :: k_block, i_block, o_block ! Number of blocks read and written 
  integer :: block_number1
  integer :: cal_step, nbp_liss,tm_range_min, tm_range_max
  integer :: N_Kern, N_Shift, N_win, M_Kern,N_read
  integer :: imsres,ierrtime 
  integer :: iano,LSF
  integer, dimension(8) :: int_block ! for reading block
  
  logical :: pr_out=.true., detrend
  
  real (kind=4) :: fdet,fc,f1,f2,fmax,ff1, f, pui_W, R1_tmp,R2_tmp !,R3_tmp
  real (kind=4) :: vo_range_min, vo_range_max
  real (kind=4) :: mod_sr1, mod_sr2, mod_sr3, mod_d1,mod_d2,mod_d3
  real (kind=4) :: pha_sr1, pha_sr2, pha_sr3, pha_d1,pha_d2,pha_d3
  real (kind=4) :: mod_d4,mod_d5, pha_d4,pha_d5, Bdc_4,Bdc_5, Bdc_1,Bdc_2
  real (kind=4) :: Bperp_1, Bperp_2, Bperp_3,Bperp_4, Bperp_5, Bdc_x,Bdc_y
  real (kind=4) :: phacor_1, phacor_2, phacor_3, phacor_4, phacor_5, Bperp_A, pha_A,diff
  real (kind=4) :: misal_angle,spin_phase,rotdeg
  real (kind=4) :: df,edge,sigma
  real (kind=4) :: dyna, tm_offset, spin_rate,sam_rate
  real (kind=8) :: spin_period, sample_rate, D1_tmp
  real (kind=4) :: pisd,pi2 
  real (kind=4) :: depif,phicr,sinphi,cosphi
  real (kind=8) :: Pi, arg, Ddepif
  real (kind=4) :: Eul1,Eul2,Eul3, sinrot,cosrot, alpha
! real (kind=4) :: retard_x=0.018808,retard_y=0.0195 ,retard_z=0.0296,retard_xy,corr_ret

  real (kind=4), dimension(5):: flt_block
  real (kind=4), dimension(:,:),allocatable :: data_vec, proc_vec

  real (kind=4), dimension(:),  allocatable :: RA1_tmp,RA2_tmp,RA3_tmp  ! for working
  real (kind=4), dimension(:),  allocatable :: Weight    ! for apodisation
  real (kind=4), dimension(:),  allocatable :: DSS       ! for   sine table
  real (kind=4), dimension(:),  allocatable :: DCS       ! for cosine table

  complex (kind=4) :: gainGEOS,c
  complex (kind=4), dimension(:,:),allocatable :: TFcor  ! for TF correc.
  complex (kind=4), dimension(:,:),allocatable :: CSpec  ! for deconvolution
  

!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!
!
!  SUMMARY:
!  ~~~~~~~
!
!   I) INPUT  PARAMETER  &  DATA  FILE 
!      1) Reading input parameters
!      2) Check validity of input parameters
!
!  II) METADATA  &  CONSTANT  DATA  PROCESSING
!      1) Reading Metadata
!      2) Test on Metadata
!         a) mission, experiment, obsevatory....
!         b) VecTime class only with TM data as integer
!      3) Reading Constant Data
!      4) Modify metadata parameters
!
! III) PRE-PROC. INITIALISATIONS
!      1) Setting various parameters
!      2) Memory block allocation of all arrays
!      3) Ititialisation of calibration files
!      4) Ititialisation of tables for deconvolution
!
!  IV) DATA  PROCESSING
!      1) Test if there are data, if not make empty file
!      2) Load successives windows of N_Kern size for calibration
!         a) Load and calibrate first windows
!         b) Big loop on N_win calibration window
!         c) Calibration of the current window
!         d) Store in temporary file kept blocs from the calibration window
!      3) Memory block desallocation
!  
!   V) CHECK  NORMAL  TERMINATION
!      1) check if bloc number read is the same as expected
!      2) check if bloc number written is the same as expected
!
!  VI) CREATE  OUTPUT  RFF FILE
!      1) Update first & last index
!      2) Opening  output_file and writing header, meta & constant data
!      3) Complete output_file by indexed data taken from temporary file
!      4) Terminate the output_file and close
!
! VII) END
!
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!

!-----------------------!
! Executable Statements !
!-----------------------!

  write(*,*) ('-',i=1,72)
  write(*,*) 'Begin vectime_calibration_GEOULF'
  write(*,*) 'Calibration of GEOS/ULF waveform data'
  write(*,*) ('-',i=1,72)
  write(*,*)

  call print_date_time('Start of program')


! --------------------------------------------------------!  
! I)  I N P U T   P A R A M E T E R  &  D A T A   F I L E !
!---------------------------------------------------------!

! 1) Reading input parameters
!    ========================

! in_file_name : name of an input  vectime RFF file, such as VTL1.rff
! ou_file_name : name of an output vectime RFF file, such as VTL2.rff 
!         fdet : detrend frequency (0. for classis despin)"
!         fc   : Frequency cut-off for calibration"
!         f1   : frequency min for filtering (Min=0 => Fc)"
!         f2   : frequency max for filtering (Max=0 => Nyquist)"
!     cal_step : calibration step asked (1-8)"
!                1- Conversion  of telemetry into Volt
!                2- Volt with spin signal removed or detrend applied
!                3- Calibration (Volt to nT) in spinning system
!                4- Transformation of spining sensor system in SR2 (Despun)
!                5- Add DC field from Despin info on Bx and By (z unchanged)
!                6- Change Bx, By into BDCx, BDCy, z into misalignment angle
!                7- Same that 4, but in SRV, and add 2 components: DCx, DCy
!                   in SRV; option for CAA data.
!                8- same as step 4, but transformed into GSE system (set fc > 2*Fs)
!         *** Note: program tested only for step 7 for CDPP data
!      N_Kern  : Kernel size; 1024 recommanded for cont. cal.
!      N_shift : for sliding window; 2 produce best calibration but
!                high CPU time
!        apod  : trapezium (t) or Gaussian (g); 
!                g recommanded for continuous calibration
!                t recommandes for classical method, with N_shift=N_Kern 

  call read_input(in_file_name,ou_file_name,fdet, &
                  fc,f1,f2,cal_step,N_Kern, N_Shift,apod)

  write(*,*)
  write(*,*) 'Input  file : ', trim(in_file_name)
  write(*,*) 'Output file : ', trim(ou_file_name)

  write(*,*) 'Weighting function : ',trim(apod)

  
! 2) Check validity of input parameters
!    ==================================

  call print_date_time('Check input parameters...')

  if(apod(1:1) /= 't' .and. apod(1:1) /= 'g') then
     write(*,*)
     write(*,'(a)') '*** error in vectime_calibration_GEOULF: apod not valid, only "t" or "g"'
     write(*,'(a)') '    Program aborted ! '
     write(*,*)
     stop 'vectime_calibration_GEOULF.exe  : *** ERROR !! Program aborted !'
  endif    

  call print_date_time('End of part I')


! --------------------------------------------------------------------
! II)  M E T A    &   C O N S T A N T   D A T A   P R O C E S S I N G
! --------------------------------------------------------------------

  
! 1) Reading Metadata
!    ================

  write(*,*)
  write(*,*) 'READING METADATA...'
  write(*,*) '----------------'

! reading AND writing on std output manda_param & optio_param only

  call rff_R_metadata(in_file_unit,in_file_name)

  write(*,*) 'Done...'


! 2) Test on Metadata
!    ================

!    a) mission, experiment, obsevatory....

  if (manda_param%mission_name /= 'GEOS'     .or. &
      manda_param%experiment_name /= 'S300' .or. &
      manda_param%observatory_number < 1        .or. &
      manda_param%observatory_number > 2 ) then
     write(*,*)
     write(*,'(a)') ' -> ERROR vectime_calibration_GEOULF: THIS IS NO GEOS/ULF DATA ***'
     write(*,'(a)') ' -> PROGRAM ABORTED !!!'
     stop 'vectime_calibration_GEOULF.exe  : *** ERROR !! Program aborted !'
     endif    

!    b) VecTime class only with TM data as integer

  if (manda_param%file_class /= 'VecTime' ) then 
     write(*,*)
     write(*,'(a)') ' -> ERROR vectime_calibration_GEOULF: CLASS OF INPUT FILE IS NOT VECTIME ***'
     write(*,'(a)') ' -> PROGRAM ABORTED !!!'
     stop 'vectime_calibration_GEOULF.exe  : *** ERROR !! Program aborted !'
     endif    

! Check that data type in L1 RFF input file for GEOS/ULF is integer
  
  if (manda_param%data_type /= 'INT') then
     write(*,*)
     write(*,'(a)') ' -> ERROR vectime_calibration_GEOULF: INPUT DATA TYPE SHOULD BE INTEGER ***'
     write(*,'(a)') ' -> PROGRAM ABORTED !!!'
     close(UNIT=in_file_unit)
     stop 'vectime_calibration_GEOULF.exe  : *** ERROR !! Program aborted !'
                                          else
     write(*,*) 'Input file supposed to be a L1 RFF GEOS/ULF data file'
  endif

  ! *** autres tests eventuels a faire ***
 
     block_size = 1


  write(*,*)
  write(*,*) 'READING CONSTANT DATA...'
  write(*,*) '------------------------'

  call rff_R_const_data(in_file_unit)
  call rff_W_const_data(6)

  write(*,*) 'Done...'

! Retrieve usefull constant data

  sample_rate  =const_data%sample_rate
  vo_range_min =const_data%volt_range_min
  vo_range_max =const_data%volt_range_max
  tm_range_min =const_data%tm_range_min
  tm_range_max =const_data%tm_range_max
  spin_period  =const_data%spin_period
 
  spin_rate=real(1.D0/spin_period)
  sam_rate=real(sample_rate)

! check detrend frequency for detrend use or not
! requires sample rate

  if (fdet < 1.e-10 ) then
      write(*,*) 'Detrend frequency too small, detrend will not be applied !'
      detrend= .false.
                      else
      detrend= .true.
      nbp_liss= int(sam_rate/fdet +0.5)
      write(*,*) 'Detrend : NB of point for smoothing =',nbp_liss

      if (nbp_liss > N_Kern) then
          write(*,*) 'Detrend : NB point too high, nothing will be done !'
          detrend= .false.
      endif

      if (nbp_liss < 2) then
          write(*,*) 'Detrend : NB point < 2, nothing will be done !'
          detrend= .false.
      endif
  endif

  if (detrend) then
      write(*,*) 'Detrend will be applied'
               else
      write(*,*) 'Detrend will NOT be applied'
  endif

 ! modify F2 value according Nyquist frequency if f2=0.

  if (f2 <= f1) then
     f2= sam_rate/2.
  endif

  write(*,*)
  write(*,*) 'sample_rate     =', sample_rate
  write(*,*) 'time_resolution =', optio_param%time_resolution
  write(*,*) 'vo_range_min    =', vo_range_min
  write(*,*) 'vo_range_max    =', vo_range_max
  write(*,*) 'tm_range_min    =', tm_range_min
  write(*,*) 'tm_range_max    =', tm_range_max
  write(*,*) 'spin_period     =', spin_period
  write(*,*) 'spin_rate       =', spin_rate



! 4) Modify metadata parameters
!    ==========================

  write(*,*)
  write(*,*) 'MODIFY METADATA....'
  write(*,*) '-------------------'

! Update file name
! ----------------

  manda_param%file_name=trim(ou_file_name)

! Update history field
! --------------------

  call gdatiso(credate)
  call rff_update_history(credate,'RPC_vectime_calibration_GEOULF')


! Change metadata according to data processing, done in "data processing" section
! ---------------

! Common ouput values changes for cal_step in [1-5,8]
! ---------------------------------------------------

! DATA_LABEL

  manda_param%data_label= 'Bx ; By ; Bz'

! DATA_TYPE

  manda_param%data_type= 'FLT'

! DATA_DIMENSION

  manda_param%data_dimension(1)= 3

! DATA_FORMAT

! manda_param%data_format must be something as '(3(E14.6,1x))'
! set default value

  in_data_format=manda_param%data_format
  ou_data_format= '(E13.6,",",E13.6,",",E13.6)'
  manda_param%data_format= '(E13.6,1x,E13.6,1x,E13.6)'

! INDEX_EXTENSION_FORMAT
! (ajou d'une virgule pour les VTL2, Cluster compatibilité)

!  manda_param%index_extension_length= manda_param%index_extension_length+1

! DATA_FILL_VALUE

  manda_param%data_fill_value= '-0.1000E+31'

! DATA_BLOCK_NUMBER

  block_number1=manda_param%block_number

! computation of number of windows to calibrate
  N_win= ((block_number1 -N_Kern)/N_Shift) +1

! computation of number of blocks to be read:
  N_read= N_Kern + (N_win-1)*N_Shift

! computation of number of block calibrated
  manda_param%block_number= N_win*N_shift


! Specific change according processing step
! -----------------------------------------

  pisd=acos(-1.)/180.
  pi2=2.*acos(-1.)

! DATA_UNITS, DATA_COORDINATE_SYSTEM, SUB_TITLE and others

  select case (cal_step)

    case(1)
         manda_param%data_units= 'Volt ; Volt ; Volt'
         optio_param%sub_title = 'Step 1: Data in spinning system [Volt]'
         ncomp=3

    case(2)
         manda_param%data_units= 'Volt ; Volt ; Volt'
         optio_param%sub_title = 'Step 2: Data in spinning system [Volt] without DC'
         ncomp=3

    case(3)
         manda_param%data_units= 'nT ; nT ; nT'
         optio_param%sub_title = 'Step 3: Data in spinning system [nT] without DC'
         ncomp=3

    case(4)
         manda_param%data_units       = 'nT ; nT ; nT'
         call cbestfor(fc,C3_tmp)
         call cbestfor(fdet,C4_tmp)
         C1_tmp='(a," (Fc=",'//trim(C3_tmp)//'," Fdet=",'//trim(C4_tmp)//',")",)'
         write(optio_param%sub_title,C1_tmp) &
               'Step 4: Data in SRV system [nT] without DC',fc,fdet
         manda_param%data_coordinate_system='SRV'
         ncomp=3

    case(5)
         manda_param%data_units       = 'nT ; nT ; nT'
         call cbestfor(fc,C3_tmp)
         call cbestfor(fdet,C4_tmp)
         C1_tmp='(a," (Fc=",'//trim(C3_tmp)//'," Fdet=",'//trim(C4_tmp)//',")",)'
         write(optio_param%sub_title,C1_tmp) &
               'Step 5: Data in SRV system [nT] with DC',fc,fdet
         manda_param%data_coordinate_system='SRV'
         ncomp=3

    case(6)
         manda_param%data_units= 'nT ; nT ; degree'
         manda_param%data_label= 'Bx ; By ; Misalignment'
         optio_param%sub_title =  &
                     'Step 6: Perpendicular DC field in SR2 system [nT]'
         manda_param%data_coordinate_system='SR2'
         manda_param%data_format='(F8.2,1x,F8.2,1x,f7.2)'
         ou_data_format= '(f8.2,",",f8.2,",",f7.2)'
         manda_param%data_fill_value='9999.99'
         ncomp=3

!!!         time_res6= float(block_size)/sample_rate

    case(7)
         manda_param%data_units       = 'nT ; nT ; nT ; nT ; nT'
         manda_param%data_label       = 'Bx ; By ; Bz ; Dx ; Dy'
         manda_param%data_dimension   = 5
         call cbestfor(fc,C3_tmp)
         call cbestfor(fdet,C4_tmp)
         C1_tmp='(a," (Fc=",'//trim(C3_tmp)//'," Fdet=",'//trim(C4_tmp)//',")")'
         write(optio_param%sub_title,C1_tmp) &
               'Step 7: Data in SRV system [nT] + separated DC',fc,fdet
         manda_param%data_format= '(E13.6,1x,E13.6,1x,E13.6,1x,E13.6,1x,E13.6)'
         
         ou_data_format= '(E13.6,",",E13.6,",",E13.6,",",E13.6,",",E13.6)'
         manda_param%data_coordinate_system='SRV'
         ncomp=5

    case(8)
         manda_param%data_units       = 'nT ; nT ; nT'
         call cbestfor(fc,C3_tmp)
         call cbestfor(fdet,C4_tmp)
         C1_tmp='(a," (Fc=",'//trim(C3_tmp)//'," Fdet=",'//trim(C4_tmp)//',")",)'
         write(optio_param%sub_title,C1_tmp) &
               'Step 8: Data in GSE system [nT] without DC',fc,fdet
         manda_param%data_coordinate_system='GSE'
         ncomp=3


    case default
         write(*,*)
         write(*,'(a)') ' -> ERROR vectime_calibration_GEOULF: NO VALID CAL_STEP ***'
         write(*,'(a)') ' -> PROGRAM ABORTED !!!'
         close(UNIT=in_file_unit)
     stop 'vectime_calibration_GEOULF.exe  : *** ERROR !! Program aborted !'
  end select

! change values or suppress constant data according calibration  step

  if (cal_step >= 3 .and. cal_step /= 6) then

      const_data%FREQUENCY_FILTER_MIN = f1
      const_data%FREQUENCY_FILTER_MAX = f2
      const_data%FREQUENCY_CUT_OFF    = fc
      const_data%FREQUENCY_DETREND    = fdet
      const_data%CALIB_KERNEL_SIZE    = N_Kern
      const_data%CALIB_SHIFT_SIZE     = N_Shift

   endif

! elements for step 8

  if (cal_step == 8) then
         write(*,*)
         write(*,'(a)') ' -> ERROR vectime_calibration_GEOULF: NO VALID CAL_STEP ***'
         write(*,'(a)') ' -> PROGRAM ABORTED !!!'
         close(UNIT=in_file_unit)
  stop 'vectime_calibration_GEOULF.exe  : *** ERROR !! Program aborted !'

!    srasc=const_data%SPIN_GEI_RIGHT_ASCENSION
!    sdec =const_data%SPIN_GEI_DECLINATION

!    sxgei=cos(srasc*pisd)*cos(sdec*pisd)
!    sygei=sin(srasc*pisd)*cos(sdec*pisd)
!    szgei=sin(sdec*pisd)

!    call tcarsph(sxgei,sygei,szgei,r,tet,phi)

!    write(*,'(1x,a,3f10.4)') 'Spin dans le gei, x  y  z  : ',sxgei,sygei,szgei
!    write(*,'(1x,a,3f10.4)') 'Spin dans le gei, r,tet,phi: ',r,tet/pisd,phi/pisd

  endif


! Print all changed mandatory parameters
! --------------------------------------

  write(*,*)
  write(*,*)'new measurement_type      =', trim(manda_param%measurement_type)
  write(*,*)'new data_fill_val.        =', trim(manda_param%data_fill_value)
  write(*,*)'new data_form             =', trim(manda_param%data_form)
  write(*,*)'new data_format           =', trim(manda_param%data_format)
  write(*,*)'new data_dimension        =',      manda_param%data_dimension(1), &
                                                manda_param%data_dimension(2)
  write(*,*)'new data_type             =', trim(manda_param%data_type)
  write(*,*)'new data_label            =', trim(manda_param%data_label)
  write(*,*)'new data_units            =', trim(manda_param%data_units)
  write(*,*)'new coordinate            =', trim(manda_param%data_coordinate_system)
  write(*,*)'new block number          =',      manda_param%block_number
  write(*,*)

  
! Print all changed optional parameters
! -------------------------------------

  write(*,*)
  write(*,*)'new sub_title=     ', trim(optio_param%sub_title)

  
! Print all output constant data
! ------------------------------

  write(*,*)
  write(*,*) 'List of all output constant_data :'
  write(*,*) '--------------------------------'
  write(*,*)

  call rff_W_const_data(6)
  call print_date_time('End of part II')



! --------------------------------------------------------------------
! III)  P R E - P R O C .  I N I T I A L I S A T I O N S 
! --------------------------------------------------------------------
  

! 1) Setting various parameters
!    ==========================

! Setting writing format

  ou_block_fmt='(a,",",a,","'//trim(ou_data_format)//')'
  write(*,*) 'Writing format for 1 block : ',trim(ou_block_fmt)

! Setting "read" index_extension_format used after Step 3

  call rff_format_W_to_R(manda_param%index_extension_format,R_ind_ext_for)

! settinf 2.*Pi*F for SR to SR2

  depif=mod(pi2*spin_rate/sam_rate,pi2)
  Pi= dacos(-1.D0)
  Ddepif=mod(Pi*2.D0/(spin_period*sample_rate), Pi*2.D0)

  
!    Memory block allocation of all requested arrays
!    ===============================================

  write(*,*) 'Allocate memory begin'

  M_Kern=int(alog(float(N_Kern))/alog(2.)+0.5)

  write(*,*) '  Data Vector for window acquisition         : ', idv, N_Kern
                allocate(data_vec(idv,N_Kern),stat=ierr)

  write(*,*) '  Data Vector for calibration processing     : ', ipv, N_Kern
                allocate(proc_vec(ipv,N_Kern),stat=ierr)

  write(*,*) '  Float   working 1D array dimension         : ', N_Kern
                allocate(RA1_tmp(N_Kern),stat=ierr)

  write(*,*) '  Float   working 1D array dimension         : ', N_Kern
                allocate(RA2_tmp(N_Kern),stat=ierr)

  write(*,*) '  Float   working 1D array dimension         : ', N_Kern
                allocate(RA3_tmp(N_Kern),stat=ierr)

  write(*,*) '  Float   Weight  1D array dimension         : ', N_Kern
                allocate(Weight(N_Kern),stat=ierr)

  write(*,*) '  Complex working 2D array dimension         : ', N_Kern
                allocate(TFcor(3,N_Kern),stat=ierr)

  write(*,*) '  Complex working 2D array dimension         : ', N_Kern
                allocate(CSpec(3,N_Kern),stat=ierr)

  write(*,*) '  Character 1D array dimension for index     : ', N_Kern
                allocate(data_index(N_Kern),stat=ierr)

  write(*,*) '  Character 1D array dimension for ext_index : ', N_Kern
                allocate(data_exind(N_Kern),stat=ierr)
   
  write(*,*) '  Float   sine table 1D array dimension      : ', M_Kern
                allocate(DSS(M_Kern),stat=ierr)

  write(*,*) '  Float cosine table 1D array dimension      : ', M_Kern
                allocate(DCS(M_Kern),stat=ierr)

  write(*,*) 'Allocate memory done...'

  
! Computation of Nyquiest frequency

  fmax= sam_rate/2.
  write(*,*) 'Fmax=',fmax
  if (abs(f2).lt.1.e-5) then
      write(*,*) 'F2=',f2,' corrected to Nyquiest frequency=',fmax
      f2=fmax
  endif

  for_R_exti='(i1,1x,i9,f10.7,3f7.2,1x,a1,i2,1x,f7.2)'

! complex gain of transfer function at spin frequency (modulus and phase)

  call modpha(gainGEOS(spin_rate),mod_sr1,pha_sr1)
  call modpha(gainGEOS(spin_rate),mod_sr2,pha_sr2)
  call modpha(gainGEOS(spin_rate),mod_sr3,pha_sr3)

  write(*,*) 'Transfer function at spin frequency :'
  write(*,*) 'spin_rate       =', spin_rate
  write(*,*) 'mod_srx (V/nT),  pha_srx (d.) =',mod_sr1,pha_sr1
  write(*,*) 'mod_sry (V/nT),  pha_sry (d.) =',mod_sr2,pha_sr2
  write(*,*) 'mod_srz (V/nT),  pha_srz (d.) =',mod_sr3,pha_sr3
  write(*,*)


! 4) Ititialisation of tables for deconvolution
!    ==========================================

! * Weighting fonction for waveform before FFT

  if(apod == 'g') then

!    Gaussian weighting function

     edge= 1.e-3
     sigma= float(N_Kern/2)/sqrt(-alog(edge))

!    le -0.5 est pour symétriser la gaussienne avec le sommet entre N/2 et N/2 +1

     do j=1,N_Kern
        D1_tmp=(dble(j-N_Kern/2)-0.5D0)/dble(sigma)
        Weight(j)= sngl(exp(-(D1_tmp**2D0)))
     enddo
                  else

!    Trapezoidal weighting function

     Weight(:)=1.
     i=N_Kern/16
     
     do j=1,i
        Weight(j) = float(j-1)/float(i-1)
        jj=N_Kern -j +1
        Weight(jj)= float(j-1)/float(i-1)
     enddo
  endif

! total power of the weihting function

 pui_W=0.

 do j=1,N_Kern
    pui_W=pui_W +Weight(j)*Weight(j)
 enddo   

 pui_W=pui_W/float(N_Kern)

 write(*,*) 'total power of the weihting function : ',pui_W

! indice of first block to keep in the windows of N_Kern points
  i1= (N_Kern -N_Shift)/2 +1    

! indice of last block to keep in the windows of N_Kern points
  i2= i1 +N_Shift -1

  write(*,*)
  write(*,*) 'Weighting function'
  write(*,*) 'W(1)     =', Weight(1)       ! must be = 1.E-3 (edge)
  write(*,*) 'W(i1)    =', Weight(i1)
  write(*,*) 'W(i2)    =', Weight(i2)
  write(*,*) 'W(N_Kern)=', Weight(N_Kern)  ! must be = 1.E-3 (edge)
  write(*,*)
  
! * Loadind sine and cosine table for FFT

  do i=1,M_Kern
     Le1= (2**(M_Kern+1-i))/2
     arg=Pi/dble(Le1)
     DSS(i)=sngl(Dsin(arg))
     DCS(i)=sngl(Dcos(arg))
  enddo

!  computing table for transfer function correction
   
   df=sam_rate/float(N_Kern)

  do ig=1,3

        if(ig == 3) then
                    ff1=max(f1,0.15)
                    ff1=0.15
                    else
                    ff1=max(f1,fc)
        endif

     do i=1,N_Kern/2
!
        f=float(i-1)*df
!       no difference between x y z for GEOS
        c=gainGEOS(f)
        if (cabs(c).lt.1.e-6) c=cmplx(1.e-6,0.)

!       modif 21 janv 2002: filtre carre

        if(f.lt.ff1.or.f.gt.f2) then
                               TFcor(ig,i)=cmplx(0.,0.)
                               else
                               TFcor(ig,i)=cmplx(1.,0.)/c
        endif
     enddo

     do i=1,N_Kern/2
        ii=N_Kern-i+1
        TFcor(ig,ii)=conjg(TFcor(ig,i+1))
     enddo
    
     TFcor(ig,1)=cmplx(real(TFcor(ig,1)),0.)
     TFcor(ig,N_Kern/2+1)=cmplx(real(TFcor(ig,N_Kern/2+1)),0.)

  enddo

! normalisation de la TF directe de DECONVO_3C faite ici

  TFcor(:,:)= TFcor(:,:)/cmplx(float(N_Kern),0.)

  call print_date_time('End of part III')


! --------------------------------------!
!   IV )  D A T A   P R O C E S S I N G !
! --------------------------------------!


  write(*,*)
  write(*,*) 'READING / MODIFY / WRITING DATA...'
  write(*,*) '-------------------------------'


! Achtung ! All metadata are kept in memory and last_index 
! will be changed after total processing.
! Indexed_data are stored in temporary file
! Full RFF output file will be written at the end of processing.

  open(tmp_file_unit,status='scratch')

! 1) Test if there are data, if not make empty file
!    ==============================================

! Check existence of data and close files if no data
! Usefull to produce empty files

  if(manda_param%block_number==0) then
     write(*,*)
     write(*,'(2a)') '*** WARNING LINE725 vectime_calibration_GEOULF: ', &
                          'BLOCK NUMBER =0, NO DATA ***'
     write(*,'(a)')  '     NEW FILE CLOSED WITH NO DATA' 

     o_block=0
     first_index='None'
     last_index ='None'

     go to 99  ! for normal termination (this is the only one go to ! )
!    --------

  endif


! 2) load successives windows of N_Kern size for calibration
!    =======================================================

  write(*,*)
  write(*,*) 'Reading and processing all blocks, supposed equal to ', &
              manda_param%block_number
  write(*,*) '---------------------------------------'
  write(*,*)

! Positionning on START INDEXED_DATA line of input RFF file

  ierr=get_pos('START INDEXED_DATA',in_file_unit)

  IF(ierr /= 0) THEN
        write(*,*) '* WARNING ! keyword START INDEXED_DATA not found'
                ELSE
        write(*,*) 'Positionning at begin of data done'
  ENDIF

! possible blank line after the START INDEXED_DATA
  READ(in_file_unit,*) work
  IF(len_trim(work) /= 0) THEN
      BACKSPACE in_file_unit
  ENDIF

  write(*,*) 'Input file positionned on START INDEXED_DATA line'
  write(*,*)
  write(*,*) 'processing of ', N_win, ' calibration windows '
  write(*,*) '-----------------------------------------------'
  write(*,*)


!    +++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! a) load and calibrate first windows
!    +++++++++++++++++++++++++++++++++++++++++++++++++++++++++


! Read N_Kern Blocks to load the first calibration window
! -------------------------------------------------------

! time resolution in ms
  imsres= int(1000.D0/sample_rate +0.5D0)

! set time error for data gap
  ierrtime=1

! set conversion factor and offset for step 1
  dyna=(vo_range_max -vo_range_min)/float(tm_range_max -tm_range_min)
  tm_offset=(vo_range_max +vo_range_min)/2.
  
  write(*,*) 'dyna,tm_offset=',dyna,tm_offset

! set the count number of current point and points read/written
  k_block=0
  i_block=0
  o_block= 0
  ierrtime=0
  i_win=1
  i=0
  
! set integer fill value
  fill_val_int= -999999 

! This first windows of N_Kern points must be without data gaps
! so, we get N_Kern continuous vectors without data gaps

  call read_N_cont_vectors(N_Kern)

! stop if EOF reached

  if (nor_index(1:3) == 'EOF' .or. nor_index(1:3) == 'END' ) then
     write(*,*)
     write(*,'(a)') ' -> WARNING : vectime_calibration_GEOULF: EOF REACHED ***'
     write(*,'(a)') '    during first window, so output empty file !!'
     o_block=0
     first_index='None'
     last_index ='None'
     go to 99
  endif

! define first_index as the one of the first block to keep

  if(cal_step /= 6) then
     first_index=data_index(i1)
     else
     first_index=data_index(N_Kern/2)
  endif

  write(*,*) 'first index      : ',trim(first_index)

  call print_date_time('    beginning of big calibration lopp')


!    ++++++++++++++++++++++++++++++++++++
! b) Big loop on N_win calibration window
!    ++++++++++++++++++++++++++++++++++++


  write(*,*) ' '
  write(*,*) '=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-'
  write(*,*) 'Big loop on ', N_win,' calibration window'
  write(*,*) '=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-'

Big_Loop:  DO i_win =1, N_win

! No shift for the first window but look for the firts continuous window

  IF (i_win > 1) THEN

! shift window data of N_Shift points

     DO i=1,N_Kern -N_Shift
        DO kk=1,8
           data_vec(kk,i)= data_vec(kk,i +N_Shift)
        END DO

        data_index(i)= data_index(i +N_Shift)
        data_exind(i)= data_exind(i +N_Shift)
     END DO

! read N_Shift new blocks to complete the previous window

     DO i= N_Kern -N_Shift +1, N_Kern

!      reading block from input file

       int_block(:)=fill_val_int
       call r_rff_data_block_int(in_file_unit,nor_index,ext_index,int_block)

!      stop if EOF reached

       if (nor_index(1:3) == 'EOF' .or. &
           nor_index(1:3) == 'END') then
          exit Big_Loop
       endif

       i_block=i_block +1

!   *  test on block continuity

       if(nor_index == ' ') then
         write(*,*) 'nor_index vide'
         stop 'vectime_calibration_GEOULF.exe  : *** ERROR !! Program aborted !'
       endif

       call test_blk_cont(nor_index,imsres,ierrtime)

       if (ierrtime == 0) then

!         terminate load of calibration window

          DO kk=1,8
             data_vec(kk,i)=real(int_block(kk))
          ENDDO

          data_index(i)= nor_index
          data_exind(i)= ext_index

                          else
!         we have a data gap: we keep this new vector
!         and get a new series of N_Kern continuous vectors
!         without data gaps

          write(*,*) 'data gap at ',nor_index
          k_block=1

          DO kk=1,8
             data_vec(kk,1)=real(int_block(kk))
          ENDDO

          data_index(1)= nor_index
          data_exind(1)= ext_index

          call read_N_cont_vectors(N_Kern)

!         stop if EOF reached

          if (nor_index(1:3) == 'EOF' .or. &
              nor_index(1:3) == 'END') then
             exit Big_Loop
          endif

          exit

       endif

     END DO
   END IF   ! sur i_win > 1

  if (i_win > 10) then
     pr_out= .false.
     else
     pr_out= .true.
  endif

  if (i_win == 11) then
     write(*,*)
     write(*,*) '/// no more details on windows calibration after i_win=10 ///'
     write(*,*)
  endif

! loading array for data processing
! ---------------------------------


!    +++++++++++++++++++++++++++++++++
! c) Calibration of the current window
!    +++++++++++++++++++++++++++++++++


!    STEP 1 PROCESSING (always done)
!    ~~~~~~~~~~~~~~~~~

     if (pr_out) then
         write(*,*)
         write(*,1) 'Processing of calibration window #',i_win
         write(*,1) '---------------------------------------'
         write(*,*)
      endif

    1 format(1x,a,i3,1x,a)

     call w_com(pr_out,'1-Processing step 1:')

!    Conversion TM to Volt

     call w_com(pr_out,'  Conversion to Volt...')
     
!    correction des gains pour les voies xyz et despin
!    le signe - viens du montage des antennes

     proc_vec(1,:)= -(data_vec(1,:)*dyna -tm_offset)/data_vec(4,:)
     proc_vec(2,:)= -(data_vec(2,:)*dyna -tm_offset)/data_vec(5,:)
     proc_vec(3,:)= -(data_vec(3,:)*dyna -tm_offset)/data_vec(6,:)

     proc_vec(4,:)= -(data_vec(7,:)*dyna -tm_offset)/10.
     proc_vec(5,:)= -(data_vec(8,:)*dyna -tm_offset)/10.


!    STEP 2 PROCESSING
!    ~~~~~~~~~~~~~~~~~

 S2: IF (cal_step >= 2) then
!~~  
     call w_com(pr_out,'2-Processing step 2:')

!    Centering the waveforms on average value

     call w_com(pr_out,'  Centering...')
 
     proc_vec(1,:)= proc_vec(1,:) -sum(proc_vec(1,:))/real(N_Kern)
     proc_vec(2,:)= proc_vec(2,:) -sum(proc_vec(2,:))/real(N_Kern)
     proc_vec(3,:)= proc_vec(3,:) -sum(proc_vec(3,:))/real(N_Kern)
     proc_vec(4,:)= proc_vec(4,:) -sum(proc_vec(4,:))/real(N_Kern)
     proc_vec(5,:)= proc_vec(5,:) -sum(proc_vec(5,:))/real(N_Kern)
  
!    Removing spin signal

     call w_com(pr_out,'  Removing spin signal...')

     call desinus(proc_vec(1,:),N_Kern,sam_rate,spin_rate,mod_d1,pha_d1)
     call desinus(proc_vec(2,:),N_Kern,sam_rate,spin_rate,mod_d2,pha_d2)
     call desinus(proc_vec(3,:),N_Kern,sam_rate,spin_rate,mod_d3,pha_d3)
     call desinus(proc_vec(4,:),N_Kern,sam_rate,spin_rate,mod_d4,pha_d4)
     call desinus(proc_vec(5,:),N_Kern,sam_rate,spin_rate,mod_d5,pha_d5)

     if (pr_out) then
         write(*,*) '       Sample rate, N_Kern             : ', sam_rate, N_Kern
         write(*,*) '       Spin rate, Spin period          : ', spin_rate,real(spin_period)
         write(*,*) '       In Volt and degres :'
         write(*,*) '       Amplitude and phase removed on  X:',mod_d1,pha_d1
         write(*,*) '       Amplitude and phase removed on  Y:',mod_d2,pha_d2
         write(*,*) '       Amplitude and phase removed on  Z:',mod_d3,pha_d3
         write(*,*) '       Amplitude and phase removed on DX:',mod_d4,pha_d4
         write(*,*) '       Amplitude and phase removed on DY:',mod_d5,pha_d5
         write(*,*) 
     endif


!    Spin amplitude in nT, after transfert function correction

     Bperp_1= mod_d1/mod_sr1
     Bperp_2= mod_d2/mod_sr2
     Bperp_3= mod_d3/mod_sr3
     Bperp_4= mod_d4/mod_sr1
     Bperp_5= mod_d5/mod_sr2

!    Bperp_1 and Bperp_2 should be equal, assuming DC field constant
!    over window duration. Phase difference should be 90. degres.
!    To avoid modulation at twice spin frequency on DC field estimate,
!    one force theses assumptions.

     Bperp_A= (Bperp_4 +Bperp_5)/2.

     phacor_1= pha_d1 -pha_sr1
     phacor_2= pha_d2 -pha_sr2
     phacor_3= pha_d3 -pha_sr3
     phacor_4= pha_d4 -pha_sr1
     phacor_5= pha_d5 -pha_sr2

     diff= phacor_5 - phacor_4   ! must be > 0 and close to 90.

     pha_A= (phacor_4 +phacor_5 -90.)/2.

     if (pr_out) then
         write(*,*) '  Calibration of spin signal...'
         write(*,*) '  Spin amplitude and phase in nT and degres :'
         write(*,*) '       B perp. measured from Bx (nT):', Bperp_1, phacor_1
         write(*,*) '       B perp. measured from By (nT):', Bperp_2, phacor_2
         write(*,*) '       B per3. measured from Bz (nT):', Bperp_3, phacor_3
         write(*,*) '       B perp. measured from Dx (nT):', Bperp_4, phacor_4
         write(*,*) '       B per3. measured from Dy (nT):', Bperp_5, phacor_5
         write(*,*) ' '
         write(*,*) '       B perp. average          (nT):', Bperp_A
         write(*,*) '       Bxy phase difference     (nT):', diff
         write(*,*) '       Phase reference          (nT):', pha_A
     endif

!    correction of sample delay between x and y
!    corr_ret=corr_ret +retard_xy*4*float(N_shift)  ! (4 LSF)
!    very small, ignored.

!    Computation of misalignment angle (done at this step 2 only)

     call w_com(pr_out,'  Computation of misalignment angle...')

     R1_tmp= sqrt(Bperp_4**2 +Bperp_5**2 +Bperp_3**2)

     if (abs(R1_tmp).gt.1.e-10) then
         R2_tmp= Bperp_3*sqrt(2.)/R1_tmp
         if (abs(R2_tmp).le.1.) then
             misal_angle=asin(R2_tmp)*180./acos(-1.)
                               else
             misal_angle= 999.99
         endif
                                  else
         misal_angle=0.
     endif

     if (pr_out) then
         write(*,*) '       Z/Spin misalignment axis : ',misal_angle,' d.'
     endif


! Detrend used if given detrend frequency not=0 (see above)

  if (detrend) then
      call w_com(pr_out,'  Detrending waveforms...')
      write(work,*) nbp_liss
      call w_com(pr_out,'       Smoothering on N points='//trim(work))

!    process 3 xyz components

     do i=1,3

!       trend computation
        RA1_tmp= proc_vec(i,:)
        call lissage(RA1_tmp,N_Kern,nbp_liss,ierr)
        if (ierr /= 0) then
            RA1_tmp= 0.
        endif

!       substract trend to the original signal
        proc_vec(i,:)= proc_vec(i,:) -RA1_tmp
     end do
  else
     call w_com(pr_out,'  No detrending...')
  endif

! Centering again the waveforms on average value after desinus/detrend

  call w_com(pr_out,'  Centering again after despin/detrending...')
  
  proc_vec(1,:)= proc_vec(1,:) -sum(proc_vec(1,:))/real(N_Kern)
  proc_vec(2,:)= proc_vec(2,:) -sum(proc_vec(2,:))/real(N_Kern)
  proc_vec(3,:)= proc_vec(3,:) -sum(proc_vec(3,:))/real(N_Kern)

  END IF S2
! ~~~~~~~~~


!    STEP 3 PROCESSING
!    ~~~~~~~~~~~~~~~~~

 S3: IF (cal_step >= 3 .and. cal_step /= 6) then

     call w_com(pr_out,'3-Processing step 3:')
     call w_com(pr_out,'  Deconvolution. Calibration...')

!    Deconvolution :
!    waveform centering(done in step 2), weighting window, FFT, 
!    transfer function correction, filtering, inverse FFT
!    The input waveform must be despined before (see step 2)

!       weighting window

        CSpec(1,:)=cmplx(proc_vec(1,:)*Weight(:),0.)
        CSpec(2,:)=cmplx(proc_vec(2,:)*Weight(:),0.)
        CSpec(3,:)=cmplx(proc_vec(3,:)*Weight(:),0.)

!       time to frequency domain, transfer function correction, back to time domain

        call DECONVO_C3(CSpec,TFcor,N_Kern,DSS,DCS,M_Kern)

        proc_vec(1,:)=real(CSpec(1,:))
        proc_vec(2,:)=real(CSpec(2,:))
        proc_vec(3,:)=real(CSpec(3,:))

        do i=1,3

!         On fait une ponderation inverse sur les points gardes, pour ne pas
!         affecter le module et crer des harmoniques a N_Shift
!         que pour le cas Gaussien, avec N_Shift < N_Kern
!         ULF seulement.

          if(apod == 'g') then
             do j=i1,i2
                proc_vec(i,j)=proc_vec(i,j)/Weight(j)
             enddo
          endif

!         on charge le bloc calibre mais avec le passage a zero pour le 1er pt

          R1_tmp=proc_vec(i,1)
          R2_tmp=proc_vec(i,N_Kern)
          proc_vec(i,:)= proc_vec(i,:) -(R1_tmp +R2_tmp)/2.

          if (pr_out) then
              write(*,*) '  Force 1st and last point to be close to zero'
              write(*,*) '       pts(1)=', R1_tmp
              write(*,*) '       pts(N)=', R2_tmp
              call casinus( proc_vec(i,:),N_Kern,sam_rate,spin_rate, &
                           R1_tmp,R2_tmp,'       rest of spin on '//sensor(i))
          endif
        enddo

  END IF S3
! ~~~~~~~~~

!    set Euler angle at i=1

     read(data_exind(1),for_R_exti) iano,LSF,R1_tmp,Eul1,Eul2,Eul3,sensrot,itim,spin_phase

!    a rotation of +Eul1 in XY leads to a fixed system ZNY'  
!    N=Node common axis of the 2 planes (VD and XY) 
     alpha= Eul1
     
!    a second rotation of rotdeg in the same plane lead to the pseudo VDH system
!    where X is in the VH plane. Pseudo VDH is named SRV

!    compute rotation angle to get X axis in the meridian plane (SRV):
     sinrot=cos(Eul2*pisd)*sin(Eul3*pisd)
     cosrot=cos(Eul3*pisd)
     rotdeg= atan2(sinrot,cosrot)/pisd
     
     
!    STEP 4 PROCESSING
!    ~~~~~~~~~~~~~~~~~

 S4: IF (cal_step >= 4 .and. cal_step /= 6) then

     call w_com(pr_out,'4-Processing step 4:')
     call w_com(pr_out,'  Transformation Sensor System to SRV...')

     if (pr_out) then
        write(*,*) '       Index extension label = ', &
                           trim(manda_param%index_extension_label)
        write(*,*) '       Index extension format= ', &
                           trim(manda_param%index_extension_format)
        write(*,*) '       Spin phase= ',spin_phase
        write(*,*) '       F1,F2,fmax= ',f1,f2,fmax
        write(*,*) '       Spin rate, period= ', spin_rate,spin_period
        write(*,*) '       Sens rot      = ',sensrot
        write(*,*) '       rotdeg        = ',rotdeg
        write(*,*) '       Eul1,Eul2,Eul3=',Eul1,Eul2,Eul3
     endif

!    transformation to SRV

     call w_com(pr_out,'       Coordinate transformation...')

!    one process only the points that we will keep
     
     do i= i1,i2

!       phicr= -(spin_phase +0.)*pisd -mod(depif*float(i-ii),pi2) Cluster only
        read(data_exind(i),for_R_exti) iano,LSF,R1_tmp,Eul1,Eul2,Eul3,sensrot,itim,spin_phase
        phicr=  Eul1*pisd  +rotdeg*pisd
        sinphi=sin(phicr)
        cosphi=cos(phicr)
   
        R1_tmp= cosphi*proc_vec(1,i) -sinphi*proc_vec(2,i)
        R2_tmp= sinphi*proc_vec(1,i) +cosphi*proc_vec(2,i)

        proc_vec(1,i)=R1_tmp
        proc_vec(2,i)=R2_tmp
        
     enddo

  END IF S4
! ~~~~~~~~~


!    ELEMENTS FOR STEP 5,6,7
!    ~~~~~~~~~~~~~~~~~~~~~~~

S56: IF (cal_step >= 5 .and. cal_step /= 8) then

!    Computation of BX,By DC in the spin plane of SR2
!    (used both by L5, L6 and L7)
!    Bx,By are computed during a whole window of N_Kern pts

        phicr= alpha +rotdeg

!       on combine le champ mesuré par les despin et par les ULF
        
        Bdc_1= Bperp_1*sin((phacor_1-phicr)*pisd)
        Bdc_2= Bperp_2*sin((phacor_2-phicr)*pisd)
        
        Bdc_4= Bperp_4*0.52*sin((phacor_4-phicr)*pisd)
        Bdc_5= Bperp_5*0.52*sin((phacor_5-phicr)*pisd)

!       Bdc_x= -Bdc_4 -Bdc_1
!       Bdc_y=  Bdc_5 +Bdc_2
        
! la courbe est plus propre sans la faible contribution des ULF
        Bdc_x= -Bdc_4
        Bdc_y=  Bdc_5

  END IF S56
! ~~~~~~~~~~


!    STEP 5 PROCESSING
!    ~~~~~~~~~~~~~~~~~

 S5: IF (cal_step == 5 ) then

     call w_com(pr_out,'5-Processing step 5:')
!
     if (pr_out) then
        write(*,*) '       Adding DC constant in fixed system'
        write(*,*) '       B DC added on x (nT):', Bdc_x
        write(*,*) '       B DC added on y (nT):', Bdc_y
        write(*,*) '       Total Bperp         :', Bperp_A
     endif

     do i= i1,i2
        proc_vec(1,i)= proc_vec(1,i) +Bdc_x
        proc_vec(2,i)= proc_vec(2,i) +Bdc_y
     enddo

  END IF S5
! ~~~~~~~~~


!    STEP 6 PROCESSING
!    ~~~~~~~~~~~~~~~~~

 S6: IF (cal_step == 6 ) then

     call w_com(pr_out,'6-Processing step 6:')
     call w_com(pr_out,'      Control mode for DC field estimation')
     call w_com(pr_out,'      Step 3,4,5 has been skipped')
!
     if (pr_out) then
        write(*,*) '       Write DC field on a VecTime file'
        write(*,*) '       B DC  x (nT):', Bdc_x
        write(*,*) '       B DC  y (nT):', Bdc_y
        write(*,*) '       Total Bperp :', Bperp_A
        write(*,*) '       misAlignemnt:', misal_angle
     endif
!
!    only one point for each windows

     manda_param%block_number= N_win
!
     proc_vec(1,i1)= Bdc_x
     proc_vec(2,i1)= Bdc_y
     proc_vec(3,i1)= misal_angle

  END IF S6
! ~~~~~~~~~

!    STEP 7 PROCESSING
!    ~~~~~~~~~~~~~~~~~

 S7: IF (cal_step == 7 ) then

     call w_com(pr_out,'7-Processing step 7:')
!
     if (pr_out) then
        write(*,*) '       Same on step 4 but adding DCx, DCy on separate fields (SRV)'
        write(*,*) '       B DC on x (nT):', Bdc_x
        write(*,*) '       B DC on y (nT):', Bdc_y
        write(*,*) '       Total Bperp   :', Bperp_A
     endif
     
     do i= i1,i2
        proc_vec(4,i)=  Bdc_x
        proc_vec(5,i)= -Bdc_y
     enddo

  END IF S7
! ~~~~~~~~~

!    STEP 8 PROCESSING
!    ~~~~~~~~~~~~~~~~~

 S8: IF (cal_step == 8 ) then
         write(*,*)
         write(*,'(a)') ' -> ERROR vectime_calibration_GEOULF: NO VALID CAL_STEP ***'
         write(*,'(a)') ' -> PROGRAM ABORTED !!!'
         close(UNIT=in_file_unit)
     stop 'vectime_calibration_GEOULF.exe  : *** ERROR !! Program aborted !'

! pas fait; sorti sur erreur

!    call w_com(pr_out,'8-Processing step 8:')

!    transform SR2 to GSE
!
!    if (pr_out) then
!       write(*,*) '       Same on step 4 but data in GSE'
!    endif

!    if (spin_phase >= -360.) then
!        GSE spin axis is supposed to not change during a spectra (a few seconds)
!        to reduce CPU time (one day~ 360°/365, so 1 mn ~ 6 E-4 deg.)

!        Rocotlib time dependant matrix computation

!        call decode_datiso(data_index(i1),iyear,imon,iday,ih,im,is,ims,imc)
!        call ctimpar(iyear,imon,iday,ih,im,is)

!        spin axis in gse

!        call tgeigse(sxgei,sygei,szgei,sxgse,sygse,szgse)

!        if (pr_out) then
!           write(*,'(1x,a,3f10.4)') 'Spin dans le gse, x  y  z  : ',sxgse,sygse,szgse
!        endif

!        do i= i1,i2

!           transform data in GSE

!           call tsr2gse(proc_vec(1,i),proc_vec(2,i),proc_vec(3,i),sxgse,sygse,szgse, &
!                        R1_tmp,R2_tmp,R3_tmp)

!           proc_vec(1,i)=R1_tmp
!           proc_vec(2,i)=R2_tmp
!           proc_vec(3,i)=R3_tmp

!        enddo
!                            else
!       proc_vec(1,:)=-0.1000E+31
!       proc_vec(2,:)=-0.1000E+31
!       proc_vec(3,:)=-0.1000E+31
!    endif

!    ncomp=3

  END IF S8
! ~~~~~~~~~


!    ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! d) Store in temporary file kept blocs from the calibration window
!    ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


     do j= 1,N_Shift

        if(cal_step == 6 .and. j == 1) then
          nor_index= data_index(N_Kern/2)
          ext_index= data_exind(N_Kern/2)
          flt_block(1:ncomp)= proc_vec(1:ncomp,i1)
          o_block= o_block +1
          write(tmp_file_unit,ou_block_fmt) trim(nor_index), &
                                            trim(ext_index), &
                                            flt_block(1:ncomp)
          exit
        endif

        jj= i1 +j -1

        nor_index= data_index(jj)
        ext_index= data_exind(jj)
        flt_block(1:ncomp)= proc_vec(1:ncomp,jj)
        o_block= o_block +1
        write(tmp_file_unit,ou_block_fmt) trim(nor_index), &
                                          trim(ext_index), &
                                          flt_block(1:ncomp)
     enddo

     last_index=nor_index


! =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
! End loop on all shifted windows
! =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

  END DO Big_Loop

  call print_date_time('    End       of big calibration lopp')

  write(*,*)
  write(*,*) 'All blocks are read and written'
  write(*,*) '-------------------------------'
     
  
! 3) Memory block desallocation
!    ==========================================

  write(*,*) 'Deallocate all arays'

  write(*,*) '  Data Vector for window acquisition         : ', i, N_Kern
                deallocate(data_vec,stat=ierr)

  write(*,*) '  Data Vector for calibration processing     : ', i, N_Kern
                deallocate(proc_vec,stat=ierr)

  write(*,*) '  Float   working 1D array dimension         : ', N_Kern
                deallocate(RA1_tmp,stat=ierr)

  write(*,*) '  Float   working 1D array dimension         : ', N_Kern
                deallocate(RA2_tmp,stat=ierr)

  write(*,*) '  Float   working 1D array dimension         : ', N_Kern
                deallocate(RA3_tmp,stat=ierr)

  write(*,*) '  Float   Weight  1D array dimension         : ', N_Kern
                deallocate(Weight,stat=ierr)

  write(*,*) '  Complex working 2D array dimension         : ', N_Kern
                deallocate(TFcor,stat=ierr)

  write(*,*) '  Complex working 2D array dimension         : ', N_Kern
                deallocate(CSpec,stat=ierr)

  write(*,*) '  Character 1D array dimension for index     : ', N_Kern
                deallocate(data_index,stat=ierr)

  write(*,*) '  Character 1D array dimension for ext_index : ', N_Kern
                deallocate(data_exind,stat=ierr)
   
  write(*,*) '  Float   sine table 1D array dimension      : ', M_Kern
                deallocate(DSS,stat=ierr)

  write(*,*) '  Float cosine table 1D array dimension      : ', M_Kern
                deallocate(DCS,stat=ierr)

  write(*,*) 'Deallocation done'
  
  call print_date_time('End of part IV')
  
  
  
! ------------------------------------------------------!  
!  V )  C H E C K   N O R M A L   T E R M I N A T I O N !
! ------------------------------------------------------!

! connection for empy or incomplete files
! ---------------------------------------

  99 continue
  
! Closing input file

  close(in_file_unit)

  write(*,*)
  write(*,*) 'CHECKING NORMAL TERMINATION :'
  write(*,*) '-----------------------------'


! 1) check if bloc number read is the same as expected
!    =================================================

  write(*,*)
  write(*,*) 'Number of blocks of input file: ', block_number1
  write(*,*) 'Number of blocks to be read   : ', N_read
  write(*,*) 'Number of blocks in fact read : ', i_block
  write(*,*) 'Number of blocks missing      : ', N_read-i_block

  if (i_block/=N_read) then
     write(*,*)
     if(i_block < N_read) write(*,'(a)') ' -> There was unread blocks !!!'
     if(i_block > N_read) write(*,'(a)') ' -> There was useless blocks !!!'
                       else
     write(*,*) 'OK, all requested blocks has been read'
  endif


! 2) check if bloc number written is the same as expected
!    ====================================================

  write(*,*)
  write(*,*) 'Number of blocks expected to be written : ', &
                    manda_param%block_number
  write(*,*) 'Number of blocks effectively written    : ',o_block
  write(*,*) 'Number of blocks missing                : ', &
                    manda_param%block_number -o_block


  if (o_block/=manda_param%block_number) then
     write(*,*)
     write(*,'(a)')   ' -> There has been data gaps '
                       else
     write(*,*) 'OK, no data gap'
  endif

  call print_date_time('End of part V')

  
! ------------------------------------------------------!
! VI )  C R E A T E   O U T P U T   R F F  F I L E      !
! ------------------------------------------------------!

! 1) update first & last index, and block written
!    ============================================

  manda_param%block_first_index=trim(first_index)
  manda_param%block_last_index=trim(last_index)
  manda_param%block_number = o_block

! change INDEX_EXTENSION_FORMAT (compatibilite CLUSTER visu_vectime)

  manda_param%index_extension_format= '(a46,1x,f7.2)'
  write(*,*)'new index_extension_format=', trim(manda_param%index_extension_format)


! 2) Opening output_file and writing header, metadata and constant_data
!    ==================================================================

  write(*,*)
  write(*,*) 'OPENING RFF OUTPUT FILE & WRITING NEW METADATA....'
  write(*,*) '--------------------------------------------------'

  call rff_W_metadata(ou_file_unit,ou_file_name)

  write(*,*)
  write(*,*) 'First block index from Output Mand. Param. : ', &
              trim(manda_param%block_first_index)
  write(*,*) 'Last  block index from Output Mand. Param. : ', &
              trim(manda_param%block_last_index)

  write(*,*) 'Done...'

! writting constant data with new one

  write(*,*)
  write(*,*) 'WRITING OLD CONSTANT DATA and NEW ONES...'
  write(*,*) '-----------------------------------------'

  call rff_W_const_data(ou_file_unit)

  write(*,*) 'Done...'


! 2) complete output_file by indexed data
!    ====================================

  write(*,*)
  write(*,*) 'WRITING INDEXED DATA...'
  write(*,*) '-----------------------'

  write(ou_file_unit,'(80a)') '#',('-',i=1,79)
  write(ou_file_unit,'(a)') 'START INDEXED_DATA'

  rewind(tmp_file_unit)

  do j=1,o_block 
     read(tmp_file_unit,'(a)') work
     write(ou_file_unit,'(a)') trim(work)
  end do

  close(tmp_file_unit)

  write(*,*) 'Done...'


! 3) terminate the output_file and close
!    ===================================

  write(*,*)
  write(*,*) 'TERMINATE AND CLOSE RFF OUTPUT FILE'
  write(*,*) '-----------------------------------'

  call rff_W_tail(ou_file_unit)

  write(*,*) 'Done...'

  call print_date_time('End of part VI')

  
!---------------!
! VII )   E N D !
!---------------!
     
  write(*,*)
  write(*,'(a)')  ' -> INFO : vectime_calibration_GEOULF / NORMAL TERMINATION'

  if(manda_param%block_number==0) then
     write(*,'(3a)') '           file ', trim(ou_file_name), &
                     ' created with no data'
     write(*,*)
     write(*,'(a)')  'XXXXXXXXXX  E N D XXXXXXXXXX'
     stop 'vectime_calibration_GEOULF.exe   : NORMAL TERMINATION BUT NO DATA'
     else
     write(*,'(3a)') '           file ', trim(ou_file_name), ' created'
     write(*,*)
     write(*,'(a)')  'XXXXXXXXXX  E N D XXXXXXXXXX'
     print*, 'vectime_calibration_GEOULF.exe   : NORMAL TERMINATION'
        stop 'vectime_calibration_GEOULF.exe   : NORMAL TERMINATION'
  endif

  call print_date_time('End of part VII')

!//////////////////////////////////////////////////////////////////////!
!
! INTERNAL SUBROUTINES
!
!//////////////////////////////////////////////////////////////////////!

contains

  subroutine read_N_cont_vectors(N_Vectors)

! ------------------------------------------------------------------
! read a suit of N_Vectors without data gaps
! ------------------------------------------------------------------


  integer :: N_Vectors

  k_block= 0

  DO WHILE (k_block /= N_Vectors)

!    reading block from input file
!    ----------------------------

     int_block(:)=fill_val_int

     call r_rff_data_block_int(in_file_unit,nor_index,ext_index,int_block)

!    stop if EOF reached

     if (nor_index(1:3) == 'EOF' ) then
        write(*,'(a)') ' -> WARNING : vectime_calibration_GEOULF: EOF REACHED ***'
        write(*,'(a,i8)')      '    at k_block=', k_block
        write(*,'(a,i8)')      '    for iblock=', i_block
        write(*,'(a,i8,a,i8)') '    for  i_win=', i_win,' i=',i
        exit
     endif

     if ( nor_index(1:3) == 'END') then
        write(*,'(a)') ' -> WARNING : vectime_calibration_GEOULF: END REACHED ***'
        write(*,'(a,i8)')      '    at k_block=', k_block
        write(*,'(a,i8)')      '    for iblock=', i_block
        write(*,'(a,i8,a,i8)') '    for  i_win=', i_win,' i=',i
        exit
     endif

     i_block=i_block +1

! *  test on block continuity

     call test_blk_cont(nor_index,imsres,ierrtime)

     if(ierrtime == 0) then
!                      OK, block counter +1
                       k_block=k_block+1
                       else
                       k_block=1
     endif

!    load calibration window and index info

     DO kk=1,8
        data_vec(kk,k_block)=real(int_block(kk))
     ENDDO

     data_index(k_block)= nor_index
     data_exind(k_block)= ext_index

  END DO

  return

  end subroutine read_N_cont_vectors

!//////////////////////////////////////////////////////////////////////!

  subroutine r_rff_data_block_int(in_file_unit,nor_index,ext_index,int_block)

! ------------------------------------------------------------------
! read a data block integer of the vectime RFF file: GEOS => dim8
! ------------------------------------------------------------------

  integer :: in_file_unit, ieof,ipos1,ipos2
  integer, dimension(8) :: int_block
  character(len= *) :: nor_index, ext_index
  character(len=255) :: work

  READ(in_file_unit,'(a)', iostat=ieof) work
  
  IF(ieof /= 0) THEN
              print*, '*   Warning ! eof reached on unit ',in_file_unit
              nor_index='EOF'
              RETURN
  ENDIF

  IF(work(1:4) == 'END ') THEN
              print*, '*   Warning ! end of indexed data on unit ',in_file_unit
              print*, '    ',trim(work)
              nor_index='END '
              RETURN
  ENDIF


! find where the ISO time ends
  ipos1=INDEX(work,'Z')

  IF(ipos1 == 0) THEN
         print*, '*** Error, no Z terminator in ISO time index'
         print*, '    at i_win=',i_win,' i=',i
         print*, '    Program aborted'
         stop 'vectime_calibration_GEOULF:.exe  : *** ERROR !! Program aborted !'
  ENDIF

! save ISO time
! -------------

  nor_index=work(1:ipos1)

! save extented index
! -------------------

  IF (manda_param%INDEX_EXTENSION_LENGTH == 0) THEN
     ext_index=' '
     ipos2=ipos1+2
  ELSE
     ipos2=ipos1 +1 +manda_param%INDEX_EXTENSION_LENGTH +2
     ext_index=work(ipos1+2:ipos2-2)
  ENDIF
 
! save vector
! -----------

! read(work(ipos2:255),in_data_format,iostat=ierr) int_block 
! sous gfortran le format de lecture in_data_format=(I5,",",I5,",",I5) ne marche pas
! seul x est lu
! P. Robert Fev 2018

  read(work(ipos2:255),*,iostat=ierr) int_block
 
  return

  end subroutine r_rff_data_block_int

!//////////////////////////////////////////////////////////////////////!

  end program vectime_calibration_GEOULF

!XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX!
!
! EXTERNAL SUBROUTINES
!
!XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX!

  subroutine read_input(in_file_name,ou_file_name, &
                        fdet,fc,f1,f2,cal_step,N_Kern, N_Shift,apod)

!----------------------------------------------------------------------!
! Reading input parameters for vectime_calibration_GEOULF program
!----------------------------------------------------------------------!

  character*(*) in_file_name, ou_file_name, apod
  integer cal_step
  integer :: N_Kern, N_Shift
  real (kind=4) fdet,fc,f1,f2

!                    *********************

! *** exemple de fichier vectime_calibration_GEOULF.in

!   GEOS1_ULF_VTL1.rff
!   GEOS1_ULF_VTL2.rff
!   /users/.../RPC/CLUSTER_STAFF-SC_Nbr4.cal
!   0.0           # detrend frequency (0. for classis despin)
!   0.1           # Frequency cut-off for calibration
!    0.000  00.00 # Min/Max frequencies for filtering (Max=0 => Nyquist)
!   4             # Processing steps asked (1-6)
!   512           # N_Kern = Kernel size
!   4             # N_shift for sliding window
!   trapezium     # trapezium (t) or Gaussian (g)
!
!      # Comments on processing steps:
!      # 1: Volts,  spinning  system, with    DC field
!      # 2: Volts,  spinning  system, without DC field
!      # 3: nTesla, spinning  system, without DC field
!      # 4: nTesla, fixed SR2 system, without DC field
!      # 5: nTesla, fixed SR2 system, with    DC field
!      # 6: nTesla, fixed SR2 system, only DC-field
!      # 7: nTesla, fix. SRV system + Dx,Dy in sperate fields
!      # 8: nTesla, fixed GSE system, without DC field ***/not done


! *   lecture du path du fichiers N1

  write(*,*) 'nom du fichier de donnees N1 ?'
  write(*,*) '(path complet)'
  read (*,'(a)')  in_file_name
  write(*,*) trim(in_file_name)

! *   lecture du path du fichiers N2

  write(*,*) 'Name of RFF output file ?'
  write(*,*) '(path complet)'
  read (*,'(a)')  ou_file_name
  write(*,*) trim(ou_file_name)

! *   frequence de detrend repere tournant

  write(*,*) 'frequence de detrend pour le despinage special ? [0.3]'
  read (*,*)  fdet
  write(*,*)  fdet

! *   frequence de coupure repere tournant

  write(*,*) 'frequence de coupure pour la calibration ? [0.1]'
  read (*,*)  fc
  write(*,*)  fc

! *   bornes en frequences pour le filtrage repere fixe

  write(*,*) 'fmin, fmax pour filtrage en repere fixe [0.2, 0.]'
  write(*,*) '(0. pour f2 deviendra fmax selon le bit rate)'
  read (*,*)  f1,f2
  write(*,*)  f1,f2

! *   niveaux de traitements

  write(*,*) 'niveaux de traitements  pour le .resu ? [1-6]'
  read (*,*) cal_step
  write(*,*) cal_step

! *   Kernel size

  write(*,*) 'Kernel size ? (must ne M**N, ex : 512)'
  read (*,*) N_Kern
  write(*,*) N_Kern

! *   Shift size and NBP kept in the kernel

  write(*,*) 'N_shift ? (ex: 4)'
  read (*,*) N_Shift
  write(*,*) N_Shift

! *   Weighting function for calibration (trapezium or gaussian)

  write(*,*) 'Name of Weighting function for calibration ? '// &
             '(trapezium or gaussian)'
  read (*,'(a)')  apod
  write(*,*) trim(apod)


  write(*,*) ' '
  write(*,*) 'resume des parametres d''entree:'
  write(*,*) 'fichier VTL1 :'
  write(*,*) trim(in_file_name)
  write(*,*) 'fichier VTL2 :'
  write(*,*) trim(ou_file_name)
  write(*,*) 'parametres d''entree:'
  write(*,*) 'Fc, F1,F2=',fc,f1,f2
  write(*,*) 'Cal step=',cal_step
  write(*,*) 'N_Kern, N_Shift =',N_Kern, N_Shift
  write(*,*) trim(apod)

  return
  end

!XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX!
!
      complex function gainGEOS(f)
!
!     ------------------------------------------------------------------
! **  categorie: depouillement specifique GEOS/UBF
! **  objet    : fonction gain complexe des antennes magnetiques geos
! **  auteur   : P. Robert CRPE 1977-1984, revu P.R. 1998
!     ------------------------------------------------------------------
!
      complex j,a,b,c
!
!                    *********************
!
!
      j=cmplx(0.,1.)
!
      f1=  0.01
      f2= 31.
      fc=  5.
      ga=0.417
!
      a=(j*f/f1)/(1.+j*f/f1)
      b=(j*f/f2)/(1.+j*f/f2)
      c=(1.+j*f/fc)*(2./(1.+j*sqrt(3.))+j*f/fc)
      c=c*(2./(1.-j*sqrt(3.))+j*f/fc)
!
      gainGEOS=cmplx(ga,0.)*a*b/c
!
      return
      end
!
!     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
!
      subroutine correch(s,nbp,fe,j)
!
!     ------------------------------------------------------------------
! **  categorie: depouillement specifique GEOS/UBF
! **  objet    : correction des retards d echantillonage des voies xyz
! **  auteur   : P. Robert CRPE 1977-1984
! **  revision : P. Robert, ScientiDev, Fev. 2019
!     ------------------------------------------------------------------
!
      complex s(nbp)
      dimension taur(3)
!
      data taur/0.0188,0.0195,0.0296/
!
!                    *********************
!
!
      t=nbp/fe
      nbp2=nbp/2
!
      do  i=1,nbp
          if (i.gt.nbp2) n=i-1-nbp
          if (i.le.nbp2) n=i-1
          teta=2.*3.1415927*n*taur(j)/t
          s(i)=s(i)*cmplx(cos(teta),sin(teta))
      enddo
!
      return
      end
!
!     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
!
      subroutine VDH_to_xyz(v,d,h,a1,a2,a3,x,y,z)
!
!     ------------------------------------------------------------------
! **  categorie: astronomie et changement de base
! **  objet    : passage du repere vdh  au repere satellite tournant
! **  auteur   : P. Robert CRPE 1977-1984
!     ------------------------------------------------------------------
!
!                    *********************
!

      pird=3.1415926
!
      a1r=a1*pird
      a2r=a2*pird
      a3r=a3*pird
!
      c1=cos(a1r)
      c2=cos(a2r)
      c3=cos(a3r)

      s1=sin(a1r)
      s2=sin(a2r)
      s3=sin(a3r)
!
!     calcul de 1977-84
      x=( c1*c3 -s1*c2*s3)*v  +( c1*s3 +s1*c2*c3)*d +s1*s2*h
      y=(-s1*c3 -c1*c2*s3)*v  +(-s1*s3 +c1*c2*c3)*d +c1*s2*h
      z=             s2*s3*v               -s2*c3*d    +c2*h
!
! matrice from wiki 
      x=( c1*c3 -s1*c2*s3)*v  +(-c1*s3 -s1*c2*c3)*d +s1*s2*h
      y=( s1*c3 +c1*c2*s3)*v  +(-s1*s3 +c1*c2*c3)*d -c1*s2*h
      z=             s2*s3*v  +             s2*c3*d +   c2*h

! donc matrice directe different de 1977
!     x=( c1*c3 -s1*c2*s3)*v  +( s1*c3 +c1*c2*s3)*d +s2*s3*h
!     y=(-c1*s3 -s1*c2*c3)*v  +(-s1*s3 +c1*c2*c3)*d +s2*c3*h
!     z=             s1*s2*v               -c1*s2*d +   c2*h
!
!     Si a2 vaut 0 ou 180. les 3 formules sont identiques pour V
!
      return
      end
!
!     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
!
      subroutine xyz_to_VDH(x,y,z,a1,a2,a3,v,d,h)
!
!     ------------------------------------------------------------------
! **  categorie: astronomie et changement de base
! **  objet    : passage du repere satellite tournant au repere VDH
! **  auteur   : P. Robert ScientiDev, Feb. 2019
!     ------------------------------------------------------------------
!
!                    *********************

      pird=3.1415926
!
      a1r=a1*pird
      a2r=a2*pird
      a3r=a3*pird
!
      c1=cos(a1r)
      c2=cos(a2r)
      c3=cos(a3r)

      s1=sin(a1r)
      s2=sin(a2r)
      s3=sin(a3r)

      v=( c1*c3 -s1*c2*s3)*x  +( s1*c3 +c1*c2*s3)*y +s2*s3*z
      d=(-c1*s3 -s1*c2*c3)*x  +(-s1*s3 +c1*c2*c3)*y +s2*c3*z
      h=             s1*s2*x               -c1*s2*y +   c2*z

      return
      end
!
!     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
!
