; BEGIN ROCOTLIB 3.2

; XX XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
; XX                                                                  XX
; XX   Robert's Coordinates Transformation Library  ROCOTLIB          XX
; XX                                                                  XX
; XX            Revised  version 3.2, February 2020                   XX
; XX                    ___________________                           XX
; XX                                                                  XX
; XX             Version 1.0 initially supported by                   XX
; XX                                                                  XX
; XX                  EUROPEAN  SPACE  AGENCY                         XX
; XX                                                                  XX
; XX               Study of the Cluster Mission                       XX
; XX                 Planning Related Aspects                         XX
; XX          within the Numerical Simulations Network                XX
; XX                                                                  XX
; XX            Patrick ROBERT, CRPE, November 1992                   XX
; XX                    ___________________                           XX
; XX                                                                  XX
; XX Further versions developped by the Author:                       XX
; XX                                                                  XX
; XX    version 1.0, November 1992                                    xx
; XX    version 1.1, July     1993                                    xx
; XX    version 1.2, January  1995                                    xx
; XX    version 1.3, July     2000 (Julian day 2000 et sun dir)       xx
; XX    version 1.4, June     2001 (for automatic documentation)      xx
; XX    version 1.5, December 2001 (adding cp_sunrise_sunset)         xx
; XX    version 1.6, Juin     2002 (upgrade IGRF -> 2005)             xx
; XX    version 1.7, December 2002 (Version for CDPP)                 xx
; XX    version 1.8, November 2003 (adding t_sr2_to_sr)               xx
; XX    version 1.9, March    2004 (compatibility with IDL)           xx
; XX    version 2.0, November 2006 (Update  IGRF -> 2010)             xx
; XX    version 2.1, November 2010 (Update  IGRF -> 2015)             xx
; XX    version 2.2, December 2011 (cp_sunrise_sunset on polar zone   xx
; XX    version 3.0, February 2016 (IGRF -> table of geomagnetic pole XX
; XX                                -> 2020 + some other coord. sys.) XX
; XX    version 3.1, January  2019 (add name compatibility with V2.2) XX
; XX    version 3.2, February 2020 (add trans. with Euler angles)     XX
; XX                                                                  XX
; XX Copyright 1992, Patrick ROBERT, CNRS-ESA, All Rights reserved    XX
; XX                    ___________________                           XX
; XX                                                                  XX
; XX         For details, see the orginal document untitled:          XX
; XX CLUSTER Software Tools, part I: Coordinate Transformation LibraryXX
; XX             Document de travail DT/CRPE/1231                     XX
; XX             Patrick Robert, CRPE/TID, Juillet 1993               XX
; XX                                                                  XX
; XX         Available in CDPP:                                       XX
; XX             ROCOTLIB: a coordinate Transformation Library        XX
; XX             for Solar-Terrestrial studies                        XX
; XX             Patrick ROBERT, version 1.7 - January 2003,          XX
; XX             Rapport Interne no RI-CETP/02/2003                   XX
; XX                    ___________________                           XX
; XX                                                                  XX
; XX         The previous version 3.0 contains new transformations     XX
; XX         (TPN, MVA), and the cp_sunrise_sunset module allowing    XX
; XX         computation of sunrise and sunset anywhere on Earth,     XX
; XX         including polar zones.                                   XX
; XX                                                                  XX
; XX         The determination of the dipole axis direction           XX
; XX         has been replaced by a table available from years        XX
; XX         1900 to 2020, in order to have an easy way to code       XX
; XX         maintenance in the coming years.                         XX
; XX                                                                  XX
; XX         Operations on matrix (somme, product, inversion,         XX
; XX         computation of eigen vectors, diagonalization...)        XX
; XX         required for Minimum Variance Analysis coordinates       XX
; XX         have been added with prefix "mat_".                      XX
; XX                                                                  XX
; XX         The previous and temporary V3.1 is the same as 3.0       XX
; XX         but is compatible with V2.2 version concerning           XX
; XX         PRO name , ex: tmaggsm -> t_mag_to_gsm            XX
; XX                                                                  XX
; XX         The present V3.2 version contains transformations        XX
; XX         with Euler angles and their interpolation.               XX
; XX                                                                  XX
; XX         Fortran 90, C, and IDL codes are also available.         XX
; XX                                                                  XX
; XX                                                                  XX
; XX XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX


;     ***************************************************************0^

;     Reminder FOR the developper:
;     Any changes leadind to a new version implies following actions:
;        - Update the file header above
;        - Update the PRO g_rocot_version_number
;        - Update the PRO PRINT_rocot_info
;        - Update the readme.txt file in the full package

;     ***************************************************************0^

;     floor for f90 and f77 portability

      FUNCTION Ifloor, a
      
      ap=FIX(a)
      an=FIX(a)-1
      
      IF (a LT 0.) THEN RETURN, an ELSE RETURN, ap
      
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx      
      
;     modulo for f77 and portability
;
      FUNCTION Fmodulo, a,p
      
      rr= a/p
      
      IF(TYPENAME(rr) EQ 'FLOAT') THEN RETURN, a -Ifloor(a/p)*p $
                                  ELSE RETURN, a -Ifloor(FLOAT(a)/FLOAT(p))*p

      END 

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx
;
;     mod for portability
;
      FUNCTION Fmod, a,p

      RR= a - (FIX(a/p)*p)
      RETURN, RR
      END 
;
;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx
;
;     Tsign for portability
;
      FUNCTION Tsign, a,p

      RR= a*ABS(p)/p
      RETURN, RR
      END 
;
;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx
;
;     Dprod for portability
;
      FUNCTION Dprod, x,y

      RR= FLOAT(DOUBLE(x)*DOUBLE(y))
      RETURN, RR
      END 
;
;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO cp_angle_and_ratio, ux,uy,uz,vx,vy,vz,angle,ratio

; ----------------------------------------------------------------------
; *   Class  : basic compute modules of Rocotlib Software
; *   Object : compute_angle_and_ratio beetween U and V vectors
; *   Author : P. Robert, CRPE, 1992

; *   Input  : ux,uy,uz
;              vx,vy,vz

; *   Output : sp=U.V
;              angle=angle beetween U and V (radians)
;              ratio= mod(U)/mod(V)
; ----------------------------------------------------------------------


;     double precision u1,u2,u3,v1,v2,v3,dp,ru,rv,cot

      u1=DOUBLE(ux)
      u2=DOUBLE(uy)
      u3=DOUBLE(uz)
      v1=DOUBLE(vx)
      v2=DOUBLE(vy)
      v3=DOUBLE(vz)

      dp= u1*v1 + u2*v2 + u3*v3
      ru= SQRT(u1*u1 + u2*u2 + u3*u3)
      rv= SQRT(v1*v1 + v2*v2 + v3*v3)
      cot=dp/(ru*rv)
      cot=cot -Tsign(1.d-13,cot)
      ratio=FLOAT(ru/rv)

     IF(cot GE 1.d0)  THEN BEGIN
                 PRINT, ' *** Rocotlib/cp_angle_and_ratio: cos > 1 !!!'
                 PRINT, '                       angle set to 0.'
                 angle=0.
                 RETURN
                 ENDIF

     IF(cot LT -1.d0)  THEN BEGIN
                 PRINT, ' *** Rocotlib/cp_angle_and_ratio: cos < 1 !!!'
                 PRINT, '                       angle set to 0.'
                 angle=0.
                 RETURN
                 ENDIF

      angle=FLOAT(ACOS(cot))

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cp_Euler_interp, a1,b1,c1,a2,b2,c2,ti,dt,ai,bi,ci

; ----------------------------------------------------------------------
; *   Class  : basic compute modules of Rocotlib Software
; *   Object : compute_Euler_angles_interpolation
; *   Author : P. Robert, SDev, 2020

; *   Input  :  a1,b1,c1 Euler angles at time t1
;               a2,b2,c2 Euler angles at time t2
;               ti time FOR interpolation t1 < ti < t2
;               dt = t2 - t1

; *   Output :  ai,bi,ci Euler angles interpolated at time ti
; ----------------------------------------------------------------------


      Diff_a = a2 -a1
      Diff_b = b2 -b1
      Diff_c = c2 -c1

;     WARNING: a does not vary linearly, it turns fast
;     while b and c vary slowly, $
;     so we can FOR the linear interpolation on 2 and 3 DO BEGIN
;     but not on a
 
      Diff_a= Fmodulo(Diff_a,360.)
  
;     FOR 2 and 3 management of zero crossings
 
      IF(ABS(Diff_b) > 180.)  THEN BEGIN
              IF(Diff_b < 0)  THEN BEGIN
                                Diff_b= Diff_b +360.
                              ENDIF ELSE BEGIN
                                Diff_b= Diff_b -360.              
             ENDELSE
      ENDIF

      IF(ABS(Diff_c) > 180.)  THEN BEGIN
              IF(Diff_c < 0)  THEN BEGIN
                                Diff_c= Diff_c +360.
                              ENDIF ELSE BEGIN
                                Diff_c= Diff_c -360.              
              ENDELSE
      ENDIF 

;     interpolate Euler angles

      ai = a1 +Diff_a*ti/dt
      bi = b1 +Diff_b*ti/dt
      ci = c1 +Diff_c*ti/dt
    
      ai = Fmodulo(a2,360.)
      bi = Fmodulo(b2,360.)
      ci = Fmodulo(c2,360.)

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cp_geo_dipole_dir, iyear,idoy,d1,d2,d3

; ----------------------------------------------------------------------
; *   Class  : basic compute modules of Rocotlib Software
; *   Object : compute_dipole_direction in GEO system
; *   Author : P. Robert, LPP , 2016

; *   Input  :  iyear (1900 - 2020), idoy= day of year (1/1=1)
; *   Output :  d1,d2,d3  cartesian dipole components in GEO system
; ----------------------------------------------------------------------

      COMMON dipdir, iy,id,ipr, d1p,d2p,d3p

; *** table of geomagnetic Nort pole
;     International Geomagnetic Reference Field: the 12th generation, $
;     Thbault et al. Earth, Planets and Space (2015) 67:79 , $
;     DOI 10.1186/s40623-015-0228-9

;     geomagnetice pole (N & S) are symmetric, so there are used to
;     define dipole axis. Do not mix up within the the North and south
;     magnetic pole (not symetric).


      year=     [1900.0, 1905.0, 1910.0, 1915.0, 1920.0, 1925.0, 1930.0, $
                 1935.0, 1940.0, 1945.0, 1950.0, 1955.0, 1960.0, 1965.0, $
                 1970.0, 1975.0, 1980.0, 1985.0, 1990.0, 1995.0, 2000.0, $
                 2005.0, 2010.0, 2015.0, 2020.0]

      rlat=     [ 78.68, 78.68, 78.66, 78.64, 78.63, 78.62, 78.60, 78.57 $
                , 78.55, 78.55, 78.55, 78.54, 78.58, 78.60, 78.66, 78.76 $
                , 78.88, 79.04, 79.21, 79.39, 79.61, 79.82, 80.09, 80.37 $
                , 80.65]

      rlon=     [-68.79,-68.75,-68.72,-68.57,-68.38,-68.27,-68.26,-68.36 $
                ,-68.51,-68.53,-68.85,-69.16,-69.47,-69.85,-70.18,-70.47 $
                ,-70.76,-70.90,-71.13,-71.42,-71.57,-71.81,-72.21,-72.63 $
                ,-73.17]

;                        ------------------------

      IF(ISA(ipr,'Undefined ', /INTEGER) EQ 0) THEN ipr= -1
      IF(ISA(iy ,'Undefined ', /INTEGER) EQ 0) THEN iy = -1
      IF(ISA(id ,'Undefined ', /INTEGER) EQ 0) THEN id = -1


; *** Computation are not done if date is the same as previous call

      IF(iyear EQ iy AND idoy EQ id) THEN BEGIN
        d1=d1p
        d2=d2p
        d3=d3p
        RETURN
      ENDIF

      iy=iyear
      id=idoy

      fmt10="(' * ROCOTLIB/cp_geo_dipole_dir: Warning! year=',i4.4," + $
             "'   must be only in [1900-2020]',/," + $
             "'   It will be computed for year ',i4.4)"

; *** Check date interval of validity

;     we are restricted by the interval 1965-2010, for which the igrf
;     coefficients are known;
;     if iyear is outside this interval, then the PRO uses the
;     nearest limiting value and print a warning:

      IF(iy LT 1900)  THEN BEGIN
                       iy=1900
                       IF(ipr NE 1) THEN PRINT, FORMAT=fmt10, iyear, iy
                       ipr=1
                       alat=rlat(0)
                       alon=rlon(0)
                     ENDIF

      IF(iy GT 2020)  THEN BEGIN
                       iy=2020
                        IF(ipr NE 1) THEN PRINT, FORMAT=fmt10, iyear, iy
                       ipr=1
                       alat=rlat(24)
                       alon=rlon(24)
                      ENDIF

; *** decimal year

      dyear=float(iy) +float(id)/365.25

; *** find two known intervals

      FOR i=1,24 DO BEGIN
         IF(FLOAT(iy) GE year(i-1) AND FLOAT(iy) LE year(i))  THEN BEGIN 
             alat= rlat(i-1) + (rlat(i)-rlat(i-1))*(dyear-year(i-1))/5.
             alon= rlon(i-1) + (rlon(i)-rlon(i-1))*(dyear-year(i-1))/5.
             GOTO, WWW20
         ENDIF
      ENDFOR

      WWW20:

; *** direction of dipole axis in GEO system:

      clat=(90. -alat)*3.141593/180.
      clon=alon*3.141593/180.

      t_sph_to_car, 1.,clat,clon,d1,d2,d3

      d1p=d1
      d2p=d2
      d3p=d3


;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cp_gei_sun_dir, iyear,idoy,ih,im,is,gst,slong,sra,sdec,obliq

; ----------------------------------------------------------------------
; *   Class  : basic compute modules of Rocotlib Software
; *   Object : compute_sun_direction in GEI system
; *   Author : CT.Russel, CE-D, 1971, rev. P.R., 1992,2001,2002

; *   Comment: calculates four quantities in gei system necessary for
;              coordinate transformations dependent on sun position
;              (and, hence, on universal time and season)
;              Initial code from C.T. Russel, cosmic electro-dynamics, $
;              v.2, 184-196, 1971.
;              accuracy: 0.006 degree.
;              Adaptation P.Robert, November 1992.
;              Revised and F90 compatibility, P. Robert June 2001.
;              Optimisation of DBLE computations and comments, $
;              P. Robert, December 2002

; *   Input  : iyear : year (1901-2099)
;              idoy : day of the year (1 FOR january 1)
;              ih,im,is : hours, minutes, seconds U.T.

; *   Output : gst      greenwich mean sideral time (radians)
;              slong    longitude along ecliptic (radians)
;              sra      right ascension (radians)
;              sdec     declination of the sun (radians)
;              obliq    inclination of Earth's axis (radians)
; ----------------------------------------------------------------------

;     double precision dj,fday
      dj  =1.D0
      fday=1.D0

      IF(iyear LT 1901 OR iyear GT 2099)  THEN BEGIN
        PRINT,'*** Rocotlib/cp_gei_sun_dir: year = ',iyear
        PRINT,'*** Rocotlib/cp_gei_sun_dir: year must be >1900 and <2100'
        MESSAGE, /NONAME, /NOPREFIX,   '*** Rocotlib/cp_gei_sun_dir: year must be >1900 and <2100'
      ENDIF

      pi=ACOS(-1.)
      pisd= pi/180.

; *** Julian day and greenwich mean sideral time

      fday=DOUBLE(ih*3600L+im*60L+is)/86400.d0
      ileap=(iyear-1901L)/4L
;     Note: year 2000 is a leap year, line above is right
      dj=DOUBLE(365*(iyear-1900L) +ileap +idoy) -0.5d0 +fday
      gst=FLOAT(Fmod(279.690983d0 +0.9856473354d0*dj+360.d0*fday +180.d0,360.d0))*pisd

; *** longitude along ecliptic

      vl= FLOAT( Fmod(279.696678d0 +0.9856473354d0*dj,360.d0) )
      t=FLOAT(dj/36525.d0)
      g=FLOAT( Fmod(358.475845d0 +0.985600267d0*dj, 360.d0) )*pisd
      slong=(vl+(1.91946 -0.004789*t)*SIN(g) +0.020094*SIN(2.*g))*pisd

; *** inclination of Earth's axis

      obliq=(23.45229 -0.0130125*t)*pisd
      sob=SIN(obliq)
      cob=COS(obliq)

;     precession of declination (about 0.0056 deg., ajou P. Robert)

      pre= (0.0055686 - 0.025e-4*t)*pisd

; *** declination of the sun

      slp=slong -pre
      sind=sob*SIN(slp)
      cosd=SQRT(1. -sind^2)
      sc=sind/cosd
      sdec=atan(sc)

; *** right ascension of the sun

      sra=pi -ATAN((cob/sob)*sc, -COS(slp)/cosd)

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cp_sunrise_sunset, iyear,imon,iday,rlat,rlon, $
                             tmer,tris,tset,durd, $
                             elemer,azimer,eleris,aziris,eleset,aziset,icor

      COMMON sunVDH,  suntim,sunele,sunazi

      suntim=fltarr(1441)
      sunele=fltarr(1441)
      sunazi=fltarr(1441)

; ----------------------------------------------------------------------
; *   Class  : basic compute modules of Rocotlib Software
; *   Object : compute_sunset_time and others
; *   Author : P. Robert, CRPE, 2001 Revised Dec. 2011
; *   Comment: Spherical Earth assumed

; *   Input  : iyear,imon,iday, rlat, rlong (rad)
;              icor: type de correction
;                  0 pas de correction
;                  1: correction de la refraction atmospherique seule
;                     (calcul comparable a l'IMC, ancien BDL)
;                  2: correction de la refraction et du bord superieur
;                     du soleil (calcul de la SAF).
;                  3: correction de la refraction avec l'elevation
;                     pour le passage au meridien (utile pour les zones
;                     polaires, ou le soleil peut raser l'horizon
;                     a midi)

;                  voir explications plus precises ci-dessous

; *   Output : tmer,tris,tset,durd on char*8 format, as '23:42:37'
;              tmer: Sun Meridian time (local noon)
;              tris: Sunrise time
;              tset: Sunset  time
;              durd: Duration of the day

;              elemer; elevation, en degres, du soleil au meridin
;              azimer: azimuth, en degres, a partir du nord vers l'est
;                      du soleil au meridien
;              eleris,aziris: meme chose pour le lever de soleil
;              eleset,aziset: meme chose pour le coucher

;              /sunVDH/ trajectoire du Soleil pendant la journe
;              Accuracy: 10 sec.

;    Les valeures pour Paris (latitude moyenne) et Tromso (zone polaire)
;    ont ete comparees aux ephemerides de l'institut de mecanique 
;    celeste (anciennement bureau des longitudes) avec icor=1
;    Les resultats sont comparables a la minute prs pour les levers
;    et couchers, et a quelques secondes d'arc pour les elevations et 
;    azimuts, y compris dans les zones polaires et pour les transitions
;    avec les nuits polaires ou jours polaires (pas de lever ni coucher)

; ----------------------------------------------------------------------

; *** calcul des heures de lever et coucher de soleil:
;     correction du diametre apparent du soleil et de la refraction
;     a l'horizon. Les valeurs prises sont extraites de l'ouvrage
;     "introduction aux ephemerides astronomiques", publie par le
;     Bureau Des Longitudes.
;     Diametre apparent du soleil: 32'
;     Refraction a l'horizon     : 34' (36.6 pour le BDL)
;     correction= 32'/2 + 34' = 50'

; *** Attention :
;     Les Ephemrides Astronomiques du Bureau des Longitudes ne tiennent
;     compte que de la refraction a laquelle elles donnent la valeur
;     r = 36,6'.
;     Les Ephemrides Astronomiques de la S.A.F. considere le lever comme
;     l'apparition de son bord superieur.
;     Elles utilisent la valeur ht = -50'.

; *   variation avec la temperature et la pression:
;     on peut utiliser la formule de Bennet, pour corriger la refraction
;     en la multipliant par le coefficient:
;     (P/1010)(283/(273+T))
;     avec P en millibar et T en Celsius.
;     Note: Le BDL prend 1 pour cette correction, donc
;     suppose une presion P=1010 et une temperature T=10 deg.

; *   variation de R avec la hauteur apparente:
;     Une formule assez simple est donne par Bennet :
;     R = 1/Tan[h+7,31/(h+4,4)]
;     avec h en degre et R en minute d'arc
;     soit pour h=0 on trouve R= 34.5 
;          pour h=45          R=  0.99' 
;     (34' et 1' etant les valeurs couramment admises)

; *   Crepuscules : par definition la fin (le soir) ou le debut 
;     (le matin) des crepuscules civil, nautique et astronomique 
;     se produit quand le centre du Soleil est abaisse de 6, 12 et 18 
;     degres sous l'horizon. 
; ----------------------------------------------------------------------

      mimer=LONARR(2)
      vemer=FLTARR(2)

      lmer=STRLEN(tmer)
      lris=STRLEN(tris)
      lset=STRLEN(tset)
      ldur=STRLEN(durd)

      IF(lmer LT 8)  THEN BEGIN
           PRINT, '*** Rocotlib/cp_sunrise_sunset: tmer must be char*8'
           PRINT, '    len=',lmer
           MESSAGE, /NONAME, /NOPREFIX, 'cp_sunrise_sunset: tmer must be char*8 variable'
      ENDIF

      IF(lris LT 8)  THEN BEGIN
           PRINT, '*** Rocotlib/cp_sunrise_sunset: tris must be char*8'
           PRINT, '    len=',lris
           MESSAGE, /NONAME, /NOPREFIX, 'cp_sunrise_sunset: tris must be char*8 variable'
      ENDIF

      IF(lset LT 8)  THEN BEGIN
           PRINT, '*** Rocotlib/cp_sunrise_sunset: tset must be char*8'
           PRINT, '    len=',lset
           MESSAGE, /NONAME, /NOPREFIX, 'cp_sunrise_sunset: tset must be char*8 variable'
      ENDIF

      IF(ldur LT 8)  THEN BEGIN
           PRINT, '*** Rocotlib/cp_sunrise_sunset: durd must be char*8'
           PRINT, '    len=',ldur
           MESSAGE, /NONAME, /NOPREFIX, 'cp_sunrise_sunset: durd must be char*8 variable'
      ENDIF

      milris= -1L
      milset= -1L
      milmer= -1L
      milmin= -1L
      
      imer=0
      vemer(0)= -2.
      vemer(1)= -2.

      sdp=0.
      svp=0.

      deg=180./ACOS(-1.)

      fmt100="(i2.2,':',i2.2,':',i2.2)"
      fmt200="(i2  ,':',i2.2,':',i2.2)"

; *   tolerance de 1 degre sur le meridien quand le soleil est 
;     au ras de l'horizon et donc sujet a la refraction

      cormax=1.
      svmin= -SIN(cormax/deg)
      
; *** initialisation de sdp=sd et svp=sv pour le jour d'avant

      cv_date_to_jul2000, iyear,imon,iday,jd00
      jd00=jd00-1
      cv_jul2000_to_date, jd00,iyearp,imonp,idayp
      cp_time_param, iyearp,imonp,idayp,23,59,30
      g_gei_geo_sun_dir, sxgei,sygei,szgei,sxgeo,sygeo,szgeo
      t_geo_to_vdh, sxgeo,sygeo,szgeo,rlat,rlon,sv,sd,sh

; *   correction de la refraction (voir + loin)

      t_car_to_sph, sd,sh,sv,r,tetnc,phi
; *   pas de correction de refraction par defaut
      tet=tetnc
; *   calcul de la refraction a l'horizon 
      IF(icor GT 0) THEN BEGIN
                    IF(icor EQ 1) THEN BEGIN
                                  cor= 36.6/60.
                                  ENDIF ELSE BEGIN
                                  cor= 50./60.
                    ENDELSE         
                    tet=tetnc -cor/deg
                    t_sph_to_car, r,tet,phi,sd,sh,sv
      ENDIF
       
      sdp=sd
      svp=sv
      iyearp=iyear
      imonp =imon
      idayp =iday
      ismax=50L

;     ----------------------------------------------
;     boucle sur le temps de la journee 
;     ----------------------------------------------

      FOR imin=0,1440 DO BEGIN
      
        IF(imin EQ 1440)  THEN BEGIN
          cv_date_to_jul2000, iyear,imon,iday,jd00
          jd00=jd00+1
          cv_jul2000_to_date, jd00,iyear,imon,iday
          ismax=30L
        ENDIF   
      
      FOR iss  =0L,ismax,10L DO BEGIN

        IF(imin LT 1440) THEN BEGIN
                         isec=imin*60L+iss 
                         ENDIF ELSE BEGIN
                         isec=iss
        ENDELSE

; *** calcul de la direction du Soleil dans le VDH pour chaque pas de 
;     temps de la journee en cours ( precision 10 sec)

      millsec=LONG(isec)*1000L

      cv_msotd_to_hmsms, millsec,ih,im,is,ims
      cp_time_param, iyear,imon,iday,ih,im,is
      g_gei_geo_sun_dir, sxgei,sygei,szgei,sxgeo,sygeo,szgeo
      t_geo_to_vdh, sxgeo,sygeo,szgeo,rlat,rlon,sv,sd,sh

; *** calcul des heures de lever et coucher de soleil:
;     correction du diametre apparent du soleil et de la refraction
;     a l'horizon. Les valeurs prises sont extraites de l'ouvrage
;     "introduction aux ephemerides astronomiques", publie par le
;     Bureau Des Longitudes.
;     Diametre apparent du soleil: 32'
;     Refraction a l'horizon     : 34'
;     correction= 32'/2 + 34' = 50'

; *** Attention :
;     Les Ephemrides Astronomiques du Bureau des Longitudes ne tiennent
;     compte que de la refraction a laquelle elles donnent la valeur
;     r = 36,6'.
;     Les Ephemrides Astronomiques de la S.A.F. considere le lever comme
;     l'apparition de son bord superieur.
;     Elles utilisent la valeur ht = -50'.

; *   dans le reperes DHV, teta est l'angle entre la verticale sortante 
;     et la direction du Soleil

      t_car_to_sph, sd,sh,sv,r,tetnc,phi

; *   pas de correction de refraction par defaut

      tet=tetnc

; *   calcul de la refraction a l'horizon 

      IF(icor GT 0) THEN BEGIN
                    IF(icor EQ 1)  THEN BEGIN
                                  cor= 36.6/60.
                                  ENDIF ELSE BEGIN
                                  cor= 50./60.
                    ENDELSE
                                  
                    tet=tetnc -cor/deg
                    t_sph_to_car, r,tet,phi,sd,sh,sv

; *   calcul de la refraction a teta pour le calcul de la trajectoire
;     Quand le Soleil est vertical (tet=0), la correction est minime;
;     Quand le soleil est proche de l'horizon (tet=90 cor=34'

                    IF(icor EQ 3) THEN BEGIN
                                  h  = 90. -tet*deg
                                  cor= 1./tan( (h +7.31/(h+4.4))/deg )
                                  cor=cor/60.
                    ENDIF

                    tet=tetnc -cor/deg
      ENDIF

; *** chargement du COMMON /sunVDH/
;     resolution en temps: 1mn 
;     l'azimut est compt positivement depuis le Nors vers l'Est

      IF(is EQ 0)  THEN BEGIN
          suntim(imin)= float(imin)
          sunele(imin)= 90. -tet*deg
          sunazi(imin)= 90. -phi*deg
      ENDIF

; *** calcul du midi local, quand D (vers l'est) change de signe
;     attention: dans le cas du soleil de minuit, il y a deux meridien
;     ou le soleil est visible.
;     On prend celui ou le soleil est au plus haut 
;     (a midi, et non pas a minuit)

      IF(sv GT svmin AND sd*sdp LT 0.) THEN BEGIN
           imer=imer+1
           IF(imer GT 2)then begin
                      PRINT,'^Rocotlib more than 2 meridian crossing!'
                      PRINT,'           last is taken into account'
                        GOTO, WWW21
           ENDIF
           
           mimer(imer)=isec*1000L
           vemer(imer)=sv
      ENDIF
      
      WWW21:

; *** calcul du temps ou l'elevation est minimum (soleil de minuit)

      IF(sv GE 0. AND svp GE 0. AND sv LT svp) THEN milmin= isec*1000L

; *** on a une transition (lever ou couche) quand V (vertical sortante)
;     change de signe (unite: millisec du jour)

; *   cas ou sv est nul (le soleil est a l'horizon)

      epsi=1.e-6
      
      IF(ABS(sv) LT epsi) THEN BEGIN
                   sv=0.
                   IF(svp LT 0.) THEN BEGIN
;                    juste avant, le soleil etait sous l'horizon
;                    la transition est donc un lever de soleil
                                 milris= isec*1000L
                                 ENDIF ELSE BEGIN
;                    juste avant, le soleil etait au dessus de l'horizon
;                    la transition est donc un coucher de soleil
                                 milset= isec*1000L
                   ENDELSE
      GOTO, WWW20
      ENDIF

; *   cas ou svp est nul (le soleil etait a l'horizon)

      IF(ABS(svp) LT epsi) THEN BEGIN
                   IF(sv GT 0.) THEN BEGIN
;                    le soleil est au dessus de l'horizon
;                    la transition est donc un lever de soleil
                                 milris= isec*1000L
                                 ENDIF ELSE BEGIN
;                    le soleil est au dessous de l'horizon
;                    la transition est donc un coucher de soleil
                                 milset= isec*1000L
                    ENDELSE
      GOTO, WWW20
      ENDIF

; *   sv et svp sont non nuls

      IF(sv*svp LT 0.) THEN BEGIN
;              si sv et svp sont de signe contraire, on a une transition
;              (lever ou coucher)
                       IF(sv GT 0.) THEN BEGIN
;                    le soleil passe au passe au dessus de l'horizon
;                    la transition est donc un lever de soleil
                                    milris= isec*1000L
                                    ENDIF ELSE BEGIN
;                    le soleil passe au passe au dessous de l'horizon
;                    la transition est donc un coucher de soleil
                                    milset= isec*1000L
                       ENDELSE    
      ENDIF
      
      WWW20:   
      sdp=sd
      svp=sv
      
      ENDFOR     
      ENDFOR

;     ----------------------------------------------
;     la boucle sur le temps de la journee est finie
;     ----------------------------------------------
      
; *   on restaure le jour en cours

      iyear=iyearp
      imon =imonp
      iday =idayp
      
; *** encodage des resultats sous la forme HH:MM:SS
;     et calcul de l'elevation et de l'azimuth pour chacun des 3 cas

      tmer='night   '
      tris='no SR   '
      tset='no SS   '
      durd='00:00:00'
      
      elemer=999.
      eleris=999.
      eleset=999.
      
      azimer=999.
      aziris=999.
      aziset=999.

; *   choix du midi local si soleil de minuit

      IF(imer GT 0) THEN BEGIN
                    IF(vemer(0) GT vemer(1)) THEN BEGIN
                                             milmer=mimer(0)
                                             ENDIF ELSE BEGIN
                                             milmer=mimer(1)
                    ENDELSE
      ENDIF

; *   calcul des positions au temps du meridien

      IF(milmer GT -1L) THEN BEGIN

            cv_msotd_to_hmsms, milmer,ih,im,is,ims
            tmer=STRING(FORMAT=fmt100, ih,im,is)
                      
            cp_time_param, iyear,imon,iday,ih,im,is
            g_gei_geo_sun_dir, sxgei,sygei,szgei,sxgeo,sygeo,szgeo
            t_geo_to_vdh, sxgeo,sygeo,szgeo,rlat,rlon,sv,sd,sh
            t_car_to_sph, sd,sh,sv,r,tet,phi

            IF(icor EQ 3) THEN BEGIN
;                         calcul de la refraction a teta
                          h  = 90. -tet*deg
                          cor= 1./tan( (h +7.31/(h+4.4))/deg)
                          cor=cor/60.
                          tet=tet -cor/deg
            ENDIF
      
            elemer= 90. -tet*deg
            azimer= 90. -phi*deg
      ENDIF

; *   duree du jour calculee seuleument si le lever et le coucher
;     sont definis

      IF(milris GT -1L AND milset GT -1L) THEN BEGIN
                 idiff= milset-milris
                 IF(idiff LT 0) THEN idiff=idiff +86400000L
                 cv_msotd_to_hmsms, idiff,ih,im,is,ims
                 durd=STRING(FORMAT=fmt200, ih,im,is)
                                            ENDIF ELSE BEGIN
                 IF(elemer GT 0. AND elemer LT 180.) THEN durd='24:00:00'
      ENDELSE

; *** cas du soleil de minuit: lever = coucher= temp elevation min

      IF(durd EQ '24:00:00' AND milmin GT -1L) THEN BEGIN
                      milris=milmin
                      milset=milmin
      ENDIF

      IF(milris GT -1L) THEN BEGIN           
            cv_msotd_to_hmsms, milris,ih,im,is,ims
            tris=STRING(FORMAT=fmt100, ih,im,is)
                      
            cp_time_param, iyear,imon,iday,ih,im,is
            g_gei_geo_sun_dir, sxgei,sygei,szgei, sxgeo,sygeo,szgeo
            t_geo_to_vdh, sxgeo,sygeo,szgeo,rlat,rlon,sv,sd,sh
            t_car_to_sph, sd,sh,sv,r,tet,phi
      
            eleris= 90. -tet*deg
            aziris= 90. -phi*deg
      ENDIF

      IF(milset GT -1L) THEN BEGIN
            cv_msotd_to_hmsms, milset,ih,im,is,ims
            tset=STRING(FORMAT=fmt100, ih,im,is)
                      
            cp_time_param, iyear,imon,iday,ih,im,is
            g_gei_geo_sun_dir, sxgei,sygei,szgei,sxgeo,sygeo,szgeo
            t_geo_to_vdh, sxgeo,sygeo,szgeo,rlat,rlon,sv,sd,sh
            t_car_to_sph, sd,sh,sv,r,tet,phi
      
            eleset= 90. -tet*deg
            aziset= 90. -phi*deg                      
      ENDIF

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cp_time_param, iyear,imonth,iday,ih,im,is

; ----------------------------------------------------------------------
; *   Class  : basic compute modules of Rocotlib Software
; *   Object : compute_time_parameters and time-dependent matrix
; *   Author : P. Robert, CRPE, 1992

; *   Comment: Prepare all time varying quantities FOR computations of
;              coordinate transforms of the library.

; *   Input  : iyear,imonth,iday (1901<year<2099)
;              ih,im,is : hours, minutes, seconds U.T.
; *   Output : in COMMON statements
; ----------------------------------------------------------------------

      COMMON timp00,  sgst,cgst, gst,slong,srasn,sdecl,obliq
      COMMON timp01,  gs1,gs2,gs3, gm1,gm2,gm3, ge1,ge2,ge3,gr1,gr2,gr3
      COMMON timp02,  ps1,ps2,ps3, gd1,gd2,gd3, pe1,pe2,pe3,pr1,pr2,pr3

      COMMON timp03,  gmgs1,gmgs2,gmgs3,  gegs1,gegs2,gegs3
      COMMON timp04,  grgs1,grgs2,grgs3,  grge1,grge2,grge3
      COMMON timp05,  gdps1,gdps2,gdps3
      COMMON timp06,  peps1,peps2,peps3,  prps1,prps2,prps3

      COMMON timp07,  xeima1,xeima2,xeima3,  yeima1,yeima2,yeima3
      COMMON timp08,  xeism1,xeism2,xeism3,  yeism1,yeism2,yeism3
      COMMON timp09,  yeigm1,yeigm2,yeigm3,  zeigm1,zeigm2,zeigm3
      COMMON timp10,  yeigq1,yeigq2,yeigq3,  zeigq1,zeigq2,zeigq3

      COMMON timp11,  stetq,ctetq, sdze,cdze, smu,cmu, sphi,cphi

      COMMON timp12,  xeoma1,xeoma2,xeoma3,  yeoma1,yeoma2,yeoma3
      COMMON timp13,  xeosm1,xeosm2,xeosm3,  yeosm1,yeosm2,yeosm3
      COMMON timp14,  yeogm1,yeogm2,yeogm3,  zeogm1,zeogm2,zeogm3
      COMMON timp15,  yeogq1,yeogq2,yeogq3,  zeogq1,zeogq2,zeogq3

      COMMON iyidipr, iy,id,ipr
      COMMON ily,ily

      iy=-1
      id=-1
      ipr=0
      ily=-1

; WWW  data idoy /-1/

; *** day of year

      cv_date_to_doty, iyear,imonth,iday,idoy

; *** Dipole direction in GEO system

      cp_geo_dipole_dir, iyear,idoy,gd1,gd2,gd3

      qd12=SQRT(gd1^2 + gd2^2)

; *** Sun direction in GEI (from right ascension and declination), $
;     greenwich mean sideral time and longitude along ecliptic

      cp_gei_sun_dir, iyear,idoy,ih,im,is, gst,slong,srasn,sdecl,obliq

      gs1=COS(srasn)*COS(sdecl)
      gs2=SIN(srasn)*COS(sdecl)
      gs3=SIN(sdecl)

; *** sin and cos of GMST

      sgst=SIN(gst)
      cgst=COS(gst)

; *** ecliptic pole in GEI system

      ge1=  0.
      ge2= -SIN(obliq)
      ge3=  COS(obliq)

; *** direction of the rotation axis of the sun in GEI system
;     (from C.T. Russell, Cosmic Electro-dynamics, V.2, 184-196, 1971.)

      rad=  57.29578

      ras= -74.0/rad
      dec=  63.8/rad

      gr1=COS(ras)*COS(dec)
      gr2=SIN(ras)*COS(dec)
      gr3=SIN(dec)

; *** dipole direction in GEI system

      gm1= gd1*cgst - gd2*sgst
      gm2= gd1*sgst + gd2*cgst
      gm3= gd3

      qm12=SQRT(gm1^2 + gm2^2)

; *** direction of the sun in GEO system

      ps1=  gs1*cgst + gs2*sgst
      ps2= -gs1*sgst + gs2*cgst
      ps3=  gs3

; *** direction of the ecliptic in GEO system

      pe1=  ge1*cgst + ge2*sgst
      pe2= -ge1*sgst + ge2*cgst
      pe3=  ge3

; *** direction of the rotation axis of the sun in GEO system

      pr1=  gr1*cgst + gr2*sgst
      pr2= -gr1*sgst + gr2*cgst
      pr3=  gr3

; *** cross product MxS in GEI system

      gmgs1= gm2*gs3 - gm3*gs2
      gmgs2= gm3*gs1 - gm1*gs3
      gmgs3= gm1*gs2 - gm2*gs1

      rgmgs=SQRT(gmgs1^2 + gmgs2^2 + gmgs3^2)

; *** cross product ExS in GEI system

      gegs1= ge2*gs3 - ge3*gs2
      gegs2= ge3*gs1 - ge1*gs3
      gegs3= ge1*gs2 - ge2*gs1

; *** cross product RxS in GEI system

      grgs1= gr2*gs3 - gr3*gs2
      grgs2= gr3*gs1 - gr1*gs3
      grgs3= gr1*gs2 - gr2*gs1

      rgrgs=SQRT(grgs1^2 + grgs2^2 + grgs3^2)

; *** cross product RxE in GEI system

      grge1= gr2*ge3 - gr3*ge2
      grge2= gr3*ge1 - gr1*ge3
      grge3= gr1*ge2 - gr2*ge1

; *** cross product DxS in GEO system

      gdps1= gd2*ps3 - gd3*ps2
      gdps2= gd3*ps1 - gd1*ps3
      gdps3= gd1*ps2 - gd2*ps1

      rgdps=SQRT(gdps1^2 + gdps2^2 + gdps3^2)

; *** cross product ExS in GEO system

      peps1= pe2*ps3 - pe3*ps2
      peps2= pe3*ps1 - pe1*ps3
      peps3= pe1*ps2 - pe2*ps1

; *** cross product RxS in GEO system

      prps1= pr2*ps3 - pr3*ps2
      prps2= pr3*ps1 - pr1*ps3
      prps3= pr1*ps2 - pr2*ps1

      rprps=SQRT(prps1^2 + prps2^2 + prps3^2)

; *** computation of gei to mag vectors

      IF(qm12 LT 1.e-30) THEN MESSAGE, /NONAME, /NOPREFIX, '*** Rocotlib error qm12'
      xeima1=  gm1*gm3/qm12
      xeima2=  gm2*gm3/qm12
      xeima3= -qm12

      yeima1= -gm2/qm12
      yeima2=  gm1/qm12
      yeima3=  0.

; *** computation of gei to sm vectors

       IF(rgmgs LT 1.e-30) THEN MESSAGE, /NONAME, /NOPREFIX, '*** Rocotlib error rgmgs'
      yeism1=gmgs1/rgmgs
      yeism2=gmgs2/rgmgs
      yeism3=gmgs3/rgmgs

      xeism1= yeism2*gm3 - yeism3*gm2
      xeism2= yeism3*gm1 - yeism1*gm3
      xeism3= yeism1*gm2 - yeism2*gm1

; *** computation of gei to gsm vectors

      yeigm1= gmgs1/rgmgs
      yeigm2= gmgs2/rgmgs
      yeigm3= gmgs3/rgmgs

      zeigm1= gs2*yeigm3 - gs3*yeigm2
      zeigm2= gs3*yeigm1 - gs1*yeigm3
      zeigm3= gs1*yeigm2 - gs2*yeigm1

; *** computation of gei to gseq vectors

       IF(rgrgs LT 1.e-30) THEN MESSAGE, /NONAME, /NOPREFIX, '*** Rocotlib error rgrgs'
      yeigq1= grgs1/rgrgs
      yeigq2= grgs2/rgrgs
      yeigq3= grgs3/rgrgs

      zeigq1= gs2*yeigq3 - gs3*yeigq2
      zeigq2= gs3*yeigq1 - gs1*yeigq3
      zeigq3= gs1*yeigq2 - gs2*yeigq1

; *** computation of tetq angle

      stetq= (grge1*gs1 + grge2*gs2 + grge3*gs3)/rgrgs
      ctetq= SQRT(1.-stetq^2)

; *** computation of mu angle

      smu= ps1*gd1 + ps2*gd2 + ps3*gd3
      cmu= SQRT(1.-smu*smu)

; *** computation of dzeta angle

      cdze= (ge1*gm1   + ge2*gm2   + ge3*gm3)/rgmgs
      sdze= (ge1*gmgs1 + ge2*gmgs2 + ge3*gmgs3)/rgmgs
;     accuracy low on this angle
      epsi=1.e-4
      IF(ABS(sdze^2 +cdze^2-1.) GT epsi) THEN MESSAGE, /NONAME, /NOPREFIX, '*** Rocotlib error 3'

; *** computation of phi angle

      q=qd12*rgdps
      IF(q LT 1.e-30) THEN MESSAGE, /NONAME, /NOPREFIX, '*** Rocotlib error q'

      cphi=(gd1*gd3*ps1 + gd2*gd3*ps2 - (gd1^2+gd2^2)*ps3)/q
      sphi=(gd2*ps1 - gd1*ps2)/q
       IF(ABS(sphi^2 +cphi^2-1.) GT epsi) THEN MESSAGE, /NONAME, /NOPREFIX, '*** Rocotlib error 4'

; *** computation of geo to mag vectors

      IF(qd12 LT 1.e-30) THEN MESSAGE, /NONAME, /NOPREFIX, '*** Rocotlib error qd12'
      yeoma1= -gd2/qd12
      yeoma2=  gd1/qd12
      yeoma3=  0.

      xeoma1=  yeoma2*gd3
      xeoma2= -yeoma1*gd3
      xeoma3=  yeoma1*gd2 - yeoma2*gd1

; *** computation of geo to sm vectors

      IF(rgdps LT 1.e-30) THEN MESSAGE, /NONAME, /NOPREFIX, '*** Rocotlib error rgdps'
      yeosm1= gdps1/rgdps
      yeosm2= gdps2/rgdps
      yeosm3= gdps3/rgdps

      xeosm1= yeosm2*gd3 - yeosm3*gd2
      xeosm2= yeosm3*gd1 - yeosm1*gd3
      xeosm3= yeosm1*gd2 - yeosm2*gd1

; *** computation of geo to gsm vectors

      yeogm1=yeosm1
      yeogm2=yeosm2
      yeogm3=yeosm3

      zeogm1= ps2*yeogm3 - ps3*yeogm2
      zeogm2= ps3*yeogm1 - ps1*yeogm3
      zeogm3= ps1*yeogm2 - ps2*yeogm1

; *** computation of geo to gsq vectors

      IF(rprps LT 1.e-30) THEN MESSAGE, /NONAME, /NOPREFIX, '*** Rocotlib error rprps'
      yeogq1= prps1/rprps
      yeogq2= prps2/rprps
      yeogq3= prps3/rprps

      zeogq1= ps2*yeogq3 - ps3*yeogq2
      zeogq2= ps3*yeogq1 - ps1*yeogq3
      zeogq3= ps1*yeogq2 - ps2*yeogq1

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cp_time_param2, jd1950,houday

; ----------------------------------------------------------------------
; *   Class  : basic compute modules of Rocotlib Software
; *   Object : compute_time_parameters and time-dependent matrix
; *   Author : P. Robert, CRPE, 2001

; *   Comment: Prepare all time varying quantities FOR computations of
;              coordinate transforms of the library.
;              Same as cp_time_param, only input arguments are changed.

; *   Input  : jd1950: Julian day 1950 (0= 1/1/1950)
;              houday: decimal hour of the day (U.T.)
; *   Output : in COMMON statements
; ----------------------------------------------------------------------

      COMMON ihimis, ih,im,is

      cv_jul1950_to_date, jd1950,iyear,imonth,iday
      cv_dech_to_hms, houday,ih,im,is

      cp_time_param, iyear,imonth,iday,ih,im,is

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cp_time_param3, jd2000,houday

; ----------------------------------------------------------------------
; *   Class  : basic compute modules of Rocotlib Software
; *   Object : compute_time_parameters and time-dependent matrix
; *   Author : P. Robert, CRPE, 2001

; *   Comment: Prepare all time varying quantities FOR computations of
;              coordinate transforms of the library.
;              Same as cp_time_param, only input arguments are changed.

; *   Input  : jd2000: Julian day 2000 (0= 1/1/2000)
;              houday: decimal hour of the day (U.T.)
; *   Output : in COMMON statements
; ----------------------------------------------------------------------

      COMMON ihimis, ih,im,is

      cv_jul1950_to_date, jd2000,iyear,imonth,iday
      cv_dech_to_hms, houday,ih,im,is

      cp_time_param, iyear,imonth,iday,ih,im,is

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cp_tpn_param, xo,yo,zo,xs, Tx,Ty,Tz, Px,Py,Pz, Nx,Ny,Nz

;     real Tx,Ty,Tz, Px,Py,Pz, Nx,Ny,Nz
      

;     ---------------------------------------------------------------+--
; *   Class  : basic compute modules of Rocotlib Software
; *   Object : compute_TPN_system
; *   Author : P. Robert, CETP, 2004
; *   Comment: Compute TPN vector in GSE system or any system having X
;              axis towards the SUN.

;         N: Output normal to the paraboloid
;         T: tengente to the paraboloid, towards the summit
;         P: tengente to the paraboloid, P=N X T

;     The paraboloid is defined by its summit Xs and the local point 
;     Xo, Yo, Zo

;     Note: the paraboloid is close to the magnetopause if the summit
;           is defined as the subsolar point (by T87,T89 model or other)
;           and if the local point Xo,Yo,Zo correspond to magnetopause 
;           crossing.

; *   Input : xo,yo,zo,xs
; *   Output: Nx,Ny,Nz,Tx,Ty,Tz,Px,Py,Pz
;     ---------------------------------------------------------------+--
;      

      r= SQRT(yo^2 +zo^2)
      dx=xs-xo

      Nx= r
      Ny= 2.*dx*yo/r
      Nz= 2.*dx*zo/r

      Tx= 2.*dx
      Ty= -yo
      Tz= -zo

      Px= Ny*Tz -Nz*Ty
      Py= Nz*Tx -Nx*Tz
      Pz= Nx*Ty -Ny*Tx

;     normalisation

      rnoN= SQRT(Nx^2 +Ny^2 +Nz^2)
      rnoT= SQRT(Tx^2 +Ty^2 +Tz^2)
      rnoP= SQRT(Px^2 +Py^2 +Pz^2)

      Nx=Nx/rnoN
      Ny=Ny/rnoN
      Nz=Nz/rnoN

      Tx=Tx/rnoT
      Ty=Ty/rnoT
      Tz=Tz/rnoT

      Px=Px/rnoP
      Py=Py/rnoP
      Pz=Pz/rnoP

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cp_nbday_in_month, iyear,imonth,nbday

; ----------------------------------------------------------------------
; *   Class  : calendar modules of Rocotlib Software
; *   Object : compute_number_of_day_of_the_month
; *   Author : P. Robert, CRPE, 2001

; *   Input  : iyear,imonth (1-12)
; *   Output : nbday
; ----------------------------------------------------------------------

      nday=[31,28,31,30,31,30,31,31,30,31,30,31]

      IF(imonth LT 1 OR imonth GT 12) THEN BEGIN
       PRINT,'*** Rocotlib/cp_nbday_in_month: month= ',imonth
       PRINT,'*** Rocotlib/cp_nbday_in_month: month must be >0 and <13'
       MESSAGE, /NONAME, /NOPREFIX,   '*** Rocotlib/cp_nbday_in_month: month must be >0 and <13'
                                      ENDIF

      cp_leap_year, iyear,ily

      IF(ily EQ 1) THEN BEGIN
                   nday(1)=29
                   ENDIF ELSE BEGIN
                   nday(1)=28
                   ENDELSE

      nbday=nday(imonth-1)

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cp_en_day_name, iday,cday,nbcha

; ----------------------------------------------------------------------
; *   Class  : calendar modules of Rocotlib Software
; *   Object : compute_english_day_name, ex: 'Monday' FOR iday=1
; *   Author : P. Robert, CRPE, 2001

; *   Input  : iday (1-7, otherwise modulo is done)
; *   Output : cday,nchar
; ----------------------------------------------------------------------

      days =['Monday','Tuesday','Wednesday','Thurday','Friday','Saturday','Sunday']
               
      nbca =[6,7,9,7,6,8,6]

      iweek= iday/7
      nday= iday -7*iweek 
      IF(nday LT 1) THEN nday=nday+7

      cday= days(nday-1)
      nbcha=nbca(nday-1)

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cp_en_month_name, imonth,cmonth,nchar

; ----------------------------------------------------------------------
; *   Class  : calendar modules of Rocotlib Software
; *   Object : compute_english_month_name
; *   Author : P. Robert, CRPE, 2001

; *   Input  : imonth (1-12)
; *   Output : cmonth,nchar
; ----------------------------------------------------------------------

      cara=['January','February','March','April','May','June', $
           'July','August','September','October','November','December']
      nbca =[7,8,5,5,3,4,4,6,9,7,8,8]


       IF(imonth LT 1 OR imonth GT 12) THEN BEGIN
        PRINT,'*** Rocotlib/cp_en_month_name: month= ',imonth
        PRINT,'*** Rocotlib/cp_en_month_name: month must be >0 and <13'
        MESSAGE, /NONAME, /NOPREFIX,   '*** Rocotlib/cp_en_month_name: month must be >0 and <13'
                                      ENDIF

      cmonth=cara(imonth-1)
      nchar =nbca(imonth-1)

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cp_fr_day_name, iday,cday,nbcha

; ----------------------------------------------------------------------
; *   Class  : calendar modules of Rocotlib Software
; *   Object : compute_french_day_name, ex: 'Lundi' FOR iday=1
; *   Author : P. Robert, CRPE, 2001

; *   Input  : iday (1-7, otherwise modulo is done)
; *   Output : cday,nchar
; ----------------------------------------------------------------------

      days =['Lundi','Mardi','Mercredi','Jeudi','Vendredi','Samedi','Dimanche']
                 
      nbca =[5,5,8,5,8,6,8]

      iweek= iday/7
      nday= iday -7*iweek
       IF(nday LT 1) THEN nday=nday+7

      cday= days(nday-1)
      nbcha=nbca(nday-1)

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cp_fr_month_name, imonth,cmonth,nchar

; ----------------------------------------------------------------------
; *   Class  : calendar modules of Rocotlib Software
; *   Object : compute_french_month_name
; *   Author : P. Robert, CRPE, 2001

; *   Input  : imonth (1-12)
; *   Output : cmonth,nchar
; ----------------------------------------------------------------------

     cara =['Janvier','Fvrier','Mars','Avril','Mai','Juin', $
            'Juillet','Aot','Septembre','Octobre','Novembre', $
            'Dcembre']

     nbca =[7,7,4,5,3,4,7,4,9,7,8,8]

      IF(imonth LT 1 OR imonth GT 12) THEN BEGIN
        PRINT,'*** Rocotlib/cp_fr_month_name: month= ',imonth
        PRINT,'*** Rocotlib/cp_fr_month_name: month must be >0 and <13'
        MESSAGE, /NONAME, /NOPREFIX,   '*** Rocotlib/cp_fr_month_name: month must be >0 and <13'
                                      ENDIF

      cmonth=cara(imonth-1)
      nchar =nbca(imonth-1)

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cp_leap_year, iyear,ileap

; ----------------------------------------------------------------------
; *   Class  : calendar modules of Rocotlib Software
; *   Object : compute_leap_year with ileap=1 FOR leap year, 0 if not
; *   Author : P. Robert, CRPE, 1992

; *   Input  : iyear (ex: 1980)
; *   Output : ileap (1 or 0 if iyear is or not a leap year)
; ----------------------------------------------------------------------

      IF(iyear LT 1900) THEN BEGIN
                PRINT,'*** Rocotlib/cp_leap_year: iyear= ',iyear
                PRINT,'*** Rocotlib/cp_leap_year: iyear must be > 1900'
                MESSAGE, /NONAME, /NOPREFIX,   '*** Rocotlib/cp_leap_year: iyear must be > 1900'
                        ENDIF

      ir=iyear-(iyear/4L)*4L
      IF(ir EQ 0) THEN BEGIN
                  ileap=1
                  ENDIF ELSE BEGIN
                  ileap=0
                  ENDELSE

      is=iyear-(iyear/100L)*100L
      IF(is EQ 0) THEN BEGIN
                  ir=iyear-(iyear/400L)*400L
                  IF(ir EQ 0) THEN BEGIN
                              ileap=1
                              ENDIF ELSE BEGIN
                              ileap=0
                              ENDELSE
                  ENDIF ELSE BEGIN
                  RETURN
                  ENDELSE

      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cp_seasons, iyear,id_sso,id_wso,id_seq,id_feq, $
                                  ct_sso,ct_wso,ct_seq,ct_feq


; ----------------------------------------------------------------------
; *   Class  : calendar modules of Rocotlib Software
; *   Object : compute_seasons, i.e. solstice & equinox
; *   Author : P. Robert, SDev, 2017

; *   Input  : iyear (ex: 1980)
; *   Output : id_sso,id_wso : June and december day of summer and
;                              winter solstice.
;              id_seq,id_feq : same FOR march and september spring
;                              and fall equinoxes.
;              ct_sso,ct_wso : June and december time of summer and
;                              winter solstice.
;              ct_seq,ct_feq : same FOR march and september spring
;                             and fall equinoxes; Cha*5 (hh:mm)
; *
; *   Comment: calcul des saisons: solstices & equinoxes, 
; *            precision 1 heure environ 
; ----------------------------------------------------------------------    

      id_sso=0
      id_wso=0
      id_seq=0
      id_feq=0

      ct_sso='?'
      ct_wso='?'
      ct_seq='?'
      ct_feq='?'

; *** calcul de l'equinoxe de printemps

      imon=3
      tetmax=0.

      cp_nbday_in_month, iyear,imon,nbday

      FOR iday=1,nbday DO BEGIN
      FOR ih= 0,23     DO BEGIN
      FOR im=0,50,10   DO BEGIN
      is=0

      cp_time_param, iyear,imon,iday,ih,im,is
      g_gei_geo_sun_dir, sxgei,sygei,szgei,sxgeo,sygeo,szgeo

      teta=ACOS(szgeo)*180./3.14159

      IF(teta GT tetmax AND szgeo GT 0.) THEN BEGIN
                         tetmax=teta
                         id_seq=iday
                         ct_seq=STRING(FORMAT="(i2.2,':',i2.2)", ih,im)
      ENDIF

      ENDFOR
      ENDFOR
      ENDFOR

; *** calcul de l'equinoxe d'automne

      imon=9
      tetmax=0.

      cp_nbday_in_month, iyear,imon,nbday

      FOR iday=1,nbday DO BEGIN
      FOR ih=0,23      DO BEGIN
      FOR im=0,50,10   DO BEGIN
      is=0

      cp_time_param, iyear,imon,iday,ih,im,is
      g_gei_geo_sun_dir, sxgei,sygei,szgei,sxgeo,sygeo,szgeo

      teta=ACOS(szgeo)*180./3.14159

      IF(teta GT tetmax AND szgeo GT 0.) THEN BEGIN
                         tetmax=teta
                         id_feq=iday
                         ct_feq=STRING(FORMAT="(i2.2,':',i2.2)", ih,im)
      ENDIF

      ENDFOR
      ENDFOR
      ENDFOR

; *** calcul du solstice d'ete

      imon=6
      zmax=0.

      cp_nbday_in_month, iyear,imon,nbday

      FOR iday=1,nbday DO BEGIN
      FOR ih=0,23 DO   BEGIN
      FOR im=0,50,10   DO BEGIN
      is=0

      cp_time_param, iyear,imon,iday,ih,im,is
      g_gei_geo_sun_dir, sxgei,sygei,szgei,sxgeo,sygeo,szgeo

      IF(szgeo GT zmax) THEN BEGIN
                        zmax=szgeo
                        id_sso=iday
                        ct_sso=STRING(FORMAT="(i2.2,':',i2.2)", ih,im)
      ENDIF

      ENDFOR
      ENDFOR
      ENDFOR

; *** calcul du solstice d'hiver

      imon=12
      zmin=1.

      cp_nbday_in_month, iyear,imon,nbday

      FOR iday=1,nbday DO BEGIN
      FOR ih=0,23      DO BEGIN
      FOR im=0,50,10   DO BEGIN
      is=0

      cp_time_param, iyear,imon,iday,ih,im,is
      g_gei_geo_sun_dir, sxgei,sygei,szgei,sxgeo,sygeo,szgeo

      IF(szgeo LT zmin) THEN BEGIN
                        zmin=szgeo
                        id_wso=iday
                        ct_wso=STRING(FORMAT="(i2.2,':',i2.2)", ih,im)
      ENDIF

      ENDFOR
      ENDFOR
      ENDFOR

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cv_doty_to_date, idoy,iyear,imonth,iday

; ----------------------------------------------------------------------
; *   Class  : calendar modules of Rocotlib Software
; *   Object : convert_day_of_year FOR a given year in date
; *   Author : P. Robert, CRPE, 1992

; *   Input  : iyear,idoy (idoy=1 FOR january 1)
; *   Output : imonth,iday
; ----------------------------------------------------------------------

      month=[31,28,31,30,31,30,31,31,30,31,30,31]

      cp_leap_year, iyear,ily

      IF(idoy GT 365 AND ily EQ 0) THEN BEGIN
           PRINT,'*** Rocotlib/cv_doty_to_date: idoy= ',idoy
           PRINT,'*** Rocotlib/cv_doty_to_date: iyear not a leap year'
           PRINT,'                      no more than 365 day'
           MESSAGE, /NONAME, /NOPREFIX,   '*** Rocotlib/cv_doty_to_date: iyear not a leap year'
                                    ENDIF

      IF(idoy LT 1) THEN BEGIN
        PRINT,'*** Rocotlib/cv_doty_to_date: idoy= ',idoy
        PRINT,'*** Rocotlib/cv_doty_to_date: idoy must be > 0'
        MESSAGE, /NONAME, /NOPREFIX,   '*** Rocotlib/cv_doty_to_date: idoy must be > 0'
                     ENDIF

      IF(ily EQ 1) THEN BEGIN
                   month(1)=29
                   ENDIF ELSE BEGIN
                   month(1)=28
                   ENDELSE

      m=0

      FOR im=1,12 DO BEGIN
        mp=m
        m=m+month(im-1)
         IF(idoy LE m) THEN GOTO, WWW20
      ENDFOR
      WWW20:

      imonth=im
      iday=idoy-mp

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cv_jul2000_to_date, jd00,iyear,imonth,iday

; ----------------------------------------------------------------------
; *   Class  : calendar modules of Rocotlib Software
; *   Object : convert_julian_day_2000 in date
; *   Author : P. Robert, CRPE, 1992
; *   Comment: compute date as year, month, day from julian day 2000

; *   Input  : jd00 julian day 2000 (0= 1/1/2000)
; *   Output : iyear,imonth,iday
; ----------------------------------------------------------------------


      jd50 = jd00 +18262

      cv_jul1950_to_date, jd50,iyear,imonth,iday

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cv_jul1950_to_date, jd50,iyear,imonth,iday

; ----------------------------------------------------------------------
; *   Class  : calendar modules of Rocotlib Software
; *   Object : convert_julian_day_1950 in date
; *   Author : P. Robert, CRPE, 1992
; *   Comment: compute date as year, month, day from julian day 1950

; *   Input  : jd50  julian day 1950 (0= 1/1/1950)
; *   Output : iyear,imonth,iday
; ----------------------------------------------------------------------

      jd1= -1

      FOR iy=1950,3000 DO BEGIN
      cp_leap_year, iy,ily
      jdp=jd1
       IF(ily EQ 1) THEN BEGIN
                      jd1=jd1+366
                    ENDIF ELSE BEGIN
                      jd1=jd1+365
                   ENDELSE
       IF(jd1 GE jd50) THEN BEGIN
                       iyear=iy
                       GOTO, WWW20
                       ENDIF

      ENDFOR

      PRINT,'*** Rocotlib/cv_jul1950_to_date: jd50= ',jd50
      PRINT,'*** Rocotlib/cv_jul1950_to_date: jd50 leads year > 3000'
      MESSAGE, /NONAME, /NOPREFIX,   '*** Rocotlib/cv_jul1950_to_date: jd50 leads year > 3000'

      WWW20:
      jd=jd50-jdp
      cv_doty_to_date, jd,iy,imonth,iday

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cv_weekn_to_date, iweek,iyear,imonth,iday

; ----------------------------------------------------------------------
; *   Class  : calendar modules of Rocotlib Software
; *   Object : convert_first_day_of_week_number in date
; *   Author : P. Robert, CRPE, 2001

; *   Input  : iweek,iyear
; *   Output : imonth,iday
; ----------------------------------------------------------------------


;  calcul de la date correspondant au premier jour de la semaine

; *** premier lundi de l'annee

      FOR ipl=1,7 DO BEGIN
        cv_date_to_dotw, iyear,1,ipl,idow
         IF(idow EQ 1) THEN GOTO, WWW20
      ENDFOR

      PRINT, '*** Rocotlib/cv_weekn_to_date: error in idow computation'
      MESSAGE, /NONAME, /NOPREFIX,    '*** Rocotlib/cv_weekn_to_date: error in idow computation'
      WWW20:

; *** semaine correspondant au premier lundi

      cv_date_to_weekn, iyear,1,ipl,iweek1

; *** jour de l'annee correspondant au lundi de la semaine

      idoy= ipl +(iweek-iweek1)*7

       IF(idoy LT 1) THEN BEGIN
                       iyear=iyear-1
                       imonth=12
                       iday=31+idoy
                       RETURN
                     ENDIF

; *** date correspondante

      cv_doty_to_date, idoy,iyear,imonth,iday

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cv_date_to_dotw, iyear,imonth,iday,idow

; ----------------------------------------------------------------------
; *   Class  : calendar modules of Rocotlib Software
; *   Object : convert_date in day_of_the_week
; *   Author : P. Robert, CRPE, 2001

; *   Input  : iyear,imonth,iday
; *   Output : idow (1-7)
; ----------------------------------------------------------------------

; WWW data jdref,julday /-100000,-100000/
      jdref =-100000L
      julday=-100000L

; *** jour julien du lundi de reference (le 1/1/2001 etait un lundi)

      cv_date_to_jul2000, 2001,1,1,jdref

; *** jour julien de la date demandee

      cv_date_to_jul2000, iyear,imonth,iday,julday

; *** calcul du jour de la semaine

      idiff= julday-jdref

      iweek= idiff/7
      idow= idiff -7*iweek +1
      IF(idow LT 1) THEN idow=idow+7

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cv_date_to_doty, iyear,imonth,iday,idoy

; ----------------------------------------------------------------------
; *   Class  : calendar modules of Rocotlib Software
; *   Object : convert_date in day_of_year with idoy=1 FOR january 1
; *   Author : P. Robert, CRPE, 1992

; *   Input  : iyear,imonth,iday   ex: 1990,10,17
; *   Output : idoy                ex: 290
; ----------------------------------------------------------------------

      month=[31,28,31,30,31,30,31,31,30,31,30,31]

      cp_leap_year, iyear,ily

       IF(ily EQ 1) THEN BEGIN
                      month(1)=29
                    ENDIF ELSE BEGIN
                      month(1)=28
                    ENDELSE

       IF(imonth LT 1 OR imonth GT 12) THEN BEGIN
         PRINT, '*** Rocotlib/cv_date_to_doty: imonth = ',imonth
         PRINT, '*** Rocotlib/cv_date_to_doty: imonth must be >0 and <13'
         MESSAGE, /NONAME, /NOPREFIX,    '*** Rocotlib/cv_date_to_doty: imonth must be >0 and <13'
       ENDIF

       IF(iday GT month(imonth-1)) THEN BEGIN
         PRINT, '*** Rocotlib/cv_date_to_doty: iday= ',iday, '  imonth=',imonth
         PRINT, '*** Rocotlib/cv_date_to_doty: this month has only',month(imonth-1),'days'
         MESSAGE, /NONAME, /NOPREFIX, '*** Rocotlib: error iday in cv_date_to_doty'
       ENDIF

      idoy=iday
      FOR i=1,imonth-1 DO BEGIN
       idoy=idoy+month(i-1)
      ENDFOR

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cv_hms_to_dech, ih,im,is,houday

; ----------------------------------------------------------------------
; *   Class  : calendar modules of Rocotlib Software
; *   Object : convert_hours_minutes_seconds in decimal hour of the day
; *   Author : P. Robert, CRPE, 1992

; *   Input  : ih,im,is
; *   Output : houday  decimal hour of the day
; ----------------------------------------------------------------------

      houday=float(ih)+float(im)/60.+float(is)/3600.

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cv_date_to_jul1950, iyear,imonth,iday,jd50

; ----------------------------------------------------------------------
; *   Class  : calendar modules of Rocotlib Software
; *   Object : convert_date in julian_day_1950 with jd50=0 FOR jan 1
; *   Author : P. Robert, CRPE, 1992

; *   Input  : iyear,imonth,iday   ex: 1990,10,17
; *   Output : jd50
; ----------------------------------------------------------------------

      IF(iyear LT 1950) THEN BEGIN
         PRINT, '*** Rocotlib/cv_date_to_jul1950: iyear= ',iyear
         PRINT, '*** Rocotlib/cv_date_to_jul1950: iyear must be > 1950'
         MESSAGE, /NONAME, /NOPREFIX,    '*** Rocotlib/cv_date_to_jul1950: iyear must be > 1950'
      ENDIF

      cv_date_to_doty, iyear,imonth,iday,idoy

      jd50=idoy-1

      FOR i=1950,iyear-1 DO BEGIN
      cp_leap_year, i,ily
      IF(ily EQ 1) THEN BEGIN
                     jd50=jd50+366
                   ENDIF ELSE BEGIN
                     jd50=jd50+365
                   ENDELSE
      ENDFOR

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cv_date_to_jul2000, iyear,imonth,iday,jd00

; ----------------------------------------------------------------------
; *   Class  : calendar modules of Rocotlib Software
; *   Object : convert_date in julian_day_2000 with jd00=0 FOR january 1
; *   Author : P. Robert, CRPE, 1992

; *   Input  : iyear,imonth,iday   ex: 2001,10,17
; *   Output : jd00 (may be negative)
; ----------------------------------------------------------------------

; WWW data jd50 /-1/
      COMMON ily,ily

      IF(iyear LT 1950) THEN BEGIN
         PRINT, '*** Rocotlib/cv_date_to_jul2000: iyear= ',iyear
         PRINT, '*** Rocotlib/cv_date_to_jul2000: iyear must be > 1950'
         MESSAGE, /NONAME, /NOPREFIX,    '*** Rocotlib/cv_date_to_jul2000: iyear must be > 1950'
      ENDIF

      cv_date_to_jul1950, iyear,imonth,iday,jd50

      jd00= jd50 - 18262

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cv_dhms_to_msotd, ih,im,is,ims,milday

; ----------------------------------------------------------------------
; *   Class  : calendar modules of Rocotlib Software
; *   Object : convert_hours_minutes_seconds_ms in millisec_of_day
; *   Author : P. Robert, CRPE, 2001

; *   Input  : ih,im,is,ims
; *   Output : milday  millisecond of the day
; ----------------------------------------------------------------------

      milday= ih*3600000L + im*60000L + is*1000L +ims*1L

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cv_dech_to_hms, houday,ih,im,is

; ----------------------------------------------------------------------
; *   Class  : calendar modules of Rocotlib Software
; *   Object : convert_decimal hour of the day in time
; *   Author : P. Robert, CRPE, 1992

; *   Input  : houday decimal hour of the day
; *   Output : ih,im,is
; ----------------------------------------------------------------------


      nj=FIX(ABS(houday)/24.)
      rdech=ABS(houday)-float(nj*24)

      ih=FIX(rdech)
      im=FIX((rdech-float(ih))*60.)
      is=FIX((rdech-float(ih)-float(im)/60.)*3600. +0.001)

      IF(is EQ 60) THEN BEGIN
                     is=0
                     im=im +1
                   ENDIF

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cv_msotd_to_hmsms, milday,ih,im,is,ims

; ----------------------------------------------------------------------
; *   Class  : calendar modules of Rocotlib Software
; *   Object : convert_millisec. of the day in time
; *   Author : P. Robert, CRPE, 2001

; *   Input  : milday millisecond of the day
; *   Output : ih,im,is,ims
; ----------------------------------------------------------------------


      nj= milday/86400000L
      mil2= milday-nj*86400000L

      IF(nj NE 0) THEN BEGIN
          PRINT, '*** Rocotlib/cv_msotd_to_hmsms: milday=',milday,' is > 86400000'             
          PRINT, '                      assumed:',mil2
      ENDIF

      ih=  milday/3600000L
      im= (milday-ih*3600000L)/60000L
      is= (milday-ih*3600000L -im*60000L)/1000L
      ims= milday-ih*3600000L -im*60000L -is*1000L

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cv_date_to_weekn, iyear,imonth,iday,iweek

; ----------------------------------------------------------------------
; *   Class  : calendar modules of Rocotlib Software
; *   Object : convert_date in week_of_the_year
; *   Author : P. Robert, CRPE, 2001, revised SD April 2017

; *   Input  : iyear,imonth,iday
; *   Output : iweek
; ----------------------------------------------------------------------

;     la semaine 1 de toute anne est celle qui contient le 4 janvier 
;     ou celle qui contient le 1er jeudi de janvier(norme ISO-8601). 

; *** calcul du jour de la semaine du 1er janvier de l'annee

      cv_date_to_dotw, iyear,1,1,idow

; *** calcul du lundi commencant la semaine 2

      IF(idow EQ 1) THEN idoty2= 8
      IF(idow EQ 2) THEN idoty2= 7
      IF(idow EQ 3) THEN idoty2= 6
      IF(idow EQ 4) THEN idoty2= 5
      IF(idow EQ 5) THEN idoty2=11
      IF(idow EQ 6) THEN idoty2=10
      IF(idow EQ 7) THEN idoty2= 9

; *** jour de l'annee de la date demandee

      cv_date_to_doty, iyear,imonth,iday,idoty

; *** jour de la semaine de la date demandee

      cv_date_to_dotw, iyear,imonth,iday,idow

; *** jour de l'annee du lundi de la semaine de la date demandee

      idotL= idoty -idow +1

; *** numero de la semaine

      iweek= (idotL -idoty2)/7 +2

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO g_gei_geo_dipole_dir, dxgei,dygei,dzgei, dxgeo,dygeo,dzgeo

; ----------------------------------------------------------------------
; *   Class  : give modules of Rocotlib Software
; *   Object : give_dipole_direction in GEI and GEO system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: values are extracted from common

; *   Input  : none
; *   Output : dxgei,dygei,dzgei  cartesian dipole GEI coordinates
;              dxgeo,dygeo,dzgeo  cartesian dipole GEO coordinates
; ----------------------------------------------------------------------

      COMMON timp01,  gs1,gs2,gs3, gm1,gm2,gm3, ge1,ge2,ge3,gr1,gr2,gr3
      COMMON timp02,  ps1,ps2,ps3, gd1,gd2,gd3, pe1,pe2,pe3, pr1,pr2,pr3

      dxgei=gm1
      dygei=gm2
      dzgei=gm3

      dxgeo=gd1
      dygeo=gd2
      dzgeo=gd3

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO g_gsm_dipole_tilt_angle, diptan

; ----------------------------------------------------------------------
; *   Class  : give modules of Rocotlib Software
; *   Object : give_dipole_tilt_angle in radians
; *   Author : P. Robert, CRPE, 1992
; *   Comment: values are extracted from common

; *   Input  : none
; *   Output : diptan (radians)
; ----------------------------------------------------------------------

      COMMON timp11,  stetq,ctetq, sdze,cdze, smu,cmu, sphi,cphi

      diptan=ATAN(smu,cmu)

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO g_gei_geo_ecliptic_dir, exgei,eygei,ezgei, exgeo,eygeo,ezgeo

; ----------------------------------------------------------------------
; *   Class  : give modules of Rocotlib Software
; *   Object : give_ecliptic_direction in GEI and GEO system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: values are extracted from common

; *   Input  : none
; *   Output : exgei,eygei,ezgei  cartesian ecliptic GEI coordinates
;              exgeo,eygeo,ezgeo  cartesian ecliptic GEO coordinates
; ----------------------------------------------------------------------

      COMMON timp01,  gs1,gs2,gs3, gm1,gm2,gm3, ge1,ge2,ge3,gr1,gr2,gr3
      COMMON timp02,  ps1,ps2,ps3, gd1,gd2,gd3, pe1,pe2,pe3,pr1,pr2,pr3

      exgei=ge1
      eygei=ge2
      ezgei=ge3

      exgeo=pe1
      eygeo=pe2
      ezgeo=pe3

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO g_gei_geo_sun_rot, rxgei,rygei,rzgei,rxgeo,rygeo,rzgeo

; ----------------------------------------------------------------------
; *   Class  : give modules of Rocotlib Software
; *   Object : give_sun_rotation_direction in GEI and GEO system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: values are extracted from common

; *   Input  : none
; *   Output : rxgei,rygei,rzgei cartesian sun rotation GEI coordinates
;              rxgeo,rygeo,rzgeo cartesian sun rotation GEO coordinates
; ----------------------------------------------------------------------

      COMMON timp01,  gs1,gs2,gs3, gm1,gm2,gm3, ge1,ge2,ge3,gr1,gr2,gr3
      COMMON timp02,  ps1,ps2,ps3, gd1,gd2,gd3, pe1,pe2,pe3,pr1,pr2,pr3

      rxgei=gr1
      rygei=gr2
      rzgei=gr3

      rxgeo=pr1
      rygeo=pr2
      rzgeo=pr3

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO g_gei_geo_sun_dir, sxgei,sygei,szgei,sxgeo,sygeo,szgeo

; ----------------------------------------------------------------------
; *   Class  : give modules of Rocotlib Software
; *   Object : give_sun_direction in GEI and GEO system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: values are extracted from common

; *   Input  : none
; *   Output : sxgei,sygei,szgei  cartesian sun GEI coordinates
;              sxgeo,sygeo,szgeo  cartesian sun GEO coordinates
; ----------------------------------------------------------------------

      COMMON timp01,  gs1,gs2,gs3, gm1,gm2,gm3, ge1,ge2,ge3,gr1,gr2,gr3
      COMMON timp02,  ps1,ps2,ps3, gd1,gd2,gd3, pe1,pe2,pe3,pr1,pr2,pr3

      sxgei=gs1
      sygei=gs2
      szgei=gs3

      sxgeo=ps1
      sygeo=ps2
      szgeo=ps3

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO g_gei_sun_param, gmst,slon,sras,sdec,obli

; ----------------------------------------------------------------------
; *   Class  : give modules of Rocotlib Software
; *   Object : give_sun_parameter dependant of time in GEI system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: values are extracted from common

; *   Input  : none
; *   Output : gmst   greenwich mean sideral time (radians)
;              slon   longitude along ecliptic (radians)
;              sras   right ascension (radians)
;              sdec   declination of the sun (radians)
; ----------------------------------------------------------------------

      COMMON timp00,  sgst,cgst, gst,slong,srasn,sdecl,obliq

      gmst= gst
      slon= slong
      sras= srasn
      sdec= sdecl
      obli= obliq

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO g_rocot_version_number, vernum,verdat

; WWW  character*14 verdat

; ----------------------------------------------------------------------
; *   Class  : give modules of Rocotlib Software
; *   Object : give_version_number and modification date of the library
; *   Author : P. Robert, CRPE, 1992
; *   Comment: values are extracted from common

; *   Input  : none
; *   Output : vernum (ex 1.0) and verdat (ex: 'January 1995')
; ----------------------------------------------------------------------

;     For previous versions informations, see PRINT_rocot_info subrout.

      vernum=3.2
      verdat='Fev 2020'

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO print_rocot_info

; ----------------------------------------------------------------------
; *   Class  : PRINT modules of Rocotlib Software
; *   Object : PRINT_library_informations
; *   Author : P. Robert, CRPE, 1992
; *   Comment: could be a main program

; *   Input  : none
; *   Output : none; PRINT infos on output
; ----------------------------------------------------------------------

      g_rocot_version_number, vernum,verdat

      fmt100="(3a)"
      fmt200="(a,f4.1,2a)"

      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt100, '*****************************************************'
      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt100, '    Coordinates Transformation Library  ROCOTLIB'
      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt200, '        Revised Version ',vernum, ' - ',verdat
      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt100, '               ___________________'
      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt100, '              initially supported by '
      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt100, '              EUROPEAN  SPACE  AGENCY'
      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt100, '           Study of the Cluster Mission'
      PRINT, FORMAT=fmt100, '             Planning Related Aspects'
      PRINT, FORMAT=fmt100, '      within the Numerical Simulations Network'
      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt100, '        Patrick ROBERT, CRPE, November 1992'
      PRINT, FORMAT=fmt100, '               ___________________'
      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt100, ' version 1.0, November 1992                          '
      PRINT, FORMAT=fmt100, ' version 1.1, July     1993                          '
      PRINT, FORMAT=fmt100, ' version 1.2, January  1995                          '
      PRINT, FORMAT=fmt100, ' version 1.3, July     2000 (Jul. day 2000/sun dir)  '
      PRINT, FORMAT=fmt100, ' version 1.4, June     2001 (for automatic docum.)   '
      PRINT, FORMAT=fmt100, ' version 1.5, December 2001 (add cp_sunrise_sunset   '
      PRINT, FORMAT=fmt100, ' version 1.6, Juin     2002 (upgrade IGRF -> 2005)   '
      PRINT, FORMAT=fmt100, ' version 1.7, December 2002 (Version for CDPP)       '
      PRINT, FORMAT=fmt100, ' version 1.8, November 2003 (add t_sr2_to_sr)        '
      PRINT, FORMAT=fmt100, ' version 1.9, March    2004 (compatibility with IDL) '
      PRINT, FORMAT=fmt100, ' version 2.0, November 2006 (Update  IGRF -> 2010)   '
      PRINT, FORMAT=fmt100, ' version 2.1, November 2006 (Update  IGRF -> 2015)   '
      PRINT, FORMAT=fmt100, ' version 2.2, December 2011 (cp_sunrise_sunset polar)'
      PRINT, FORMAT=fmt100, ' version 3.0, May      2017 (IGRF->table geomag. pole'
      PRINT, FORMAT=fmt100, '                     -> 2020 +some other coord. sys.)'
      PRINT, FORMAT=fmt100, ' version 3.1, January  2019 (compatibility with V2.2)'
      PRINT, FORMAT=fmt100, ' version 3.2, February 2020 (add trans. Euler angles)'
      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt100, '      Copyright 1992, Patrick ROBERT, CNRS-ESA,'
      PRINT, FORMAT=fmt100, '               All Rights reserved'
      PRINT, FORMAT=fmt100, '               ___________________'
      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt100, '    For details, see the orginal document untitled:'
      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt100, '              CLUSTER Software Tools'
      PRINT, FORMAT=fmt100, '      Part I: Coordinate Transformation Library'
      PRINT, FORMAT=fmt100, '          Document de travail DT/CRPE/1231'
      PRINT, FORMAT=fmt100, '        Patrick Robert, CRPE/TID, Juillet 1993'
      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt100, '    Available at CDPP:'
      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt100, '    ROCOTLIB: a coordinate Transformation Library'
      PRINT, FORMAT=fmt100, '              for Solar-Terrestrial studies'
      PRINT, FORMAT=fmt100, '    Patrick ROBERT, version 1.7 - January 2003,'
      PRINT, FORMAT=fmt100, '        Rapport Interne no RI-CETP/02/2003'
      PRINT, FORMAT=fmt100, '               ___________________'
      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt100, ' The present version 3.0 contains new transformations'
      PRINT, FORMAT=fmt100, ' (TPN, MVA), and the cp_sunrise_sunset sub. allowing '
      PRINT, FORMAT=fmt100, ' computation of sunrise and sunset anywhere on Earth,'
      PRINT, FORMAT=fmt100, ' including polar zones.          '
      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt100, ' The determination of the dipole axis direction'
      PRINT, FORMAT=fmt100, ' has been replaced by a table available from years'
      PRINT, FORMAT=fmt100, ' 1900 to 2020, in order to have an easy way to code '
      PRINT, FORMAT=fmt100, ' maintenance in the coming years.'
      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt100, ' Operations on matrix (somme, product, inversion, '
      PRINT, FORMAT=fmt100, ' computation of eigen vectors, diagonalization...)'
      PRINT, FORMAT=fmt100, ' required for Minimum Variance Analysis coordinates'
      PRINT, FORMAT=fmt100, ' have been added with prefix "mat_".'
      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt100, ' The previous and temporary V3.1 is the same as 3.0 '
      PRINT, FORMAT=fmt100, ' but is compatible with V2.2 version concerning '
      PRINT, FORMAT=fmt100, ' PRO name , ex: tmaggsm -> t_mag_to_gsm '
      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt100, ' The present V3.2 version contains transformations '
      PRINT, FORMAT=fmt100, ' with Euler angles and their interpolation. '
      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt100, ' Original code developped in Fortran 77.'
      PRINT, FORMAT=fmt100, ' Fortran 90, C, and IDL codes are also available.'
      PRINT, FORMAT=fmt100, ' '
      PRINT, FORMAT=fmt100, '*****************************************************'
      PRINT, FORMAT=fmt100, ' '

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO r_coordinate_values, x,y,z,cs

; ----------------------------------------------------------------------
; *   Class  : READ modules of Rocotlib Software
; *   Object : READ coordinate values from input
; *   Author : P. Robert, CRPE, 2002

; *   Comment: READ cs and x,y,z cartesian  or spherical coordinates
; *            PRINT error if cs is not valid, and ask again

; *   Input  : cs (c or s) and x,y,z on standard input
; *   Output : cs and x,y,z always in cartesian coordinates
; ----------------------------------------------------------------------

      pisd=ACOS(-1.)/180.

      fmt100="(a)"
      fmt110="(e14.6,1x,f8.3,1x,f8.3)"
      fmt120="(1x,a,3(e11.3),a)"
      fmt200="(1x,a,e11.3,f8.3,f8.3,a)"

      WWW10:

      PRINT, FORMAT=fmt100, 'input coordinates: cartesian (c) or spherical (s)'
      READ, cs
      PRINT, FORMAT=fmt100, cs

      IF(cs NE 's' AND cs NE 'c') THEN BEGIN
                                    PRINT, 'only c or s'
                                    PRINT, 'again...'
                                    GOTO, WWW10
      ENDIF
      IF(cs EQ 's') THEN BEGIN
                      PRINT, FORMAT=fmt100, 'R,tet, phi (RE,deg.) ?'
                      READ,  r,tet, phi
                      PRINT, FORMAT=fmt110,  r,tet, phi

                      tet=tet*pisd
                      phi=phi*pisd
                      t_sph_to_car, r,tet,phi,x,y,z
                      PRINT, FORMAT=fmt120, 'then x,y,z = ',x,y,z,'(RE)'

                    ENDIF ELSE BEGIN
                      PRINT, 'x,y,z ? (RE) '
                      READ,  x,y,z
                      PRINT,  x,y,z

                      t_car_to_sph, x,y,z,r,tet,phi
                      tet=tet/pisd
                      phi=phi/pisd
                      PRINT, FORMAT=fmt200, 'then R,tet,phi= ',r,tet,phi,' (RE,deg.)'
                    ENDELSE

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO r_coordinate_system, csys

; ----------------------------------------------------------------------
; *   Class  : READ modules of Rocotlib Software
; *   Object : READ coordinate system from input and check validity
; *   Author : P. Robert, CRPE, 2002

; *   Comment: READ csys string variable and check validity
; *            (only gei, geo, mag, sma, gsm, gse, gsq)
; *            PRINT error if csys is not valid, and ask again

; *   Input  : csys on standard input
; *   Output : csys
; ----------------------------------------------------------------------

      fmt100="(a)"

      WWW10:

      PRINT, FORMAT=fmt100, 'repere ? (gei, geo, mag, sma, gsm, gse, gsq)'
      READ,csys
      PRINT, FORMAT=fmt100,csys

      IF(csys EQ 'gei') THEN RETURN
      IF(csys EQ 'geo') THEN RETURN
      IF(csys EQ 'mag') THEN RETURN
      IF(csys EQ 'sma') THEN RETURN
      IF(csys EQ 'gsm') THEN RETURN
      IF(csys EQ 'gse') THEN RETURN
      IF(csys EQ 'gsq') THEN RETURN

      PRINT, 'Only gei, geo, mag, sma, gsm, gse or gsq please...'
      PRINT, 'again...'
      GOTO, WWW10

      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO r_datef, ifc, iyear,imonth,iday

; ----------------------------------------------------------------------
; *   Class  : READ modules of Rocotlib Software
; *   Object : read_date from a file and check validity
; *   Author : P. Robert, CRPE, 1992

; *   Comment: test if year is gt 1900
; *            test if imonth is not greater than 12
;              test if iday is not greather than lengh of the month, $
;                        takink into account the leap years.
;               PRINT error if date is not valid, and ask again

; *   Input  : iyear,imonth,iday read from file
; *   Output : iyear,imonth,iday
; ----------------------------------------------------------------------

      month=[31,28,31,30,31,30,31,31,30,31,30,31]
      iyear=0
      imonth=0
      iday=0

      fmt100="(a)"
      fmt110="(1x,i4,1x,i2,1x,i2)"

      WWW10:
      PRINT, FORMAT=fmt100, ' iyear,imonth,iday ? (ex: 1990,10,17)'
      READF, ifc,  iyear,imonth,iday
      PRINT, FORMAT=fmt110,  iyear,imonth,iday



      IF(iyear LT 1900) THEN BEGIN
         PRINT, '*** Rocotlib/r_date: iyear must be greater than 1900'
         PRINT, '                     again ...'
                        GOTO, WWW10
      ENDIF

      cp_leap_year, iyear,ily

      IF(ily EQ 1) THEN BEGIN
                   month(1)=29
                   ENDIF ELSE BEGIN
                   month(1)=28
                   ENDELSE

      IF(imonth LT 1 OR imonth GT 12) THEN BEGIN
          PRINT, '*** Rocotlib/r_date: imonth must be between 1 or 12 '
          PRINT, '                     again...'
                     GOTO, WWW10
      ENDIF

      IF(iday GT month(imonth-1)) THEN BEGIN
                     PRINT, '*** Rocotlib/r_date: this month has only', $
                                  month(imonth-1),'days'
                     PRINT, '                     again...'
                GOTO, WWW10
      ENDIF
;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO r_timef, ifc, ih,im,is

; ----------------------------------------------------------------------
; *   Class  : READ modules of Rocotlib Software
; *   Object : read_time from file and check validity
; *   Author : P. Robert, CRPE, 1992
; *   Comment: READ hour, minute and second and verifie validity
;              ih must be between 1 and 23, im and is between 1 and 59
;              PRINT error if time is not valid, and ask again

; *   Input  : ih,im,is read from file
; *   Output : ih,im,is
; ----------------------------------------------------------------------

      ih=0
      is=0
      im=0

      fmt100="(a)"
      fmt110="(1x,i2,1x,i2,1x,i2)"

      WWW10:
      PRINT, FORMAT=fmt100, ' hour, minute, second ? (ex: 10,45,50)'
      READF, ifc, ih,im,is
      PRINT, FORMAT=fmt110, ih,im,is


      IF(ih LT 0 OR ih GT 23) THEN BEGIN
                              PRINT, 'hour between 0 and 23 please'
                              PRINT, 'again...'
                              GOTO, WWW10
      ENDIF

      IF(im LT 0 OR im GT 59) THEN BEGIN
                              PRINT, 'minute between 0 and 59 please'
                              PRINT, 'again...'
                              GOTO, WWW10
      ENDIF

      IF(is LT 0 OR is GT 59) THEN BEGIN
                              PRINT, 'second between 0 and 59 please'
                              PRINT, 'again...'
                              GOTO, WWW10
      ENDIF

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_car_to_sph, x,y,z,r,teta,phi

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_car_to_sph: CAR -> SPH  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: none

; *   Input  :   x,y,z        cartesian coordinates
; *   Output :   r,teta,phi   spherical coordinates (angles in radians)
; ----------------------------------------------------------------------


      teta=0.
      phi=0.
      sq=x^2+y^2
      r=SQRT(sq+z^2)
      pi=ACOS(-1.)
      pisd=pi/180.

      IF(r LT 1.e-30) THEN RETURN

; *** en dessous de 1/10000 degres, on considere que teta =0 ou 180
;     et phi indefini, mis a zero

      IF(sq/r GT 1.7e-6) THEN BEGIN
                   phi=ATAN(y,x)
                   IF(phi LT -179.999*pisd AND phi GT -180.*pisd)  THEN phi=pi
                   IF(phi LT  0.0001*pisd AND phi GT -0.0001*pisd) THEN phi=0.
                   teta=ACOS(z/r)
                   RETURN
      ENDIF

      IF(z LT 0.) THEN BEGIN
                   teta=ACOS(-1.)
                   ENDIF ELSE BEGIN
                   teta=0.
                   ENDELSE

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_dm_to_geo, xdme,ydme,zdme,rlat,rlong,xgeo,ygeo,zgeo

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_dme_to_geo: DM  -> GEO  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xdme,ydme,zdme   cartesian dme coordinates
;              rlat,rlong       latitude and longitude of the point
;                               of observation (radians)
; *   Output : xgeo,ygeo,zgeo   cartesian geo coordinates
; ----------------------------------------------------------------------


      COMMON timp02,  ps1,ps2,ps3, gd1,gd2,gd3, pe1,pe2,pe3, pr1,pr2,pr3

      q=COS(rlat)
      r1=q*COS(rlong)
      r2=q*SIN(rlong)
      r3=SIN(rlat)

      y1= FLOAT(Dprod(gd2,r3) - Dprod(gd3,r2))
      y2= FLOAT(Dprod(gd3,r1) - Dprod(gd1,r3))
      y3= FLOAT(Dprod(gd1,r2) - Dprod(gd2,r1))

      q=SQRT(y1^2 + y2^2 + y3^2)

      y1=y1/q
      y2=y2/q
      y3=y3/q

      x1= FLOAT(Dprod(y2,gd3) - Dprod(y3,gd2))
      x2= FLOAT(Dprod(y3,gd1) - Dprod(y1,gd3))
      x3= FLOAT(Dprod(y1,gd2) - Dprod(y2,gd1))


      xgeo= x1*xdme + y1*ydme + gd1*zdme
      ygeo= x2*xdme + y2*ydme + gd2*zdme
      zgeo= x3*xdme + y3*ydme + gd3*zdme

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gei_to_geo, xgei,ygei,zgei,xgeo,ygeo,zgeo

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gei_to_geo: GEI -> GEO  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgei,ygei,zgei   cartesian gei coordinates
; *   Output : xgeo,ygeo,zgeo   cartesian geo coordinates
; ----------------------------------------------------------------------


      COMMON timp00,  sgst,cgst, gst,slong,srasn,sdecl,obliq

      xgeo=  cgst*xgei + sgst*ygei
      ygeo= -sgst*xgei + cgst*ygei
      zgeo=  zgei

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gei_to_gse, xgei,ygei,zgei,xgse,ygse,zgse

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gei_to_gse: GEI -> GSE  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgei,ygei,zgei cartesian gei coordinates
; *   Output : xgse,ygse,zgse cartesian gse coordinates
; ----------------------------------------------------------------------


      COMMON timp01,  gs1,gs2,gs3, gm1,gm2,gm3, ge1,ge2,ge3,gr1,gr2,gr3
      COMMON timp03,  gmgs1,gmgs2,gmgs3,  gegs1,gegs2,gegs3

      xgse=   gs1*xgei +   gs2*ygei +   gs3*zgei
      ygse= gegs1*xgei + gegs2*ygei + gegs3*zgei
      zgse=   ge1*xgei +   ge2*ygei +   ge3*zgei

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gei_to_gsm, xgei,ygei,zgei,xgsm,ygsm,zgsm

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gei_to_gsm: GEI -> GSM  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgei,ygei,zgei   cartesian gei coordinates
; *   Output : xgsm,ygsm,zgsm   cartesian gsm coordinates
; ----------------------------------------------------------------------


      COMMON timp01,  gs1,gs2,gs3, gm1,gm2,gm3, ge1,ge2,ge3,gr1,gr2,gr3
      COMMON timp09,  yeigm1,yeigm2,yeigm3,  zeigm1,zeigm2,zeigm3

      xgsm=    gs1*xgei +    gs2*ygei +    gs3*zgei
      ygsm= yeigm1*xgei + yeigm2*ygei + yeigm3*zgei
      zgsm= zeigm1*xgei + zeigm2*ygei + zeigm3*zgei

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gei_to_gseq, xgei,ygei,zgei,xgsq,ygsq,zgsq

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gei_to_gsq: GEI -> GSEQ system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgei,ygei,zgei   cartesian gei coordinates
; *   Output : xgsq,ygsq,zgsq   cartesian gsq coordinates
; ----------------------------------------------------------------------


      COMMON timp01,  gs1,gs2,gs3, gm1,gm2,gm3, ge1,ge2,ge3,gr1,gr2,gr3
      COMMON timp10,  yeigq1,yeigq2,yeigq3,  zeigq1,zeigq2,zeigq3

      xgsq=    gs1*xgei +    gs2*ygei +    gs3*zgei
      ygsq= yeigq1*xgei + yeigq2*ygei + yeigq3*zgei
      zgsq= zeigq1*xgei + zeigq2*ygei + zeigq3*zgei

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gei_to_mag, xgei,ygei,zgei,xmag,ymag,zmag

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gei_to_mag: GEI -> MAG  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgei,ygei,zgei   cartesian gei coordinates
; *   Output : xmag,ymag,zmag   cartesian mag coordinates
; ----------------------------------------------------------------------


      COMMON timp01,  gs1,gs2,gs3, gm1,gm2,gm3, ge1,ge2,ge3,gr1,gr2,gr3
      COMMON timp07,  xeima1,xeima2,xeima3,  yeima1,yeima2,yeima3

      xmag= xeima1*xgei + xeima2*ygei + xeima3*zgei
      ymag= yeima1*xgei + yeima2*ygei + yeima3*zgei
      zmag=    gm1*xgei +    gm2*ygei +    gm3*zgei

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gei_to_sm, xgei,ygei,zgei,xsma,ysma,zsma

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gei_to_sma: GEI -> SM   system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgei,ygei,zgei   cartesian gei coordinates
; *   Output : xsma,ysma,zsma   cartesian sma coordinates
; ----------------------------------------------------------------------


      COMMON timp01,  gs1,gs2,gs3, gm1,gm2,gm3, ge1,ge2,ge3,gr1,gr2,gr3
      COMMON timp08,  xeism1,xeism2,xeism3,  yeism1,yeism2,yeism3

      xsma= xeism1*xgei + xeism2*ygei + xeism3*zgei
      ysma= yeism1*xgei + yeism2*ygei + yeism3*zgei
      zsma=    gm1*xgei +    gm2*ygei +    gm3*zgei

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_geo_to_dm, xgeo,ygeo,zgeo,rlat,rlong,xdme,ydme,zdme

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_geo_to_dme: GEO -> DM   system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgeo,ygeo,zgeo   cartesian geo coordinates
;              rlat,rlong       latitude and longitude of the point
;                               of observation (radians)
; *   Output : xdme,ydme,zdme   cartesian dme coordinates
; ----------------------------------------------------------------------


      COMMON timp02,  ps1,ps2,ps3, gd1,gd2,gd3, pe1,pe2,pe3, pr1,pr2,pr3

      q=COS(rlat)
      r1=q*COS(rlong)
      r2=q*SIN(rlong)
      r3=SIN(rlat)

      y1= FLOAT(Dprod(gd2,r3) - Dprod(gd3,r2))
      y2= FLOAT(Dprod(gd3,r1) - Dprod(gd1,r3))
      y3= FLOAT(Dprod(gd1,r2) - Dprod(gd2,r1))

      q=SQRT(y1^2 + y2^2 + y3^2)

      y1=y1/q
      y2=y2/q
      y3=y3/q

      x1= FLOAT(Dprod(y2,gd3) - Dprod(y3,gd2))
      x2= FLOAT(Dprod(y3,gd1) - Dprod(y1,gd3))
      x3= FLOAT(Dprod(y1,gd2) - Dprod(y2,gd1))

      xdme=  x1*xgeo +  x2*ygeo +  x3*zgeo
      ydme=  y1*xgeo +  y2*ygeo +  y3*zgeo
      zdme= gd1*xgeo + gd2*ygeo + gd3*zgeo

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_geo_to_gei, xgeo,ygeo,zgeo,xgei,ygei,zgei

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_geo_to_gei: GEO -> GEI  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgeo,ygeo,zgeo cartesian geo coordinates
; *   Output : xgei,ygei,zgei cartesian gei coordinates
; ----------------------------------------------------------------------


      COMMON timp00,  sgst,cgst, gst,slong,srasn,sdecl,obliq

      xgei= cgst*xgeo - sgst*ygeo
      ygei= sgst*xgeo + cgst*ygeo
      zgei= zgeo

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_geo_to_gse, xgeo,ygeo,zgeo,xgse,ygse,zgse

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_geo_to_gse: GEO -> GSE  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgeo,ygeo,zgeo   cartesian geo coordinates
; *   Output : xgse,ygse,zgse   cartesian gse coordinates
; ----------------------------------------------------------------------


      COMMON timp02,  ps1,ps2,ps3, gd1,gd2,gd3, pe1,pe2,pe3, pr1,pr2,pr3
      COMMON timp06,  peps1,peps2,peps3,  prps1,prps2,prps3

      xgse=   ps1*xgeo +   ps2*ygeo +   ps3*zgeo
      ygse= peps1*xgeo + peps2*ygeo + peps3*zgeo
      zgse=   pe1*xgeo +   pe2*ygeo +   pe3*zgeo

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_geo_to_gsm, xgeo,ygeo,zgeo,xgsm,ygsm,zgsm

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_geo_to_gsm: GEO -> GSM  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgeo,ygeo,zgeo   cartesian geo coordinates
; *   Output : xgsm,ygsm,zgsm   cartesian gsm coordinates
; ----------------------------------------------------------------------


      COMMON timp02,  ps1,ps2,ps3, gd1,gd2,gd3, pe1,pe2,pe3, pr1,pr2,pr3
      COMMON timp14,  yeogm1,yeogm2,yeogm3,  zeogm1,zeogm2,zeogm3

      xgsm=    ps1*xgeo +    ps2*ygeo +    ps3*zgeo
      ygsm= yeogm1*xgeo + yeogm2*ygeo + yeogm3*zgeo
      zgsm= zeogm1*xgeo + zeogm2*ygeo + zeogm3*zgeo

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_geo_to_gseq, xgeo,ygeo,zgeo,xgsq,ygsq,zgsq

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_geo_to_gsq: GEO -> GSEQ system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgeo,ygeo,zgeo   cartesian geo coordinates
; *   Output : xgsq,ygsq,zgsq   cartesian gsq coordinates
; ----------------------------------------------------------------------


      COMMON timp02,  ps1,ps2,ps3, gd1,gd2,gd3, pe1,pe2,pe3, pr1,pr2,pr3
      COMMON timp15,  yeogq1,yeogq2,yeogq3,  zeogq1,zeogq2,zeogq3

      xgsq=    ps1*xgeo +    ps2*ygeo +    ps3*zgeo
      ygsq= yeogq1*xgeo + yeogq2*ygeo + yeogq3*zgeo
      zgsq= zeogq1*xgeo + zeogq2*ygeo + zeogq3*zgeo

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_geo_to_mag, xgeo,ygeo,zgeo,xmag,ymag,zmag

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_geo_to_mag: GEO -> MAG  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgeo,ygeo,zgeo   cartesian geo coordinates
; *   Output : xmag,ymag,zmag   cartesian mag coordinates
; ----------------------------------------------------------------------


      COMMON timp02,  ps1,ps2,ps3, gd1,gd2,gd3, pe1,pe2,pe3, pr1,pr2,pr3
      COMMON timp12,  xeoma1,xeoma2,xeoma3,  yeoma1,yeoma2,yeoma3

      xmag= xeoma1*xgeo + xeoma2*ygeo + xeoma3*zgeo
      ymag= yeoma1*xgeo + yeoma2*ygeo + yeoma3*zgeo
      zmag=    gd1*xgeo +    gd2*ygeo +    gd3*zgeo

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_geo_to_sm, xgeo,ygeo,zgeo,xsma,ysma,zsma

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_geo_to_sma: GEO -> SM   system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgeo,ygeo,zgeo   cartesian geo coordinates
; *   Output : xsma,ysma,zsma   cartesian sma coordinates
; ----------------------------------------------------------------------


      COMMON timp02,  ps1,ps2,ps3, gd1,gd2,gd3, pe1,pe2,pe3, pr1,pr2,pr3
      COMMON timp13,  xeosm1,xeosm2,xeosm3,  yeosm1,yeosm2,yeosm3

      xsma= xeosm1*xgeo + xeosm2*ygeo + xeosm3*zgeo
      ysma= yeosm1*xgeo + yeosm2*ygeo + yeosm3*zgeo
      zsma=    gd1*xgeo +    gd2*ygeo +    gd3*zgeo

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_geo_to_vdh, xgeo,ygeo,zgeo,rlat,rlong,xvdh,yvdh,zvdh

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_geo_to_vdh: GEO -> VDH  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: local system, non time dependent

; *   Input  : xgeo,ygeo,zgeo   cartesian geo coordinates
;              rlat,rlong       latitude and longitude of the point
;                               of observation (radians)
; *   Output : xvdh,yvdh,zvdh   cartesian vdh coordinates
; ----------------------------------------------------------------------


      q=COS(rlat)
      r1=q*COS(rlong)
      r2=q*SIN(rlong)
      r3=SIN(rlat)

      v1=r1
      v2=r2
      v3=r3

      q12=SQRT(r1^2 + r2^2)

      d1= -r2/q12
      d2=  r1/q12
      d3=  0.

      h1= -r1*r3/q12
      h2= -r2*r3/q12
      h3=  q12

      xvdh= v1*xgeo + v2*ygeo + v3*zgeo
      yvdh= d1*xgeo + d2*ygeo + d3*zgeo
      zvdh= h1*xgeo + h2*ygeo + h3*zgeo

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gse_to_gei, xgse,ygse,zgse,xgei,ygei,zgei

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gse_to_gei: GSE -> GEI  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgse,ygse,zgse    cartesian gse coordinates
; *   Output : xgei,ygei,zgei    cartesian gei coordinates
; ----------------------------------------------------------------------


      COMMON timp01,  gs1,gs2,gs3, gm1,gm2,gm3, ge1,ge2,ge3,gr1,gr2,gr3
      COMMON timp03,  gmgs1,gmgs2,gmgs3,  gegs1,gegs2,gegs3

      xgei= gs1*xgse + gegs1*ygse + ge1*zgse
      ygei= gs2*xgse + gegs2*ygse + ge2*zgse
      zgei= gs3*xgse + gegs3*ygse + ge3*zgse

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gse_to_geo, xgse,ygse,zgse,xgeo,ygeo,zgeo

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gse_to_geo: GSE -> GEO  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgse,ygse,zgse   cartesian gse coordinates
; *   Output : xgeo,ygeo,zgeo   cartesian geo coordinates
; ----------------------------------------------------------------------


      COMMON timp02,  ps1,ps2,ps3, gd1,gd2,gd3, pe1,pe2,pe3, pr1,pr2,pr3
      COMMON timp06,  peps1,peps2,peps3,  prps1,prps2,prps3

      xgeo= ps1*xgse + peps1*ygse + pe1*zgse
      ygeo= ps2*xgse + peps2*ygse + pe2*zgse
      zgeo= ps3*xgse + peps3*ygse + pe3*zgse

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gse_to_gsm, xgse,ygse,zgse,xgsm,ygsm,zgsm

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gse_to_gsm: GSE -> GSM  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgse,ygse,zgse   cartesian gse coordinates
; *   Output : xgsm,ygsm,zgsm   cartesian gsm coordinates
; ----------------------------------------------------------------------


      COMMON timp11,  stetq,ctetq, sdze,cdze, smu,cmu, sphi,cphi

      xgsm= xgse
      ygsm=  cdze*ygse + sdze*zgse
      zgsm= -sdze*ygse + cdze*zgse

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gse_to_gseq, xgse,ygse,zgse,xgsq,ygsq,zgsq

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gse_to_gsq: GSE -> GSEQ system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgse,ygse,zgse   cartesian gse coordinates
; *   Output : xgsq,ygsq,zgsq   cartesian gsq coordinates
; ----------------------------------------------------------------------


      COMMON timp11,  stetq,ctetq, sdze,cdze, smu,cmu, sphi,cphi

      xgsq= xgse
      ygsq= ctetq*ygse - stetq*zgse
      zgsq= stetq*ygse + ctetq*zgse

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gse_to_mfa, xgse,ygse,zgse,bx,by,bz,xmfa,ymfa,zmfa

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gse_to_mfa: GSE -> MFA  system
; *   Author : P. Robert, LPP , 2016
; *   Comment: local system, non time dependent

; *   Input  : xgse,ygse,gsez cartesian gse coordinates
;              bx,  by,  bz   cartesian gse coordinates of DC mag field
; *   Output : xmfa,ymfa,zmfa cartesian mfa coordinates
; ----------------------------------------------------------------------


; *** tranform the vector from gse to mfa

      bperp= SQRT(bx*bx + by*by)
      b0= SQRT(bx*bx + by*by + bz*bz)

; ^  first rotation

      sinphi=by/bperp
      cosphi=bx/bperp

      xp=  cosphi*xgse + sinphi*ygse
      yp= -sinphi*xgse + cosphi*ygse
      zp=  zgse

; ^  second rotation

      sintet= bperp/b0
      costet= bz/b0

      xmag= costet*xp - sintet*zp
      ymag= yp
      zmag= sintet*xp + costet*zp

; ^  third rotation

      sxm= costet*cosphi
      sym=-sinphi

      smperp= SQRT(sxm*sxm + sym*sym)

      sinpsi= sym/smperp
      cospsi= sxm/smperp

      xmfa=  cospsi*xmag + sinpsi*ymag
      ymfa= -sinpsi*xmag + cospsi*ymag
      zmfa=  zmag

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gse_to_sr2, xgse,ygse,zgse,rotx,roty,rotz, xsr2,ysr2,zsr2

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gse_to_sr2: GSE -> SR2  system
; *   Author : P. Robert, CETP, 2001
; *   Comment: local system, non time dependent

; *   Input  : xgse,ygse,zgse cartesian gse coordinates
;              rotx,roty,rotz cartesian gse coordinates of rotation axis
; *   Output : xsr2,ysr2,zsr2 cartesian sr2 coordinates
; ----------------------------------------------------------------------


; *** set transform matrix with Spin axis terms

      rmod= SQRT(rotx^2 + roty^2 + rotz^2)

      rx=rotx/rmod
      ry=roty/rmod
      rz=rotz/rmod

      a= 1./SQRT(ry*ry + rz*rz)

      x1= (ry*ry + rz*rz)*a
      x2= -rx*ry*a
      x3= -rx*rz*a

      y1=  0.
      y2=  rz*a
      y3= -ry*a

      z1= rx
      z2= ry
      z3= rz

; *** tranform the input vector from gse to sr2

      xsr2= x1*xgse + x2*ygse + x3*zgse
      ysr2= y1*xgse + y2*ygse + y3*zgse
      zsr2= z1*xgse + z2*ygse + z3*zgse

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gse_to_tpn, xgse,ygse,zgse,xo,yo,zo,xs,xtpn,ytpn,ztpn

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gse_to_tpn: GSE -> TPN  system
; *   Author : P. Robert, LPP , 2016
; *   Comment: local system, non time dependent

; *   Input  : xgse,ygse,zgse cartesian gse vector
;              xo,yo,zo position of the S/C in gse
;              xs subsolar point, submit of the paraboloid
;              from Earth to Sun
; *   Output : xtpn,ytpn,ztpn cartesian tpn coordinates
; ----------------------------------------------------------------------

;     real Tx,Ty,Tz, Px,Py,Pz, Nx,Ny,Nz

;     computation of the TPN axis in gse system

      cp_tpn_param, xo,yo,zo,xs, Tx,Ty,Tz, Px,Py,Pz, Nx,Ny,Nz

      xtpn= Tx*xgse + Ty*ygse +Tz*zgse
      ytpn= Px*xgse + Py*ygse +Pz*zgse
      ztpn= Nx*xgse + Ny*ygse +Nz*zgse

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gsm_to_gei, xgsm,ygsm,zgsm,xgei,ygei,zgei

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gsm_to_gei: GSM -> GEI  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgsm,ygsm,zgsm   cartesian gsm coordinates
; *   Output : xgei,ygei,zgei   cartesian gei coordinates
; ----------------------------------------------------------------------


      COMMON timp01,  gs1,gs2,gs3, gm1,gm2,gm3, ge1,ge2,ge3,gr1,gr2,gr3
      COMMON timp09,  yeigm1,yeigm2,yeigm3,  zeigm1,zeigm2,zeigm3

      xgei= gs1*xgsm + yeigm1*ygsm + zeigm1*zgsm
      ygei= gs2*xgsm + yeigm2*ygsm + zeigm2*zgsm
      zgei= gs3*xgsm + yeigm3*ygsm + zeigm3*zgsm

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gsm_to_geo, xgsm,ygsm,zgsm,xgeo,ygeo,zgeo

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gsm_to_geo: GSM -> GEO  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgsm,ygsm,zgsm   cartesian gsm coordinates
; *   Output : xgeo,ygeo,zgeo   cartesian geo coordinates
; ----------------------------------------------------------------------


      COMMON timp02,  ps1,ps2,ps3, gd1,gd2,gd3, pe1,pe2,pe3, pr1,pr2,pr3
      COMMON timp14,  yeogm1,yeogm2,yeogm3,  zeogm1,zeogm2,zeogm3

      xgeo= ps1*xgsm + yeogm1*ygsm + zeogm1*zgsm
      ygeo= ps2*xgsm + yeogm2*ygsm + zeogm2*zgsm
      zgeo= ps3*xgsm + yeogm3*ygsm + zeogm3*zgsm

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gsm_to_gse, xgsm,ygsm,zgsm,xgse,ygse,zgse

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gsm_to_gse: GSM -> GSE  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgsm,ygsm,zgsm   cartesian gsm coordinates
; *   Output : xgse,ygse,zgse   cartesian gse coordinates
; ----------------------------------------------------------------------


      COMMON timp11,  stetq,ctetq, sdze,cdze, smu,cmu, sphi,cphi

      xgse= xgsm
      ygse= cdze*ygsm - sdze*zgsm
      zgse= sdze*ygsm + cdze*zgsm

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gsm_to_gseq, xgsm,ygsm,zgsm,xgsq,ygsq,zgsq

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gsm_to_gsq: GSM -> GSQ  system
; *   Author : P. Robert, CRPE, 2002

; *   Input  : xgsm,ygsm,zgsm cartesian gsm coordinates
; *   Output : xgsq,ygsq,zgsq cartesian gsq coordinates
; ----------------------------------------------------------------------


      t_gsm_to_gse, xgsm,ygsm,zgsm,ax,ay,az
      t_gse_to_gseq, ax,ay,az,xgsq,ygsq,zgsq

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gsm_to_mag, xgsm,ygsm,zgsm,xmag,ymag,zmag

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gsm_to_mag: GSM -> MAG  system
; *   Author : P. Robert, CRPE, 2002

; *   Input  : xgsm,ygsm,zgsm cartesian gsm coordinates
; *   Output : xmag,ymag,zmag cartesian mag coordinates
; ----------------------------------------------------------------------


      t_gsm_to_sm, xgsm,ygsm,zgsm,ax,ay,az
      t_sm_to_mag, ax,ay,az,xmag,ymag,zmag

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gsm_to_sm, xgsm,ygsm,zgsm,xsma,ysma,zsma

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gsm_to_sma: GSM -> SM   system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgsm,ygsm,zgsm   cartesian gsm coordinates
; *   Output : xsma,ysma,zsma   cartesian sma coordinates
; ----------------------------------------------------------------------


      COMMON timp11,  stetq,ctetq, sdze,cdze, smu,cmu, sphi,cphi

      xsma= cmu*xgsm - smu*zgsm
      ysma= ygsm
      zsma= smu*xgsm + cmu*zgsm

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gsm_to_tpn, xgsm,ygsm,zgsm,xo,yo,zo,xs,xtpn,ytpn,ztpn

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gsm_to_tpn: GSM -> TPN  system
; *   Author : P. Robert, LPP , 2016
; *   Comment: local system, non time dependent

; *   Input  : xgsm,ygsm,zgsm cartesian gsm vector
;              xo,yo,zo position of the S/C in gsm
;              xs subsolar point, submit of the paraboloid
;              from Earth to Sun
; *   Output : xtpn,ytpn,ztpn cartesian tpn coordinates
; ----------------------------------------------------------------------

;     real Tx,Ty,Tz, Px,Py,Pz, Nx,Ny,Nz

;     computation of the TPN axis in gsm system

      cp_tpn_param, xo,yo,zo,xs, Tx,Ty,Tz, Px,Py,Pz, Nx,Ny,Nz

      xtpn= Tx*xgsm + Ty*ygsm +Tz*zgsm
      ytpn= Px*xgsm + Py*ygsm +Pz*zgsm
      ztpn= Nx*xgsm + Ny*ygsm +Nz*zgsm

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gseq_to_gei, xgsq,ygsq,zgsq,xgei,ygei,zgei

; ----------------------------------------------------------------------

; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gsq_to_gei: GSEQ-> GEI  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgsq,ygsq,zgsq   cartesian gsq coordinates
; *   Output : xgei,ygei,zgei   cartesian gei coordinates

; ----------------------------------------------------------------------


      COMMON timp01,  gs1,gs2,gs3, gm1,gm2,gm3, ge1,ge2,ge3,gr1,gr2,gr3
      COMMON timp10,  yeigq1,yeigq2,yeigq3,  zeigq1,zeigq2,zeigq3

      xgei= gs1*xgsq + yeigq1*ygsq + zeigq1*zgsq
      ygei= gs2*xgsq + yeigq2*ygsq + zeigq2*zgsq
      zgei= gs3*xgsq + yeigq3*ygsq + zeigq3*zgsq

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gseq_to_geo, xgsq,ygsq,zgsq,xgeo,ygeo,zgeo

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gsq_to_geo: GSEQ-> GEO  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgsq,ygsq,zgsq   cartesian gsq coordinates
; *   Output : xgeo,ygeo,zgeo   cartesian geo coordinates
; ----------------------------------------------------------------------


      COMMON timp02,  ps1,ps2,ps3, gd1,gd2,gd3, pe1,pe2,pe3, pr1,pr2,pr3
      COMMON timp15,  yeogq1,yeogq2,yeogq3,  zeogq1,zeogq2,zeogq3

      xgeo= ps1*xgsq + yeogq1*ygsq + zeogq1*zgsq
      ygeo= ps2*xgsq + yeogq2*ygsq + zeogq2*zgsq
      zgeo= ps3*xgsq + yeogq3*ygsq + zeogq3*zgsq

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gseq_to_gse, xgsq,ygsq,zgsq,xgse,ygse,zgse

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gsq_to_gse: GSEQ-> GSE  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xgsq,ygsq,zgsq   cartesian gsq coordinates
; *   Output : xgse,ygse,zgse   cartesian gse coordinates
; ----------------------------------------------------------------------


      COMMON timp11,  stetq,ctetq, sdze,cdze, smu,cmu, sphi,cphi

      xgse= xgsq
      ygse= ctetq*ygsq + stetq*zgsq
      zgse=-stetq*ygsq + ctetq*zgsq

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_gseq_to_gsm, xgsq,ygsq,zgsq,xgsm,ygsm,zgsm

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_gsq_to_gsm: GSQ -> GSM  system
; *   Author : P. Robert, CRPE, 2002

; *   Input  : xgsq,ygsq,zgsq cartesian gsq coordinates
; *   Output : xgsm,ygsm,zgsm cartesian gsm coordinates
; ----------------------------------------------------------------------


      t_gseq_to_gse, xgsq,ygsq,zgsq,xx,yy,zz
      t_gse_to_gsm, xx,yy,zz,xgsm,ygsm,zgsm

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_mag_to_gei, xmag,ymag,zmag,xgei,ygei,zgei

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_mag_to_gei: MAG -> GEI  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xmag,ymag,zmag cartesian mag coordinates
; *   Output : xgei,ygei,zgei cartesian gei coordinates
; ----------------------------------------------------------------------


      COMMON timp01,  gs1,gs2,gs3, gm1,gm2,gm3, ge1,ge2,ge3,gr1,gr2,gr3
      COMMON timp07,  xeima1,xeima2,xeima3,  yeima1,yeima2,yeima3

      xgei= xeima1*xmag + yeima1*ymag + gm1*zmag
      ygei= xeima2*xmag + yeima2*ymag + gm2*zmag
      zgei= xeima3*xmag + yeima3*ymag + gm3*zmag

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_mag_to_geo, xmag,ymag,zmag,xgeo,ygeo,zgeo

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_mag_to_geo: MAG -> GEO  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xmag,ymag,zmag   cartesian mag coordinates
; *   Output : xgeo,ygeo,zgeo   cartesian geo coordinates
; ----------------------------------------------------------------------


      COMMON timp02,  ps1,ps2,ps3, gd1,gd2,gd3, pe1,pe2,pe3, pr1,pr2,pr3
      COMMON timp12,  xeoma1,xeoma2,xeoma3,  yeoma1,yeoma2,yeoma3

      xgeo= xeoma1*xmag + yeoma1*ymag + gd1*zmag
      ygeo= xeoma2*xmag + yeoma2*ymag + gd2*zmag
      zgeo= xeoma3*xmag + yeoma3*ymag + gd3*zmag

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_mag_to_gsm, xmag,ymag,zmag,xgsm,ygsm,zgsm

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_mag_to_gsm: MAG -> GSM  system
; *   Author : P. Robert, CRPE, 2002

; *   Input  : xmag,ymag,zmag cartesian mag coordinates
; *   Output : xgsm,ygsm,zgsm cartesian gsm coordinates
; ----------------------------------------------------------------------


      t_mag_to_sm, xmag,ymag,zmag,xx,yy,zz
      t_sm_to_gsm, xx,yy,zz,xgsm,ygsm,zgsm

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_mag_to_sm, xmag,ymag,zmag,xsma,ysma,zsma

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_mag_to_sma: MAG -> SM   system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xmag,ymag,zmag   cartesian mag coordinates
; *   Output : xsma,ysma,zsma   cartesian sma coordinates
; ----------------------------------------------------------------------


      COMMON timp11,  stetq,ctetq, sdze,cdze, smu,cmu, sphi,cphi

      xsma= cphi*xmag - sphi*ymag
      ysma= sphi*xmag + cphi*ymag
      zsma= zmag

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_sm_to_gei, xsma,ysma,zsma,xgei,ygei,zgei

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_sma_to_gei: SM  -> GEI  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xsma,ysma,zsma   cartesian sma coordinates
; *   Output : xgei,ygei,zgei   cartesian gei coordinates
; ----------------------------------------------------------------------


      COMMON timp01,  gs1,gs2,gs3, gm1,gm2,gm3, ge1,ge2,ge3,gr1,gr2,gr3
      COMMON timp08,  xeism1,xeism2,xeism3,  yeism1,yeism2,yeism3

      xgei=xeism1*xsma + yeism1*ysma + gm1*zsma
      ygei=xeism2*xsma + yeism2*ysma + gm2*zsma
      zgei=xeism3*xsma + yeism3*ysma + gm3*zsma

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_sm_to_geo, xsma,ysma,zsma,xgeo,ygeo,zgeo

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_sma_to_geo: SM  -> GEO  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xsma,ysma,zsma   cartesian sma coordinates
; *   Output : xgeo,ygeo,zgeo   cartesian geo coordinates
; ----------------------------------------------------------------------


      COMMON timp02,  ps1,ps2,ps3, gd1,gd2,gd3, pe1,pe2,pe3, pr1,pr2,pr3
      COMMON timp13,  xeosm1,xeosm2,xeosm3,  yeosm1,yeosm2,yeosm3

      xgeo= xeosm1*xsma + yeosm1*ysma + gd1*zsma
      ygeo= xeosm2*xsma + yeosm2*ysma + gd2*zsma
      zgeo= xeosm3*xsma + yeosm3*ysma + gd3*zsma

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_sm_to_gsm, xsma,ysma,zsma,xgsm,ygsm,zgsm

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_sma_to_gsm: SM  -> GSM  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xsma,ysma,zsma   cartesian sma coordinates
; *   Output : xgsm,ygsm,zgsm   cartesian gsm coordinates
; ----------------------------------------------------------------------


      COMMON timp11,  stetq,ctetq, sdze,cdze, smu,cmu, sphi,cphi

      xgsm=  cmu*xsma + smu*zsma
      ygsm=  ysma
      zgsm= -smu*xsma + cmu*zsma

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_sm_to_mag, xsma,ysma,zsma,xmag,ymag,zmag

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_sma_to_mag: SM  -> MAG  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: terms of transformation matrix are given in common

; *   Input  : xsma,ysma,zsma   cartesian sma coordinates
; *   Output : xmag,ymag,zmag   cartesian mag coordinates
; ----------------------------------------------------------------------


      COMMON timp11,  stetq,ctetq, sdze,cdze, smu,cmu, sphi,cphi

      xmag=  cphi*xsma + sphi*ysma
      ymag= -sphi*xsma + cphi*ysma
      zmag=  zsma

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_sph_to_car, r,teta,phi,x,y,z

; ----------------------------------------------------------------------

; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_sph_to_car: SPH -> CAR  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: none

; *   Input  :   r,teta,phi  spherical coordinates (angles in radians)
; *   Output :   x,y,z       cartesian coordinates
; ----------------------------------------------------------------------


      sq=r*SIN(teta)
      x=sq*COS(phi)
      y=sq*SIN(phi)
      z=r*COS(teta)

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_sr2_to_gse, xsr2,ysr2,zsr2,rotx,roty,rotz, xgse,ygse,zgse

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_sr2_to_gse: SR2 -> GSE  system
; *   Author : P. Robert, CETP, 2001
; *   Comment: local system, non time dependent

; *   Input  : xsr2,ysr2,sr2z cartesian sr2 coordinates
;              rotx,roty,rotz cartesian gse coordinates of rotation axis
; *   Output : xgse,ygse,zgse cartesian gse coordinates
; ----------------------------------------------------------------------

; *** set transform matrix with Spin axis terms

      rmod= SQRT(rotx^2 + roty^2 + rotz^2)

      rx=rotx/rmod
      ry=roty/rmod
      rz=rotz/rmod

      a= 1./SQRT(ry*ry + rz*rz)

      x1= (ry*ry + rz*rz)*a
      x2= -rx*ry*a
      x3= -rx*rz*a

      y1=  0.
      y2=  rz*a
      y3= -ry*a

      z1= rx
      z2= ry
      z3= rz

; *** tranform the input vector from sr2 to gse

      xgse= x1*xsr2 + y1*ysr2 + z1*zsr2
      ygse= x2*xsr2 + y2*ysr2 + z2*zsr2
      zgse= x3*xsr2 + y3*ysr2 + z3*zsr2

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_sr2_to_mfa, xsr2,ysr2,zsr2,bx,by,bz,rox,roy,roz,  xm,ym,zm

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_sr2_to_mfa: SR2 -> MFA  system
; *   Author : P. Robert, CETP, 2001
; *   Comment: local system, non time dependent

; *   Input  : xsr2,ysr2,sr2z cartesian sr2 coordinates
;              bx,  by,  bz   cartesian sr2 coordinates of DC mag field
;              rox, roy, roz  cartesian gse coordinates of rotation axis
; *   Output : xm,  ym,  zm   cartesian mfa coordinates
; ----------------------------------------------------------------------


; *** normalise R supposed undependant of time

      rmod= SQRT(rox^2 + roy^2 + roz^2)

      rx=rox/rmod
      ry=roy/rmod
      rz=roz/rmod

; *** tranform the vector from sr2 to mfa

      bperp= SQRT(bx*bx + by*by)
      b0= SQRT(bx*bx + by*by + bz*bz)

; ^  first rotation

      sinphi=by/bperp
      cosphi=bx/bperp

      xp=  cosphi*xsr2 + sinphi*ysr2
      yp= -sinphi*xsr2 + cosphi*ysr2
      zp=  zsr2

; ^  second rotation

      sintet= bperp/b0
      costet= bz/b0

      xmag= costet*xp - sintet*zp
      ymag= yp
      zmag= sintet*xp + costet*zp

; ^  third rotation

      b= SQRT(ry*ry + rz*rz)

      sxm=  b*costet*cosphi - rx*sintet
      sym= -b*sinphi

      smperp= SQRT(sxm*sxm + sym*sym)

      sinpsi= sym/smperp
      cospsi= sxm/smperp

      xm=  cospsi*xmag + sinpsi*ymag
      ym= -sinpsi*xmag + cospsi*ymag
      zm=  zmag

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_sr2_to_sr, xsr2,ysr2,spifre,spipha,deltaT,xsre,ysre

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_sr2_to_sre: SR2 -> SRef system
; *   Author : P. Robert, CRPE, 2001
; *   Comment: Z component is unchanged (spin axis)

; *   Input  : xsr2, ysr2 cartesian sr2 coordinates
;              spifre     spin frequency in Hz
;              spipha     spin phase in radians, growing with time
;                         spipha= positive angle between the xsr axis
;                         and the component of the direction of the Sun
;                         in the xsr-ysr plane.
;              deltaT     (T -To) time (sec.), between the current time
;                         and the time where is measured the spin phase.
; *   Output : xsre,ysre  cartesian sr coordinates
; ----------------------------------------------------------------------


      pi2= 2.*3.1415927

      depift= Fmod(pi2*spifre*deltaT,pi2)
      phicr= spipha -depift
      sinphi=SIN(phicr)
      cosphi=COS(phicr)

      xsre=  cosphi*xsr2 -sinphi*ysr2
      ysre=  sinphi*xsr2 +cosphi*ysr2

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_sr_to_sr2, xsre,ysre,spifre,spipha,deltaT,xsr2,ysr2

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_sre_to_sr2: SRef-> SR2 system
; *   Author : P. Robert, CRPE, 2001
; *   Comment: Z component is unchanged (spin axis)

; *   Input  : xsre, ysre cartesian sr coordinates
;              spifre     spin frequency in Hz
;              spipha     spin phase in radians, growing with time
;                         spipha= positive angle between the xsr axis
;                         and the component of the direction of the Sun
;                         in the xsr-ysr plane.
;              deltaT     (T -To) time (sec.), between the current time
;                         and the time where is measured the spin phase.
; *   Output : xsr2,ysr2  cartesian sr coordinates
; ----------------------------------------------------------------------


      pi2= 2.*3.1415927

      depift= Fmod(pi2*spifre*deltaT,pi2)
      phicr= spipha -depift
      sinphi=SIN(phicr)
      cosphi=COS(phicr)

      xsr2=  cosphi*xsre +sinphi*ysre
      ysr2= -sinphi*xsre +cosphi*ysre

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_vdh_to_geo, xvdh,yvdh,zvdh,rlat,rlong,xgeo,ygeo,zgeo

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_vdh_to_geo: VDH -> GEO  system
; *   Author : P. Robert, CRPE, 1992
; *   Comment: local system, non time dependent

; *   Input  : xvdh,yvdh,zvdh   cartesian vdh coordinates
;              rlat,rlong       latitude and longitude of the point
;                               of observation (radians)
; *   Output : xgeo,ygeo,zgeo   cartesian geo coordinates
; ----------------------------------------------------------------------


      q=COS(rlat)
      r1=q*COS(rlong)
      r2=q*SIN(rlong)
      r3=SIN(rlat)

      v1=r1
      v2=r2
      v3=r3

      q12=SQRT(r1^2 + r2^2)

      d1= -r2/q12
      d2=  r1/q12
      d3=  0.

      h1= -r1*r3/q12
      h2= -r2*r3/q12
      h3=  q12

      xgeo= v1*xvdh + d1*yvdh + h1*zvdh
      ygeo= v2*xvdh + d2*yvdh + h2*zvdh
      zgeo= v3*xvdh + d3*yvdh + h3*zvdh

;     RETURN
      END
;      
;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_xyz_to_vdh, x,y,z,a1,a2,a3,v,d,h

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_xyz_to_vdh: xyz spinning -> VDH
; *   Author : P. Robert, SDev, 2020
; *   Comment: use Euler angles in degrees; VDH is the fixed system.

; *   Input : x,y,z cartesian xyz coordinates
; *   Output: v,d,h cartesian VDH coordinates
; ----------------------------------------------------------------------

      pisd=3.1415926/180.

      a1r=a1*pisd
      a2r=a2*pisd
      a3r=a3*pisd

      c1=COS(a1r)
      c2=COS(a2r)
      c3=COS(a3r)

      s1=SIN(a1r)
      s2=SIN(a2r)
      s3=SIN(a3r)

      v=( c1*c3 -s1*c2*s3)*x  +(-s1*c3 -c1*c2*s3)*y  +( s2*s3)*z
      d=( c1*s3 +s1*c2*c3)*x  +(-s1*s3 +c1*c2*c3)*y  +(-s2*c3)*z
      h=           (s1*s2)*x  +           (c1*s2)*y  +(    c2)*z

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO t_vdh_to_xyz, v,d,h,a1,a2,a3,x,y,z

; ----------------------------------------------------------------------
; *   Class  : transform modules of Rocotlib Software
; *   Object : transforms_vdh_to_xyz: VDH -> xyz spinning 
; *   Author : P. Robert, SDev, 2020
; *   Comment: use Euler angles in degrees; VDH is the fixed system.

; *   Input  : v,d,h cartesian VDH coordinates
; *   Output : x,y,z cartesian xyz coordinates
; ----------------------------------------------------------------------

      pisd=3.1415926/180.

      a1r=a1*pisd
      a2r=a2*pisd
      a3r=a3*pisd

      c1=COS(a1r)
      c2=COS(a2r)
      c3=COS(a3r)

      s1=SIN(a1r)
      s2=SIN(a2r)
      s3=SIN(a3r)

      x=( c1*c3 -s1*c2*s3)*v  +( c1*s3 +s1*c2*c3)*d +( s1*s2)*h
      y=(-s1*c3 -c1*c2*s3)*v  +(-s1*s3 +c1*c2*c3)*d +( c1*s2)*h
      z=          ( s2*s3)*v  +          (-s2*c3)*d +(    c2)*h

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

;     Supplements FOR matrix operations.
;     Allows the transformation of an input 3D signal into the
;     Minimum variance coordinate (MVA Analysis).

;     All codes in this part are extracted of Roproc Software, V 4.5
;     and rewrited properly in f77, with Rocotlib V2.2 conventions.
;     P. Robert, LPP , April 2016.

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

      PRO mat_cp_varmin, ifc,Vx,Vy,Vz,n,irep,covar,eigval,eigvec

;     real Vx(n),Vy(n),Vz(n)
;     real covar(3,3),eigval(3),eigvec(3,3)

;     ---------------------------------------------------------------+--
; *   Class  : matrix operation of Rocotlib Software
; *   Object : compute variance minimum coordinate of a signal Vx,Vy,Vz
; *   Author : P. Robert, CETP, 2001, rev. PR 2016
;     Comment: check orthogonality of the MVA matrix

; *   Input  : ifc (unit FOR writing results)
;              Vx(n),Vy(n),Vz(n) input signal
;              n number of point of the signal
;              irep=1 : passe input vector in MVA coordinates

; *   Output : covar covariance matrix
;              lamda(3), eigvec(3,3) eigen values and eigen vectors
;              results are also writted on unit ifc
;     ---------------------------------------------------------------+--

      fmt100="(1x,a)"

      IF(ifc EQ 6) THEN BEGIN
        PRINT
        PRINT, FORMAT=fmt100,'------------------------------------------------'
        PRINT, FORMAT=fmt100,'Compute Minimum Variance Analaysis coord. system'
        PRINT, FORMAT=fmt100,'------------------------------------------------'
                   ENDIF ELSE BEGIN
        PRINTF, ifc
        PRINTF, ifc, FORMAT=fmt100,'------------------------------------------------'
        PRINTF, ifc, FORMAT=fmt100,'Compute Minimum Variance Analaysis coord. system'
        PRINTF, ifc, FORMAT=fmt100,'------------------------------------------------'
      ENDELSE

; *** compute covariance matrix

      mat_cp_covariance, Vx,Vy,Vz,n,covar
      mat_write, ifc,'covariance matrix of input signal:',covar

; *** diagonalisation of covariance matrix

      mat_diagonalise, covar,eigval,eigvec
      mat_write, ifc,'covariance matrix diagonalized:',covar
      mat_write, ifc,'eigen vectors matrix:',eigvec

; *** check orthogonality and direct sens of the 3 vectors

      mat_check_ortho, ifc,eigvec

; *** PRINT eigen vectors and eigen values

      mat_write_eigen_vec, ifc,eigval,eigvec

; *** transform input signal in MVA coordinates, ie. eigen vectors

      IF(irep NE 0) THEN BEGIN
        mat_change_coord, eigvec,Vx,Vy,Vz,n
      IF(ifc EQ 6) THEN BEGIN
        PRINT
        PRINT, FORMAT=fmt100,'input signal is passed into MVA coordinates'
                   ENDIF ELSE BEGIN
        PRINTF, ifc
        PRINTF, ifc, FORMAT=fmt100,'input signal is passed into MVA coordinates'
      ENDELSE
      ENDIF

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

      PRO mat_cp_covariance, Vx,Vy,Vz,n,covar

;     ---------------------------------------------------------------+--
; *   Class  : matrix operation of Rocotlib Software
; *   Object : compute covariance matrix FOR a vector series V(n)
; *   Author : P. Robert, CETP, 2001, rev. PR 2016

; *   Input  : Vx(n),Vy(n),Vz(n) vector series
; *   Output : covar(3,3)
;     ---------------------------------------------------------------+--

;     double precision sx, sy, sz, sx2, sy2, sz2, sxy, syz, szx

; *   set summation to zero

      sx=0.d0
      sy=0.d0
      sz=0.d0

      sx2=0.d0
      sy2=0.d0
      sz2=0.d0

      sxy=0.d0
      syz=0.d0
      szx=0.d0

; *   components summation and square components summation

      FOR k=0,n-1 DO BEGIN

        sx= sx +DOUBLE(Vx(k))
        sy= sy +DOUBLE(Vy(k))
        sz= sz +DOUBLE(Vz(k))

        sx2= sx2 +DOUBLE(Vx(k)^2)
        sy2= sy2 +DOUBLE(Vy(k)^2)
        sz2= sz2 +DOUBLE(Vz(k)^2)

        sxy= sxy +DOUBLE(Vx(k)*Vy(k))
        syz= syz +DOUBLE(Vy(k)*Vz(k))
        szx= szx +DOUBLE(Vz(k)*Vx(k))

      ENDFOR

; *   average values

      sx = sx/DOUBLE(n)
      sy = sy/DOUBLE(n)
      sz = sz/DOUBLE(n)

      sx2= sx2/DOUBLE(n)
      sy2= sy2/DOUBLE(n)
      sz2= sz2/DOUBLE(n)

      sxy= sxy/DOUBLE(n)
      syz= syz/DOUBLE(n)
      szx= szx/DOUBLE(n)

; *   variance set to diagonals terms

      covar(0,0)= FLOAT(sx2-sx^2)
      covar(1,1)= FLOAT(sy2-sy^2)
      covar(2,2)= FLOAT(sz2-sz^2)

; *   covariances set to semi \diagonal terms

      covar(1,0)= FLOAT(sxy-sx*sy)
      covar(2,1)= FLOAT(syz-sy*sz)
      covar(2,0)= FLOAT(szx-sz*sx)

; *   second half triangle

      covar(0,1)= covar(1,0)
      covar(0,2)= covar(2,0)
      covar(1,2)= covar(2,1)

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

      PRO mat_diagonalise, mat,eigval,eigvec

;     real mat(3,3),eigval(3),eigvec(3,3)

;     ---------------------------------------------------------------+--
; *   Class  : matrix operation of Rocotlib Software
; *   Object : diagonalise the given matrix mat(3,3)
; *   Author : P. Robert, CETP, 2001, rev. PR 2016

; *   Input  : real mat(3,3)
; *   Output : eigval(3)   eigen values
;              eigvec(3,3) eigen vectors normalised to 1.
;     ---------------------------------------------------------------+--

;     compute eigen vectors and eigen values of real mat(3,3)

      mat_cp_eigen_vec, mat,eigval,eigvec

;     normalise to 1 eigen vectors

      mat_normalize_vec, eigvec

; *   set positive direction FOR \diag

      IF(eigvec(2,0) LT 0.) THEN BEGIN
         eigvec(0,0)= -eigvec(0,0)
         eigvec(1,0)= -eigvec(1,0)
         eigvec(2,0)= -eigvec(2,0)
      ENDIF

      IF(eigvec(0,2) LT 0.) THEN BEGIN
         eigvec(0,2)= -eigvec(0,2)
         eigvec(1,2)= -eigvec(1,2)
         eigvec(2,2)= -eigvec(2,2)
      ENDIF

; *   sort eigen vector with eigen value (z=min)

      FOR k=1,4 DO BEGIN
         i=k
         IF(k GT 2) THEN i=k-2

         IF(eigval(i-1) LT eigval(i)) THEN BEGIN
            dint=eigval(i-1)
            eigval(i-1)=eigval(i)
            eigval(i)=dint
            FOR j=1,3 DO BEGIN
               dint=eigvec(j-1,i-1)
               eigvec(j-1,i-1)=eigvec(j-1,i)
               eigvec(j-1,i)=dint
            ENDFOR
         ENDIF
      ENDFOR

; *   determinant computation:

      mat_cp_determin, eigvec,det

      IF(det LT 0.) THEN BEGIN
         eigvec(0,1)= -eigvec(0,1)
         eigvec(1,1)= -eigvec(1,1)
         eigvec(2,1)= -eigvec(2,1)
      ENDIF

; *   modif P.R. 2004: the vector corresponding to eigval min has z >0

      IF(eigvec(2,2) LT 0.) THEN BEGIN
         eigvec(0,1)= -eigvec(0,1)
         eigvec(1,1)= -eigvec(1,1)
         eigvec(2,1)= -eigvec(2,1)

         eigvec(0,2)= -eigvec(0,2)
         eigvec(1,2)= -eigvec(1,2)
         eigvec(2,2)= -eigvec(2,2)
      ENDIF

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

      PRO mat_check_ortho, ifc,mat

;     ---------------------------------------------------------------+--
; *   Class  : matrix operation of Rocotlib Software
; *   Object : check orthogonality of matrix components
; *   Author : P. Robert, CETP, 2001, rev. PR 2016

; *   Input  : ifc (unit FOR writing results), real mat(3,3)
; *   Output : writing result on unit ifc
;     ---------------------------------------------------------------+--

;     real mat(3,3)

      v1x=mat(0,0)
      v1y=mat(1,0)
      v1z=mat(2,0)

      v2x=mat(0,1)
      v2y=mat(1,1)
      v2z=mat(2,1)

      v3x=mat(0,2)
      v3y=mat(1,2)
      v3z=mat(2,2)

;     dot product
  
      v1v2= v1x*v2x + v1y*v2y + v1z*v2z
      v2v3= v2x*v3x + v2y*v3y + v2z*v3z
      v3v1= v3x*v1x + v3y*v1y + v3z*v1z

;     cross product

; *   v1 X v2

      v1cv2x= v1y*v2z - v1z*v2y
      v1cv2y= v1z*v2x - v1x*v2z
      v1cv2z= v1x*v2y - v1y*v2x

; *   v2 X v3

      v2cv3x= v2y*v3z - v2z*v3y
      v2cv3y= v2z*v3x - v2x*v3z
      v2cv3z= v2x*v3y - v2y*v3x

; *   v3 X v1

      v3cv1x= v3y*v1z - v3z*v1y
      v3cv1y= v3z*v1x - v3x*v1z
      v3cv1z= v3x*v1y - v3y*v1x

      fmt100="(3(1x,a,e14.5))"
      fmt200="(1x,a,3(e16.5))"

      IF(ifc EQ 6) THEN BEGIN

         PRINT
         PRINT, FORMAT=fmt100,'Check orthogonality of given matrix:'
         PRINT, FORMAT=fmt100,'----------------------------------- '
         PRINT

         PRINT, FORMAT=fmt100,'1) dot product of any two rows or any two colums ' $
                                                  +'must be equal to zero:'
         PRINT
         PRINT, FORMAT=fmt100,'V1.V2 = ',v1v2
         PRINT, FORMAT=fmt100,'V2.V3 = ',v2v3
         PRINT, FORMAT=fmt100,'V3.V1 = ',v3v1

         PRINT
         PRINT, FORMAT=fmt100,'2) cross product of any two rows or colums '+ $
                                                      'must be equal to the third '
         PRINT, FORMAT=fmt100,'   row or column or its negative):'

         PRINT
         PRINT, FORMAT=fmt200,'V1XV2 = ',v1cv2x,v1cv2y,v1cv2z
         PRINT, FORMAT=fmt200,'   V3 = ',v3x,v3y,v3z

         PRINT
         PRINT, FORMAT=fmt200,'V2XV3 = ',v2cv3x,v2cv3y,v2cv3z
         PRINT, FORMAT=fmt200,'   V1 = ',v1x,v1y,v1z

         PRINT
         PRINT, FORMAT=fmt200,'V3XV1 = ',v3cv1x,v3cv1y,v3cv1z
         PRINT, FORMAT=fmt200,'   V2 = ',v2x,v2y,v2z

                  ENDIF ELSE BEGIN

         PRINTF, ifc
         PRINTF, ifc, FORMAT=fmt100,'Check orthogonality of given matrix:'
         PRINTF, ifc, FORMAT=fmt100,'----------------------------------- '
         PRINTF, ifc

         PRINTF, ifc, FORMAT=fmt100,'1) dot product of any two rows or any two colums ' $
                                                  +'must be equal to zero:'
         PRINTF, ifc
         PRINTF, ifc, FORMAT=fmt100,'V1.V2 =',v1v2
         PRINTF, ifc, FORMAT=fmt100,'V2.V3 =',v2v3
         PRINTF, ifc, FORMAT=fmt100,'V3.V1 =',v3v1

         PRINTF, ifc
         PRINTF, ifc, FORMAT=fmt100,'2) cross product of any two rows or colums '+ $
                                                      'must be equal to the third '
         PRINTF, ifc, FORMAT=fmt100,'   row or column or its negative):'

         PRINTF, ifc
         PRINTF, ifc, FORMAT=fmt200,'V1XV2 =',v1cv2x,v1cv2y,v1cv2z
         PRINTF, ifc, FORMAT=fmt200,'   V3 =',v3x,v3y,v3z

         PRINTF, ifc
         PRINTF, ifc, FORMAT=fmt200,'V2XV3 =',v2cv3x,v2cv3y,v2cv3z
         PRINTF, ifc, FORMAT=fmt200,'   V1 =',v1x,v1y,v1z

         PRINTF, ifc
         PRINTF, ifc, FORMAT=fmt200,'V3XV1 =',v3cv1x,v3cv1y,v3cv1z
         PRINTF, ifc, FORMAT=fmt200,'   V2 =',v2x,v2y,v2z

      ENDELSE

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

      PRO mat_cp_determin, mat,det

;     ---------------------------------------------------------------+--
; *   Class  : matrix operation of Rocotlib Software
; *   Object : compute determinant of the given matrix
; *   Author : P. Robert, CETP, 2001, rev. PR 2016

; *   Input  : real mat(3,3)
; *   Output : det
;     ---------------------------------------------------------------+--

;     real mat(3,3)

      det=    mat(0,0)*mat(1,1)*mat(2,2)+mat(0,1)*mat(1,2)*mat(2,0)
      det=det+mat(1,0)*mat(2,1)*mat(0,2)-mat(2,0)*mat(1,1)*mat(0,2)
      det=det-mat(0,0)*mat(2,1)*mat(1,2)-mat(1,0)*mat(0,1)*mat(2,2)

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

      PRO mat_cp_eigen_vec, mat,eigval,eigvec

;     ---------------------------------------------------------------+--
; *   Class  : matrix operation of Rocotlib Software
; *   Object : compute eigen vectors and eigen values of real mat(3,3)
; *   Author : unknown, CETP, 2001, rev. PR 2016
; *   Comment: mat(3,3) must be real and symmetrical;
;              Method used is Householder.

; *   Input  : real mat(3,3)
; *   Output : eigval(3)   eigen values
;              eigvec(3,3) eigen vectors
;     ---------------------------------------------------------------+--

      rr=FLTARR(3,3)
      e =FLTARR(3)

; *   set eigen vectors to unity matrix

      FOR i=0,2 DO BEGIN
         FOR j=0,2 DO BEGIN
            eigvec(i,j)=0.0
         ENDFOR
         eigvec(i,i)=1.0
      ENDFOR

;     set eigen values

      eigval(0)=mat(0,2)
      eigval(1)=mat(1,2)

      signe=1.0
      IF(eigval(1) LT 0.0) THEN signe=-1.0
      eigval(1)=eigval(1)+signe*SQRT(eigval(0)^2+eigval(1)^2)
      eigval(2)=0.0

      h=(eigval(0)^2+eigval(1)^2)/2.0

      epsi=1.e-30
      
      IF(ABS(h) GT epsi) THEN BEGIN

;        determination of the Householder matrix (results in  eigvec).

         FOR i=0,2 DO BEGIN
            FOR j=0,2 DO BEGIN
               eigvec(i,j)=eigvec(i,j)-(eigval(i)*eigval(j))/h
            ENDFOR
         ENDFOR

;        tridiagonalisation of mat matrix

         mat_product, eigvec,mat,rr
         mat_product, rr,eigvec,mat

      ENDIF

;     set eigen values to diagonal terms

      FOR i=0,2 DO BEGIN
         eigval(i)=mat(i,i)
      ENDFOR

      FOR i=0,1 DO BEGIN
         e(i)=mat(i+1,i)
      ENDFOR
      e(2)=0.0

; *   search FOR eigen values (max iteration = 30)

      FOR l=1,3 DO BEGIN
         iter=0
  ww1:
      FOR m=l,2 DO BEGIN
            dd=ABS(eigval(m-1))+ABS(eigval(m))
; PR        IF(ABS(e(m-1))+dd EQ dd) THEN GOTO, ww2
            IF(ABS(e(m-1)) LE ABS(dd)*1E-6) THEN GOTO, ww2
      ENDFOR
      m=3
  ww2:
      IF(m NE l) THEN BEGIN

            IF(iter EQ 30) THEN BEGIN
               MESSAGE, /NONAME, /NOPREFIX, '*** Rocotlib/mat_cp_eigen_vec: ABORTED ! '+ $
                                                'TOO MANY ITERATIONS'
            ENDIF

            iter=iter+1
            g=(eigval(l)-eigval(l-1))/(2.0*e(l-1))
            mat_cp_pythag_func, g,1.0,r
            g=eigval(m-1)-eigval(l-1)+e(l-1)/(g+Tsign(r,g))
            s=1.0
            c=1.0
            p=0.0

            FOR i=m-1,l,-1 DO BEGIN
               f=s*e(i-1)
               b=c*e(i-1)
               mat_cp_pythag_func, f,g,r
               e(i)=r

               IF(ABS(r) LT epsi) THEN BEGIN
                  eigval(i)=eigval(i)-p
                  e(m)=0.0
                  GOTO, ww1
               ENDIF

               s=f/r
               c=g/r
               g=eigval(i)-p
               r=(eigval(i-1)-g)*s+2.0*c*b
               p=s*r
               eigval(i)=g+p
               g=c*r-b

               FOR k=1,3 DO BEGIN
                  f=eigvec(k-1,i)
                  eigvec(k-1,i)=s*eigvec(k-1,i-1)+c*f
                  eigvec(k-1,i-1)=c*eigvec(k-1,i-1)-s*f
               ENDFOR

            ENDFOR

            eigval(l-1)=eigval(l-1)-p
            e(l-1)=g
            e(m-1)=0.0
            GOTO, ww1

         ENDIF
      ENDFOR

; *   modif PR  sep 11 2001 to load  mat with the result of the diag.

      FOR i=0,2 DO BEGIN
         mat(i,i)=eigval(i)
      ENDFOR

      FOR i=0,1 DO BEGIN
         mat(i+1,i)=e(i)
         mat(i,i+1)=e(i)
      ENDFOR

      mat(0,2)=e(2)
      mat(2,0)=e(2)

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

      PRO mat_cp_pythag_func, a,b,fpyth

;     ---------------------------------------------------------------+--
; *   Class  : matrix operation of Rocotlib Software
; *   Object : Pythagore FUNCTION of two real (used by mat_cp_eigen_vec)
; *   Author : unknown, CETP, 2001, rev. PR 2016

; *   Input  : a,b
; *   Output : fpyth
;     ---------------------------------------------------------------+--

      absa=ABS(a)
      absb=ABS(b)

      epsi=1.e-37

      IF(absa GT absb) THEN BEGIN
             fpyth=absa*SQRT(1.0+(absb/absa)^2)
                         ENDIF ELSE BEGIN
         IF(absb LT epsi) THEN BEGIN
             fpyth=0.0
                          ENDIF ELSE BEGIN
             fpyth=absb*SQRT(1.0+(absa/absb)^2)
         ENDELSE
      ENDELSE

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

      PRO mat_normalize_vec, mat

;     ---------------------------------------------------------------+--
; *   Class  : matrix operation of Rocotlib Software
; *   Object : normalize to 1. the vectors of the input matrix
; *   Author : P. Robert, CETP, 2001, rev. PR 2016

; *   Input  : real mat(3,3)
; *   Output : real mat(3,3) with vectors normalized to 1.
;     ---------------------------------------------------------------+--

;     real mat(3,3),matmod

      epsi=1.e-37

      FOR j=0,2 DO BEGIN
         matmod=0.

         FOR i=0,2 DO BEGIN
           matmod=matmod+mat(i,j)*mat(i,j)
         ENDFOR

         FOR i=0,2 DO BEGIN
            IF(ABS(matmod) GT epsi) THEN BEGIN
               mat(i,j)=mat(i,j)/SQRT(matmod)
            ENDIF ELSE BEGIN
                PRINT, '*** Rocotlib/mat_normalize_vec: module is = 0' 
            ENDELSE
         ENDFOR
      ENDFOR

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

      PRO mat_product, mat1,mat2,mat3

;     ---------------------------------------------------------------+--
; *   Class  : matrix operation of Rocotlib Software
; *   Object : matrix product of two given matrix of dim. 3
; *   Author : P. Robert, CETP, 2001, rev. PR 2016

; *   Input  : n, mat1(3,3), mat2(3,3)
; *   Output : mat3(3,3)
;     ---------------------------------------------------------------+--

;     real mat1(3,3),mat2(3,3),mat3(3,3)

      FOR i=0,2 DO BEGIN
         FOR j=0,2 DO BEGIN
            mat3(i,j)=0.0
            FOR k=0,2 DO BEGIN
               mat3(i,j)=mat3(i,j)+mat1(i,k)*mat2(k,j)
            ENDFOR
         ENDFOR
      ENDFOR

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

      PRO mat_somme, mat1,mat2,mat3

;     ---------------------------------------------------------------+--
; *   Class  : matrix operation of Rocotlib Software
; *   Object : matrix somme of two given matrix of dim. 3
; *   Author : P. Robert, LPP , 2016

; *   Input  : n, mat1(3,3), mat2(3,3)
; *   Output : mat3(3,3)
;     ---------------------------------------------------------------+--

;     real mat1(3,3),mat2(3,3),mat3(3,3)

      FOR i=0,2 DO BEGIN
         FOR j=0,2 DO BEGIN
            mat3(i,j)=mat1(i,j)+mat2(i,j)
         ENDFOR
      ENDFOR

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

      PRO mat_diff, mat1,mat2,mat3

;     ---------------------------------------------------------------+--
; *   Class  : matrix operation of Rocotlib Software
; *   Object : matrix difference of two given matrix 
; *   Author : P. Robert, LPP , 2016

; *   Input  : n, mat1(3,3), mat2(3,3)
; *   Output : mat3(3,3)
;     ---------------------------------------------------------------+--

;     real mat1(3,3),mat2(3,3),mat3(3,3)

      FOR i=0,2 DO BEGIN
         FOR j=0,2 DO BEGIN
            mat3(i,j)=mat1(i,j)-mat2(i,j)
         ENDFOR
      ENDFOR

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

      PRO mat_transpose, mat

;     ---------------------------------------------------------------+--
; *   Class  : matrix operation of Rocotlib Software
; *   Object : transpose input matrix
; *   Author : P. Robert, CETP, 2001, rev. PR 2016

; *   Input  : real mat(3,3)
; *   Output : real mat(3,3) transposed
;     ---------------------------------------------------------------+--

;     real mat(3,3)

      r12= mat(0,1)
      r13= mat(0,2)
      r23= mat(1,2)

      mat(0,1)=mat(1,0)
      mat(0,2)=mat(2,0)
      mat(1,2)=mat(2,1)

      mat(1,0)=r12
      mat(2,0)=r13
      mat(2,1)=r23

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

      PRO mat_change_coord, mat,Vx,Vy,Vz,n

;     ---------------------------------------------------------------+--
; *   Class  : matrix operation of Rocotlib Software
; *   Object : change coordinate of a vector serie with a given matrix
; *   Author : P. Robert, CETP, 2001, rev. PR 2016
; *   Comment: compute V(n)= mat*V(n)

; *   Input  : ifc (unit FOR writing results), real mat(3,3)
; *   Output : writing result on unit ifc
;     ---------------------------------------------------------------+--

;     real Vx(n),Vy(n),Vz(n)
;     real mat(3,3)

      FOR i=0,n-1 DO BEGIN

         Vxr= mat(0,0)*Vx(i) +mat(1,0)*Vy(i) +mat(2,0)*Vz(i)
         Vyr= mat(0,1)*Vx(i) +mat(1,1)*Vy(i) +mat(2,1)*Vz(i)
         Vzr= mat(0,2)*Vx(i) +mat(1,2)*Vy(i) +mat(2,2)*Vz(i)

         Vx(i)=Vxr
         Vy(i)=Vyr
         Vz(i)=Vzr

      ENDFOR

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

      PRO mat_write, ifc,com,mat

;     ---------------------------------------------------------------+--
; *   Class  : matrix operation of Rocotlib Software
; *   Object : PRINT on ifc unit mat(3,3) with a comment
; *   Author : P. Robert, CETP, 2001, rev. PR 2016

; *   Input  : ifc (unit FOR writing results), com, real mat(3,3)
; *   Output : writing result on unit ifc
;     ---------------------------------------------------------------+--

;     real mat(3,3)
;     character*(*) com

      fmt100="(1x,a)"
      fmt200="(12x,3(e15.5))"

      IF(ifc EQ 6 ) THEN BEGIN
         PRINT
         PRINT, FORMAT=fmt100,com
         PRINT
         PRINT, FORMAT=fmt200, mat(0,0),mat(0,1),mat(0,2)
         PRINT, FORMAT=fmt200, mat(1,0),mat(1,1),mat(1,2)
         PRINT, FORMAT=fmt200, mat(2,0),mat(2,1),mat(2,2)
                    ENDIF ELSE BEGIN
         PRINTF, ifc
         PRINTF, ifc, FORMAT=fmt100,com
         PRINTF, ifc
         PRINTF, ifc, FORMAT=fmt200, mat(0,0),mat(0,1),mat(0,2)
         PRINTF, ifc, FORMAT=fmt200, mat(1,0),mat(1,1),mat(1,2)
         PRINTF, ifc, FORMAT=fmt200, mat(2,0),mat(2,1),mat(2,2)
      ENDELSE

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

      PRO mat_write_eigen_vec, ifc,eigval,mat

;     ---------------------------------------------------------------+--
; *   Class  : matrix operation of Rocotlib Software
; *   Object : PRINT on ifc unit eigen values & vectors of mat(3,3)
; *   Author : P. Robert, CETP, 2001, rev. PR 2016

; *   Input  : ifc (unit FOR writing results), eigval(3), real mat(3,3)
; *   Output : writing result on unit ifc
;     ---------------------------------------------------------------+--

      rr   =FLTARR(3)
      theta=FLTARR(3)
      phi  =FLTARR(3)

      dpi=180./3.14159

      fmt100="(1x,a,3f15.4)"
      fmt110="(1x,a,3(13x,a2) )"
      fmt120="(1x,a,3(e15.4))"


      FOR i=0,2 DO BEGIN
         t_car_to_sph, mat(0,i),mat(1,i),mat(2,i),r,teta,phir
         rr(i)=r
         theta(i)=teta*dpi
         phi(i)  =phir*dpi
      ENDFOR

      IF(ifc EQ 6) THEN BEGIN
         PRINT
         PRINT, FORMAT=fmt100,'Eigen vectors and eigen values'
         PRINT, FORMAT=fmt100,'------------------------------'
         PRINT
         PRINT, FORMAT=fmt110,'   ', 'V1','V2','V3'
         PRINT, FORMAT=fmt100,'x     ', mat(0,0),mat(0,1),mat(0,2)
         PRINT, FORMAT=fmt100,'y     ', mat(1,0),mat(1,1),mat(1,2)
         PRINT, FORMAT=fmt100,'z     ', mat(2,0),mat(2,1),mat(2,2)
         PRINT
         PRINT, FORMAT=fmt100,'r     ', rr(0),rr(1),rr(2)
         PRINT, FORMAT=fmt100,'theta ', theta(0),theta(1),theta(2)
         PRINT, FORMAT=fmt100,'phi   ', phi(0),phi(1),phi(2)
         PRINT
         PRINT, FORMAT=fmt120,'Lambda ', eigval(0),eigval(1),eigval(2)
                   ENDIF ELSE BEGIN
      PRINTF, ifc
         PRINTF, ifc, FORMAT=fmt100,'Eigen vectors and eigen values'
         PRINTF, ifc, FORMAT=fmt100,'------------------------------'
         PRINTF, ifc
         PRINTF, ifc, FORMAT=fmt110,'   ', 'V1','V2','V3'
         PRINTF, ifc, FORMAT=fmt100,'x     ', mat(0,0),mat(0,1),mat(0,2)
         PRINTF, ifc, FORMAT=fmt100,'y     ', mat(1,0),mat(1,1),mat(1,2)
         PRINTF, ifc, FORMAT=fmt100,'z     ', mat(2,0),mat(2,1),mat(2,2)
         PRINTF, ifc
         PRINTF, ifc, FORMAT=fmt100,'r     ', rr(0),rr(1),rr(2)
         PRINTF, ifc, FORMAT=fmt100,'theta ', theta(0),theta(1),theta(2)
         PRINTF, ifc, FORMAT=fmt100,'phi   ', phi(0),phi(1),phi(2)
         PRINTF, ifc
         PRINTF, ifc, FORMAT=fmt120,'Lambda', eigval(0),eigval(1),eigval(2)
      ENDELSE

;     RETURN
      END

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

; BEGIN V2.2 compatibility 

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
;     P. Robert, ScientiDev, Janvier 2019
;     PRO FOR compatibility with previous V2.2 versions
;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

      PRO cangrat, ux,uy,uz,vx,vy,vz,angle,ratio
      cp_angle_and_ratio, ux,uy,uz,vx,vy,vz,angle,ratio
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO cdatdoy, idoy,iyear,imonth,iday
      cv_doty_to_date, idoy,iyear,imonth,iday
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO cdatj00, jd00,iyear,imonth,iday
      cv_jul2000_to_date, jd00,iyear,imonth,iday
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO cdatj50, jd50,iyear,imonth,iday
      cv_jul1950_to_date, jd50,iyear,imonth,iday
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO cdatwee, iweek,iyear,imonth,iday
      cv_weekn_to_date, iweek,iyear,imonth,iday
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO cdipdir, iyear,idoy,d1,d2,d3
      cp_geo_dipole_dir, iyear,idoy,d1,d2,d3
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO cdoweek, iyear,imonth,iday,idow
      cv_date_to_dotw, iyear,imonth,iday,idow
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO cdoyear, iyear,imonth,iday,idoy
      cv_date_to_doty, iyear,imonth,iday,idoy
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO cfrdayn, iday,cday,nbcha
;     character*(*) cday
      cp_fr_day_name, iday,cday,nbcha
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx


      PRO cfrmonn, imonth,cmonth,nchar
;     character*(*) cmonth
      cp_fr_month_name, imonth,cmonth,nchar
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO chouday, ih,im,is,houday
      cv_hms_to_dech, ih,im,is,houday
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO cjd1950, iyear,imonth,iday,jd50
      cv_date_to_jul1950, iyear,imonth,iday,jd50
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO cjd2000, iyear,imonth,iday,jd00
      cv_date_to_jul2000, iyear,imonth,iday,jd00
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO cmilday, ih,im,is,ims,milday
      cv_dhms_to_msotd, ih,im,is,ims,milday
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO cnbdmon, iyear,imonth,nbday
      cp_nbday_in_month, iyear,imonth,nbday
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO coleapy, iyear,ileap
      cp_leap_year, iyear,ileap
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO csundir, iyear,idoy,ih,im,is,gst,slong,sra,sdec,obliq
      cp_gei_sun_dir, iyear,idoy,ih,im,is,gst,slong,sra,sdec,obliq
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO csunset, iyear,imon,iday,rlat,rlon,tmer,tris,tset, $
                   durd,elemer,azimer,eleris,aziris,eleset,aziset,icor


      COMMON sunVDH,  suntim,sunele,sunazi
      suntim =fltarr(1441)
      sunele =fltarr(1441)
      sunazi =fltarr(1441)

;     character*(*) tmer,tris,tset,durd

      cp_sunrise_sunset, iyear,imon,iday,rlat,rlon,tmer,tris,tset, $
                   durd,elemer,azimer,eleris,aziris,eleset,aziset,icor
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO ctimhou, houday,ih,im,is
      cv_dech_to_hms, houday,ih,im,is
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO ctimmil, milday,ih,im,is,ims
      cv_msotd_to_hmsms, milday,ih,im,is,ims
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO ctimpa2, jd1950,houday
      cp_time_param2, jd1950,houday
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO ctimpa3, jd2000,houday
      cp_time_param3, jd2000,houday
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO ctimpar, iyear,imonth,iday,ih,im,is
      cp_time_param, iyear,imonth,iday,ih,im,is
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO cusdayn, iday,cday,nbcha
;     character*(*) cday
      cp_en_day_name, iday,cday,nbcha
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO cusmonn, imonth,cmonth,nchar
;     character*(*) cmonth
      cp_en_month_name, imonth,cmonth,nchar
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO cweedoy, iyear,imonth,iday,iweek
      cv_date_to_weekn, iyear,imonth,iday,iweek
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO gdipdir, dxgei,dygei,dzgei,dxgeo,dygeo,dzgeo
      g_gei_geo_dipole_dir, dxgei,dygei,dzgei,dxgeo,dygeo,dzgeo
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO gdiptan, diptan
      g_gsm_dipole_tilt_angle, diptan
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO gecldir, exgei,eygei,ezgei,exgeo,eygeo,ezgeo
      g_gei_geo_ecliptic_dir, exgei,eygei,ezgei,exgeo,eygeo,ezgeo
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO gsrodir, rxgei,rygei,rzgei,rxgeo,rygeo,rzgeo
      g_gei_geo_sun_rot, rxgei,rygei,rzgei,rxgeo,rygeo,rzgeo
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO gsundir, sxgei,sygei,szgei,sxgeo,sygeo,szgeo
      g_gei_geo_sun_dir, sxgei,sygei,szgei,sxgeo,sygeo,szgeo
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO gsunpar, gmst,slon,sras,sdec,obli
      g_gei_sun_param, gmst,slon,sras,sdec,obli
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO gvernum, vernum,verdat
      g_rocot_version_number, vernum,verdat
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO plibinf
      PRINT_rocot_info
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO recoor, x,y,z,cs
      r_coordinate_values, x,y,z,cs
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO recsys, csys
      r_coordinate_system, csys
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO redate, iyear,imonth,iday
      r_date, iyear,imonth,iday
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO retime, ih,im,is
      r_time, ih,im,is
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tcarsph, x,y,z,r,teta,phi
      t_car_to_sph, x,y,z,r,teta,phi
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tdmegeo, xdme,ydme,zdme,rlat,rlong,xgeo,ygeo,zgeo
      t_dm_to_geo, xdme,ydme,zdme,rlat,rlong,xgeo,ygeo,zgeo
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgeigeo, xgei,ygei,zgei,xgeo,ygeo,zgeo
      t_gei_to_geo, xgei,ygei,zgei,xgeo,ygeo,zgeo
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgeigse, xgei,ygei,zgei,xgse,ygse,zgse
      t_gei_to_gse, xgei,ygei,zgei,xgse,ygse,zgse
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgeigsm, xgei,ygei,zgei,xgsm,ygsm,zgsm
      t_gei_to_gsm, xgei,ygei,zgei,xgsm,ygsm,zgsm
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgeigsq, xgei,ygei,zgei,xgsq,ygsq,zgsq
      t_gei_to_gseq, xgei,ygei,zgei,xgsq,ygsq,zgsq
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgeimag, xgei,ygei,zgei,xmag,ymag,zmag
      t_gei_to_mag, xgei,ygei,zgei,xmag,ymag,zmag
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgeisma, xgei,ygei,zgei,xsma,ysma,zsma
      t_gei_to_sm, xgei,ygei,zgei,xsma,ysma,zsma
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgeodme, xgeo,ygeo,zgeo,rlat,rlong,xdme,ydme,zdme
      t_geo_to_dm, xgeo,ygeo,zgeo,rlat,rlong,xdme,ydme,zdme
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgeogei, xgeo,ygeo,zgeo,xgei,ygei,zgei
      t_geo_to_gei, xgeo,ygeo,zgeo,xgei,ygei,zgei
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgeogse, xgeo,ygeo,zgeo,xgse,ygse,zgse
      t_geo_to_gse, xgeo,ygeo,zgeo,xgse,ygse,zgse
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgeogsm, xgeo,ygeo,zgeo,xgsm,ygsm,zgsm
      t_geo_to_gsm, xgeo,ygeo,zgeo,xgsm,ygsm,zgsm
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgeogsq, xgeo,ygeo,zgeo,xgsq,ygsq,zgsq
      t_geo_to_gseq, xgeo,ygeo,zgeo,xgsq,ygsq,zgsq
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgeomag, xgeo,ygeo,zgeo,xmag,ymag,zmag
      t_geo_to_mag, xgeo,ygeo,zgeo,xmag,ymag,zmag
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgeosma, xgeo,ygeo,zgeo,xsma,ysma,zsma
      t_geo_to_sm, xgeo,ygeo,zgeo,xsma,ysma,zsma
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgeovdh, xgeo,ygeo,zgeo,rlat,rlong,xvdh,yvdh,zvdh
      t_geo_to_vdh, xgeo,ygeo,zgeo,rlat,rlong,xvdh,yvdh,zvdh
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgsegei, xgse,ygse,zgse,xgei,ygei,zgei
      t_gse_to_gei, xgse,ygse,zgse,xgei,ygei,zgei
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgsegeo, xgse,ygse,zgse,xgeo,ygeo,zgeo
      t_gse_to_geo, xgse,ygse,zgse,xgeo,ygeo,zgeo
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgsegsm, xgse,ygse,zgse,xgsm,ygsm,zgsm
      t_gse_to_gsm, xgse,ygse,zgse,xgsm,ygsm,zgsm
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgsegsq, xgse,ygse,zgse,xgsq,ygsq,zgsq
      t_gse_to_gseq, xgse,ygse,zgse,xgsq,ygsq,zgsq
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgsesr2, xgse,ygse,zgse,rotx,roty,rotz,xsr2,ysr2,zsr2
      t_gse_to_sr2, xgse,ygse,zgse,rotx,roty,rotz,xsr2,ysr2,zsr2
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgsmgei, xgsm,ygsm,zgsm,xgei,ygei,zgei
      t_gsm_to_gei, xgsm,ygsm,zgsm,xgei,ygei,zgei
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgsmgeo, xgsm,ygsm,zgsm,xgeo,ygeo,zgeo
      t_gsm_to_geo, xgsm,ygsm,zgsm,xgeo,ygeo,zgeo
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgsmgse, xgsm,ygsm,zgsm,xgse,ygse,zgse
      t_gsm_to_gse, xgsm,ygsm,zgsm,xgse,ygse,zgse
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgsmgsq, xgsm,ygsm,zgsm,xgsq,ygsq,zgsq
      t_gsm_to_gseq, xgsm,ygsm,zgsm,xgsq,ygsq,zgsq
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgsmmag, xgsm,ygsm,zgsm,xmag,ymag,zmag
      t_gsm_to_mag, xgsm,ygsm,zgsm,xmag,ymag,zmag
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgsmsma, xgsm,ygsm,zgsm,xsma,ysma,zsma
      t_gsm_to_sm, xgsm,ygsm,zgsm,xsma,ysma,zsma
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgsqgei, xgsq,ygsq,zgsq,xgei,ygei,zgei
      t_gseq_to_gei, xgsq,ygsq,zgsq,xgei,ygei,zgei
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgsqgeo, xgsq,ygsq,zgsq,xgeo,ygeo,zgeo
      t_gseq_to_geo, xgsq,ygsq,zgsq,xgeo,ygeo,zgeo
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgsqgse, xgsq,ygsq,zgsq,xgse,ygse,zgse
      t_gseq_to_gse, xgsq,ygsq,zgsq,xgse,ygse,zgse
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tgsqgsm, xgsq,ygsq,zgsq,xgsm,ygsm,zgsm
      t_gseq_to_gsm, xgsq,ygsq,zgsq,xgsm,ygsm,zgsm
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tmaggei, xmag,ymag,zmag,xgei,ygei,zgei
      t_mag_to_gei, xmag,ymag,zmag,xgei,ygei,zgei
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tmaggeo, xmag,ymag,zmag,xgeo,ygeo,zgeo
      t_mag_to_geo, xmag,ymag,zmag,xgeo,ygeo,zgeo
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tmaggsm, xmag,ymag,zmag,xgsm,ygsm,zgsm
      t_mag_to_gsm, xmag,ymag,zmag,xgsm,ygsm,zgsm
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tmagsma, xmag,ymag,zmag,xsma,ysma,zsma
      t_mag_to_sm, xmag,ymag,zmag,xsma,ysma,zsma
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tsmagei, xsma,ysma,zsma,xgei,ygei,zgei
      t_sm_to_gei, xsma,ysma,zsma,xgei,ygei,zgei
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tsmageo, xsma,ysma,zsma,xgeo,ygeo,zgeo
      t_sm_to_geo, xsma,ysma,zsma,xgeo,ygeo,zgeo
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tsmagsm, xsma,ysma,zsma,xgsm,ygsm,zgsm
      t_sm_to_gsm, xsma,ysma,zsma,xgsm,ygsm,zgsm
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tsmamag, xsma,ysma,zsma,xmag,ymag,zmag
      t_sm_to_mag, xsma,ysma,zsma,xmag,ymag,zmag
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tsphcar, r,teta,phi,x,y,z
      t_sph_to_car, r,teta,phi,x,y,z
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tsr2gse, xsr2,ysr2,zsr2,rotx,roty,rotz,xgse,ygse,zgse
      t_sr2_to_gse, xsr2,ysr2,zsr2,rotx,roty,rotz,xgse,ygse,zgse
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tsr2mfa, xsr2,ysr2,zsr2,bx,by,bz,rox,roy,roz,xm,ym,zm
      t_sr2_to_mfa, xsr2,ysr2,zsr2,bx,by,bz,rox,roy,roz,xm,ym,zm
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tsr2sre, xsr2,ysr2,spifre,spipha,deltaT,xsre,ysre
      t_sr2_to_sr, xsr2,ysr2,spifre,spipha,deltaT,xsre,ysre
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tsresr2, xsre,ysre,spifre,spipha,deltaT,xsr2,ysr2
      t_sr_to_sr2, xsre,ysre,spifre,spipha,deltaT,xsr2,ysr2
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx

      PRO tvdhgeo, xvdh,yvdh,zvdh,rlat,rlong,xgeo,ygeo,zgeo
      t_vdh_to_geo, xvdh,yvdh,zvdh,rlat,rlong,xgeo,ygeo,zgeo
;     RETURN
      END

;     xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx0xx
; END V2.2 compatibility
; END ROCOTLIB 3.2

;     XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
