; XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
;+
; -----------------------------------------------------------------------------
; PROCEDURE:
; barycentric_lib.pro : IDL serie of procedures used for computation of 
;                       reciprocal vector of the barycentric coordinates
;
; PURPOSE:
; Software for computation of reciprocal vector of the barycentric coordinates
; has been initially delivered in f77 by G. Chanteur, CETP, september 1994. 
; These vectors are used to compute easily curl(B) and div(B), 
; and then (in 2004) for the computation of the radius of curvature and 
; the normal vector to the osculating plane.
;
; The G.C. code was in fortran 77, and rewritten in fortran 90 for application
; to the CLUSTER FGM data ( Patrick ROBERT, 1994, revised in March  2004).
; Here is the translation in IDL for MMS application, P. Robert, december 2015

;$CreatedBy: P. Robert $
;$LastChangedDate: 2015-11-25 
; -----------------------------------------------------------------------------

  PRO inipobary, s

;----------------------------------------------------------------------
; * Object: set the initial position of 4 S/C for barycent. coord.
; * Author: G. Chanteur, P. Robert, CRPE, 1994
;----------------------------------------------------------------------

; input : real s(3,4)
; output: common /satel/

  common satel, x1, y1, z1, x2, y2, z2, x3, y3, z3, x4, y4, z4

  x1 = s(0,0)
  y1 = s(1,0)
  z1 = s(2,0)

  x2 = s(0,1)
  y2 = s(1,1)
  z2 = s(2,1)

  x3 = s(0,2)
  y3 = s(1,2)
  z3 = s(2,2)

  x4 = s(0,3)
  y4 = s(1,3)
  z4 = s(2,3)

  return
  end

; XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  PRO calbabary, qfbary

;----------------------------------------------------------------------
; * Object:  compute barycentric coordinates
; * Author: G. Chanteur, P. Robert, CRPE, 1994
;----------------------------------------------------------------------

; input : common /satel/
; output: common /base/ and factor qfbary

;  real*4      kx1,ky1,kz1,kx2,ky2,kz2,kx3,ky3,kz3,kx4,ky4,kz4

  common satel, x1, y1, z1, x2, y2, z2, x3, y3, z3, x4, y4, z4
  common base,  kx1,ky1,kz1,kx2,ky2,kz2,kx3,ky3,kz3,kx4,ky4,kz4,Q


  ax4 = (y2-y1)*(z3-z1) -(z2-z1)*(y3-y1)
  ay4 = (z2-z1)*(x3-x1) -(x2-x1)*(z3-z1)
  az4 = (x2-x1)*(y3-y1) -(y2-y1)*(x3-x1)
  
   v4 = ax4*(x4-x1) +ay4*(y4-y1) +az4*(z4-z1)
  kx4 = ax4/v4
  ky4 = ay4/v4
  kz4 = az4/v4

  ax3 = (y1-y4)*(z2-z4) -(z1-z4)*(y2-y4)
  ay3 = (z1-z4)*(x2-x4) -(x1-x4)*(z2-z4)
  az3 = (x1-x4)*(y2-y4) -(y1-y4)*(x2-x4)
  
   v3 = ax3*(x3-x4) +ay3*(y3-y4)+az3*(z3-z4)
  kx3 = ax3/v3
  ky3 = ay3/v3
  kz3 = az3/v3

  ax2 = (y4-y3)*(z1-z3) -(z4-z3)*(y1-y3)
  ay2 = (z4-z3)*(x1-x3) -(x4-x3)*(z1-z3)
  az2 = (x4-x3)*(y1-y3) -(y4-y3)*(x1-x3)
  
   v2 = ax2*(x2-x3) +ay2*(y2-y3)+az2*(z2-z3)
  kx2 = ax2/v2
  ky2 = ay2/v2
  kz2 = az2/v2

  ax1 = (y3-y2)*(z4-z2) -(z3-z2)*(y4-y2)
  ay1 = (z3-z2)*(x4-x2) -(x3-x2)*(z4-z2)
  az1 = (x3-x2)*(y4-y2) -(y3-y2)*(x4-x2)
  
   v1 = ax1*(x1-x2) +ay1*(y1-y2)+az1*(z1-z2)
  kx1 = ax1/v1
  ky1 = ay1/v1
  kz1 = az1/v1

; Chanteur quality factor

  rk1= kx1^2 + ky1^2 + kz1^2
  rk2= kx2^2 + ky2^2 + kz2^2
  rk3= kx3^2 + ky3^2 + kz3^2
  rk4= kx4^2 + ky4^2 + kz4^2

  rk2min=min([rk1,rk2,rk3,rk4])
  rk2max=max([rk1,rk2,rk3,rk4])

  qfbary=rk2min/rk2max

  return
  end

; XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  PRO inimabary, bmag

;----------------------------------------------------------------------
; * Object: set the the magnetic field values for barycent. coord.
; * Author: G. Chanteur, P. Robert, CRPE, 1994
;----------------------------------------------------------------------

; input : real bmag(3,4)
; output: common /field/

  common field, bx1,by1,bz1,bx2,by2,bz2,bx3,by3,bz3,bx4,by4,bz4


  bx1 = bmag(0,0)
  by1 = bmag(1,0)
  bz1 = bmag(2,0)

  bx2 = bmag(0,1)
  by2 = bmag(1,1)
  bz2 = bmag(2,1)

  bx3 = bmag(0,2)
  by3 = bmag(1,2)
  bz3 = bmag(2,2)

  bx4 = bmag(0,3)
  by4 = bmag(1,3)
  bz4 = bmag(2,3)

  return
  end

; XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  PRO estibbary

;----------------------------------------------------------------------
; * Object: estimate vectorial quantities of B by barycent. coord.
; * Author: G.Chanteur, CETP, 1994
;----------------------------------------------------------------------

; Estimate vectorial quantities of B by the barycentric coordinates
; 0riginal sofware delivered by G. Chanteur CETP/IPSL - CNRS & UVSQ,
; september 1994. Radius of curvature from gradB, G.C., 2003/02/26.

; input : common /satel/, /base/, /field/
; output: common /estim/

  common base,  kx1,ky1,kz1,kx2,ky2,kz2,kx3,ky3,kz3,kx4,ky4,kz4,Q
  common field, bx1,by1,bz1,bx2,by2,bz2,bx3,by3,bz3,bx4,by4,bz4
  
  common estim, gbxx,gbxy,gbxz,gbyx,gbyy,gbyz,gbzx,gbzy,gbzz $
               ,divb,rotbx,rotby,rotbz,rnormx,rnormy,rnormz,radcurv

; *** Estimation du tenseur gradB
 
  gbxx = kx1*bx1 +kx2*bx2 +kx3*bx3 +kx4*bx4
  gbxy = kx1*by1 +kx2*by2 +kx3*by3 +kx4*by4
  gbxz = kx1*bz1 +kx2*bz2 +kx3*bz3 +kx4*bz4

  gbyx = ky1*bx1 +ky2*bx2 +ky3*bx3 +ky4*bx4
  gbyy = ky1*by1 +ky2*by2 +ky3*by3 +ky4*by4
  gbyz = ky1*bz1 +ky2*bz2 +ky3*bz3 +ky4*bz4

  gbzx = kz1*bx1 +kz2*bx2 +kz3*bx3 +kz4*bx4
  gbzy = kz1*by1 +kz2*by2 +kz3*by3 +kz4*by4
  gbzz = kz1*bz1 +kz2*bz2 +kz3*bz3 +kz4*bz4

; *** Estimation de divB

  divb = gbxx +gbyy +gbzz

; *** Estimation de rotB

  rotbx = gbyz-gbzy
  rotby = gbzx-gbxz
  rotbz = gbxy-gbyx

; *** Estimation of the unit vector normal to field lines through
;     CLUSTER  and of the radius of curvature from gradB
;     G. Chanteur CETP/IPSL - CNRS & UVSQ 2003/02/26 at IASB, Brussels

; *** Magnetic field at center of mass and unit tangent vector T

  bmean_x = 0.25*(bx1 +bx2 +bx3 +bx4)
  bmean_y = 0.25*(by1 +by2 +by3 +by4)
  bmean_z = 0.25*(bz1 +bz2 +bz3 +bz4)
  
  bmean_m = sqrt(bmean_x^2 +bmean_y^2 +bmean_z^2)

  tx = bmean_x/bmean_m
  ty = bmean_y/bmean_m
  tz = bmean_z/bmean_m

; *** Estimation of the unit normal vector N and curvature 1/R

  tk1 = tx*kx1 +ty*ky1 +tz*kz1
  tk2 = tx*kx2 +ty*ky2 +tz*kz2
  tk3 = tx*kx3 +ty*ky3 +tz*kz3
  tk4 = tx*kx4 +ty*ky4 +tz*kz4

  dbds   = (tx*bx1 +ty*by1 +tz*bz1)*tk1 $
          +(tx*bx2 +ty*by2 +tz*bz2)*tk2 $
          +(tx*bx3 +ty*by3 +tz*bz3)*tk3 $
          +(tx*bx4 +ty*by4 +tz*bz4)*tk4

  Ax = tk1*bx1 +tk2*bx2 +tk3*bx3 +tk4*bx4 - dbds*tx
  Ay = tk1*by1 +tk2*by2 +tk3*by3 +tk4*by4 - dbds*ty
  Az = tk1*bz1 +tk2*bz2 +tk3*bz3 +tk4*bz4 - dbds*tz
  
  Am = sqrt(Ax^2 +Ay^2 +Az^2)

  rnormx= Ax/Am
  rnormy= Ay/Am
  rnormz= Az/Am

  radcurv = bmean_m/Am

  return
  end

; XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  PRO ccurlbary, curx,cury,curz

;----------------------------------------------------------------------
; * Object: compute curl(B) from barycentric method
; * Author: G. Chanteur, P. Robert, CRPE, 1994
;----------------------------------------------------------------------

  common estim, gbxx,gbxy,gbxz,gbyx,gbyy,gbyz,gbzx,gbzy,gbzz $
               ,divb,rotbx,rotby,rotbz,rnormx,rnormy,rnormz,radcurv

;     Estimation de rotB

  curx = rotbx
  cury = rotby
  curz = rotbz

  return
  end

; XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  PRO ccurvbary, rnormax,rnormay,rnormaz,rcurv

;----------------------------------------------------------------------
; * Object: compute radius of curvature of field lines
;           and normal to osculateur plane from barycentric method
; * Author: G.Chanteur, CETP, 2003
;----------------------------------------------------------------------

  common estim, gbxx,gbxy,gbxz,gbyx,gbyy,gbyz,gbzx,gbzy,gbzz $
               ,divb,rotbx,rotby,rotbz,rnormx,rnormy,rnormz,radcurv

; Estimation de la normale

  rnormax = rnormx
  rnormay = rnormy
  rnormaz = rnormz

; Estimation du rayon de courbure

  rcurv=radcurv

  return
  end

; XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  PRO cgrabary, gra1x,gra1y,gra1z, gra2x,gra2y,gra2z,gra3x,gra3y,gra3z

;----------------------------------------------------------------------
; * Object: compute grad(B) matrix from barycentric metho
; * Author: G.Chanteur, CETP, 1994
;----------------------------------------------------------------------

  common estim, gbxx,gbxy,gbxz,gbyx,gbyy,gbyz,gbzx,gbzy,gbzz $
               ,divb,rotbx,rotby,rotbz,rnormx,rnormy,rnormz,radcurv

;     Estimation du tenseur gradB

  gra1x = gbxx
  gra1y = gbyx
  gra1z = gbzx

  gra2x = gbxy
  gra2y = gbyy
  gra2z = gbzy

  gra3x = gbxz
  gra3y = gbyz
  gra3z = gbzz
  
  return
  end

; XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  PRO cdivbary, divdeb

;----------------------------------------------------------------------
; * Object: compute div(B) from barycentric method
; * Author: G. Chanteur, P. Robert, CRPE, 1994
;----------------------------------------------------------------------

  common estim, gbxx,gbxy,gbxz,gbyx,gbyy,gbyz,gbzx,gbzy,gbzz $
               ,divb,rotbx,rotby,rotbz,rnormx,rnormy,rnormz,radcurv

; Estimation de divB

  divdeb = divb

  return
  end

; XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  PRO ctestbary, iok

;----------------------------------------------------------------------
; * Object: compute test on gradient sign from barycentric method
; * Author: G. Chanteur, P. Robert, CRPE, 1994
;----------------------------------------------------------------------

  common estim, gbxx,gbxy,gbxz,gbyx,gbyy,gbyz,gbzx,gbzy,gbzz $
               ,divb,rotbx,rotby,rotbz,rnormx,rnormy,rnormz,radcurv

  iok=1

  if(gbyz*gbzy GT 0.) then iok=0
  if(gbzx*gbxz GT 0.) then iok=0
  if(gbxy*gbyx GT 0.) then iok=0

  return
  end

; XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
; complement to barycentric_lib (not used for Curl computation)
; XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

  PRO cdistance, s

;----------------------------------------------------------------------!
; * Object:  compute all the distances of the tetrahedron
; * Author: P. Robert, CRPE, 1994
;----------------------------------------------------------------------!

  common distan,  d12 ,  d13 ,  d14 ,  d23 ,  d24,  d34, $
                 dx12 , dx13 , dx14 , dx21 , dx31, dx41, $
                 dy12 , dy13 , dy14 , dy21 , dy31, dy41, $
                 dz12 , dz13 , dz14 , dz21 , dz31, dz41

  dx12 = s(0,1) -s(0,0)
  dy12 = s(1,1) -s(1,0)
  dz12 = s(2,1) -s(2,0)

  dx13 = s(0,2) -s(0,0)
  dy13 = s(1,2) -s(1,0)
  dz13 = s(2,2) -s(2,0)

  dx14 = s(0,3) -s(0,0)
  dy14 = s(1,3) -s(1,0)
  dz14 = s(2,3) -s(2,0)

  dx23 = s(0,2) -s(0,1)
  dy23 = s(1,2) -s(1,1)
  dz23 = s(2,2) -s(2,1)

  dx24 = s(0,3) -s(0,1)
  dy24 = s(1,3) -s(1,1)
  dz24 = s(2,3) -s(2,1)

  dx34 = s(0,3) -s(0,2)
  dy34 = s(1,3) -s(1,2)
  dz34 = s(2,3) -s(2,2)


  dx21 = s(0,0) -s(0,1)
  dy21 = s(1,0) -s(1,1)
  dz21 = s(2,0) -s(2,1)

  dx31 = s(0,0) -s(0,2)
  dy31 = s(1,0) -s(1,2)
  dz31 = s(2,0) -s(2,2)

  dx41 = s(0,0) -s(0,3)
  dy41 = s(1,0) -s(1,3)
  dz41 = s(2,0) -s(2,3)


  d12 = sqrt(dx12^2 + dy12^2 + dz12^2)
  d13 = sqrt(dx13^2 + dy13^2 + dz13^2)
  d14 = sqrt(dx14^2 + dy14^2 + dz14^2)
  d23 = sqrt(dx23^2 + dy23^2 + dz23^2)
  d24 = sqrt(dx24^2 + dy24^2 + dz24^2)
  d34 = sqrt(dx34^2 + dy34^2 + dz34^2)

  return
  end

; XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
