; XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX
;+
; PROCEDURE:
; cocurldiv.pro : IDL procedure
;         
; PURPOSE:
;      procedure to compute Curl(B) and Div(B) from magnetic fied
;      measured at four points in space, at the same time.
;      can be used for 4 S/C missions, as CLUSTER or MMS.
;
; INPUT: 
;        Valpos: position at each vertice of the tetrahedron (km)
;                DBLE [N_time, 3 (xyz), 4 (S/C) ]
;        Valmag: value of the magnetic field at each vertice of the 
;                tetrahedron (nT)
;                DBLE [N_time, 3 (xyz), 4 (S/C) ]
;
; OUTPUT:
;        mu0      : scalar,  with related unity of input/output
;                   (J=Curl/mu0 in A/km2)
;        Curl     : Curl(B)   FLOAT array [N_time] in nT/km
;        Div      : Div(B)    FLOAT array [N_time] in nT/km
;        Bave     : B average FLOAT array [N_time] in nT 
;                   (at the tetrahetron barycentre)
;        JxB      : JXBave vector, expressed as E=JXB
;                   FLOAT array [N_time, 3]  in mV/m
;        JxB_Norm : Normal to the JXBave vector,
;                   FLOAT array [N_time, 3] normalized to 1. 
;        JB_Angle : angle between (J,Bave) vectors,
;                   FLOAT array [N_time] in degree.
;        Norm_Osc : Normal to the osculator plane,
;                   FLOAT array [N_time, 3] normalized to 1.
;        Curv_Rad : curvature radius of the mag field,
;                   FLOAT array [N_time] in km
;
; This procedure requires the compilation of barycentric_lib.pro 
; which contains elements for computation of reciprocal vector of the 
; barycentric coordinates.
;
; It was initially delivered in f77 by G. Chanteur, CETP, sep. 1994. 
; These vectors are used to compute easily curl(B) and div(B), 
; and then (in 2004) for the computation of the radius of curvature and 
; the normal vector to the osculating plane.

; The code was in fortran 77, and rewritten in fortran 90 for 
; application to the CLUSTER FGM data ( Patrick ROBERT, 1994, revised 
; in March  2004).  Here is the translation in IDL for MMS application,
; P. Robert, dec. 2015.
;
;$CreatedBy: P. Robert $
;$Gerard Chanteur method, CETP, september 1994.
;$LastChangedDate: 2015-11-25 
;
; ----------------------------------------------------------------------

  PRO cocurldiv, sattim,valpos,valmag,  mu0,Curl,Div,Bave, $
                 Jcd,JXB,JXB_Norm,JB_Angle,Norm_Osc,Curv_Rad

  common satel, x1, y1, z1, x2, y2, z2, x3, y3, z3, x4, y4, z4
  common base,  kx1,ky1,kz1,kx2,ky2,kz2,kx3,ky3,kz3,kx4,ky4,kz4,Q
  common field, bx1,by1,bz1,bx2,by2,bz2,bx3,by3,bz3,bx4,by4,bz4
  common estim, gbxx,gbxy,gbxz,gbyx,gbyy,gbyz,gbzx,gbzy,gbzz $
               ,divb,rotbx,rotby,rotbz,rnormx,rnormy,rnormz,radcurv
  
  
  PRINT
  PRINT, 'coculdiv.pro / input  parameters:'
  HELP, sattim, valpos, valmag
  
  nbp = n_elements(sattim)
  
; output arrays
; -------------
  
  Curl = FLTARR(nbp,3)
  Div  = FLTARR(nbp)
  Bave = FLTARR(nbp,3)
  
  Jcd      = FLTARR(nbp,4)
  JXB      = FLTARR(nbp,4)
  JXB_Norm = FLTARR(nbp,3)
  JB_Angle = FLTARR(nbp)
  Norm_Osc = FLTARR(nbp,3)
  Curv_Rad = FLTARR(nbp)
  
; working array
  satpos = DBLARR(3,4)
  satmag = DBLARR(3,4)

; Mo en nT*km/A

  pi=acos(-1.)
  mu0=4.*pi*(1.e-7)*(1.e6)
  
;  BIG LOOP ON THE TIME 

; ======================== 
  FOR k = 0,nbp-1 DO BEGIN
; ========================  
  
; compute satpos and satmag at each time
; -------------------------------------- 

      satpos(*,0) = valpos(k,*,0)
      satpos(*,1) = valpos(k,*,1)
      satpos(*,2) = valpos(k,*,2)
      satpos(*,3) = valpos(k,*,3)
            
      satmag(*,0) = valmag(k,*,0)
      satmag(*,1) = valmag(k,*,1)
      satmag(*,2) = valmag(k,*,2)
      satmag(*,3) = valmag(k,*,3)

;     position in mass center
;     -----------------------

      gx=(satpos(0,0)+satpos(0,1)+satpos(0,2)+satpos(0,3))/4.
      gy=(satpos(1,0)+satpos(1,1)+satpos(1,2)+satpos(1,3))/4.
      gz=(satpos(2,0)+satpos(2,1)+satpos(2,2)+satpos(2,3))/4.
      
      satpos(0,*) = satpos(0,*) -gx
      satpos(1,*) = satpos(1,*) -gy
      satpos(2,*) = satpos(2,*) -gz


;     set the initial position of 4 S/C for barycent. coord.
;     ------------------------------------------------------

      inipobary, satpos 
      
;     set the the magnetic field values for barycent. coord.  
;     ------------------------------------------------------

      inimabary, satmag 
      
;     compute barycentric coordinates  
;     -------------------------------

      calbabary, qfbary 
      
;     estimate vectorial quantities of B by barycent. coord.  
;     ------------------------------------------------------

      estibbary         
      
;     compute curl(B) 
;     ---------------

      ccurlbary, cux,cuy,cuz
      
      Curl(k,0) = cux
      Curl(k,1) = cuy
      Curl(k,2) = cuz

;     compute current density in nA/m2 (1000 A/Km2) 

      gjx = 1000.*cux/mu0
      gjy = 1000.*cuy/mu0
      gjz = 1000.*cuz/mu0
      
      Jcd(k,0) = gjx
      Jcd(k,1) = gjy
      Jcd(k,2) = gjz

      Jcd(*,3)= SQRT(Jcd(*,0)^2 + Jcd(*,1)^2 + Jcd(*,2)^2)

;     compute div(B)
;     --------------

      cdivbary, divk
      
      Div(k) = divk

;     compute Baverage
;     ----------------

      bmx = (satmag(0,0) +satmag(0,1) +satmag(0,2) +satmag(0,3))/4.
      bmy = (satmag(1,0) +satmag(1,1) +satmag(1,2) +satmag(1,3))/4.
      bmz = (satmag(2,0) +satmag(2,1) +satmag(2,2) +satmag(2,3))/4.
      
      Bave(k,0) = bmx
      Bave(k,1) = bmy
      Bave(k,2) = bmz

;     compute E= J X Baverage
;     -----------------------

;     J(nA/m2) = 10-9 J(A/m2) ;  B(nT) = 10-9 B(T)  
;     conversion factor to have mV/m: 
;     E-9*E-9 (nA,nT)
;     we have to divide by density in N/m3 to have units in mV/m
;     we assume for now N=1.E6 e-/m3
     
      densi=1.e6
      conv= 1.E-18/(densi*1.60216E-19)

      JXBx = (gjy*bmz - gjz*bmy)
      JXBy = (gjz*bmx - gjx*bmz)
      JXBz = (gjx*bmy - gjy*bmx)

      JXB(k,0) = JXBx*conv
      JXB(k,1) = JXBy*conv
      JXB(k,2) = JXBz*conv

      JXB(*,3)= SQRT(JXB(*,0)^2 + JXB(*,1)^2 + JXB(*,2)^2)

;     compute normale N= J X Baverage
;     -------------------------------
      
      modJ = SQRT(gjx^2 +gjy^2 +gjz^2)
      modB = SQRT(bmx^2 +bmy^2 +bmz^2)
      modt = modJ*modB
      
      JXB_Norm(k,0) = JXBx/modt
      JXB_Norm(k,1) = JXBy/modt
      JXB_Norm(k,2) = JXBz/modt

;     compute(J,Bave) angle in degree
;     -------------------------------

      cangrat, gjx,gjy,gjz,bmx,bmy,bmz,ajb,ratio

      JB_Angle(k) = ajb*180./3.1415927

;     compute normal to osculator plane and radius of curvature
;     ---------------------------------------------------------

      ccurvbary, rnormax,rnormay,rnormaz,rcurv
  
; END OF BIG LOOP ON THE TIME 

; ====== 
  ENDFOR
; ======   

  PRINT, 'coculdiv.pro / Done...'
  PRINT
  PRINT, 'coculdiv.pro / output parameters:'
  HELP,  mu0,Curl,Div,Bave,JxB_Norm,JB_Angle,Norm_Osc,Curv_Rad

  RETURN
  END

; XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX0XX

